// Smoldot
// Copyright (C) 2019-2021  Parity Technologies (UK) Ltd.
// SPDX-License-Identifier: GPL-3.0-or-later WITH Classpath-exception-2.0

// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//! Collection of libp2p connections.
//!
//! The [`Network`] struct in this module is a collection of libp2p connections. It uses internal
//! buffering and interior mutability in order to provide a convenient-to-use API based around
//! notifications protocols and request-response protocols.
//!
//! The reasons for having a data structure providing a collection of connections, rather than
//! managing the collection manually (e.g. with a `Vec` or a `HashMap`) are:
//!
//! - Providing an API that is fundamentally compatible with back-pressure mechanisms. The
//! [`Network::next_event`] must be polled regularly in order to process the events generated by
//! the connections. If these events aren't processed quickly enough, calling
//! [`Network::read_write`] will stop reading data incoming on connections and therefore
//! back-pressure them.
//! - Making it possible to later add support for the QUIC protocol without too much friction.
//! When using QUIC, one receives UDP packets without being able to know which connection they
//! are destined to before inspecting them.
//!
//! # Basic usage
//!
//! - Whenever a new connection is established, use [`Network::insert`] to allocate a connection
//! in the collection.
//! - When a connection has received data or is ready to send more data, use
//! [`Network::read_write`] to synchronize the state of the [`Network`] with the actual state of
//! the connection.
//! - In parallel, continuously call [`Network::next_event`] to process the events generated by
//! the calls to [`Network::read_write`].
//!

// # Implementation notes
//
// This module is the synchronization point between all libp2p connections. It is the boundary
// between the "single-threaded world" (each individual libp2p connection uses exterior
// mutability) and the "multithreaded world" (the API of `Network` uses interior mutability).
// In other words, it is this module that provides a consistent view of all the connections as a
// whole, while trying to allow as many concurrent accesses as possible.
//
// As such, the code in this module is rather complex.
//
// The `Network` struct mainly consists of the following mutable components:
//
// - One instance of the `Guarded` struct, guarded by a `Mutex`, containing a view of the state
// of all connections.
// - For each active connection, one `Connection` struct, guarded by a `Mutex`, containing the
// state of this connection in particular.
//
// In order to avoid potential bugs and deadlocks, no single thread of execution must attempt
// to lock the `Mutex` protecting the `Guarded` then a `Mutex` guarding a connection at the same
// time without unlocking the `Guarded` first. The other way around, however, is authorized: one
// can lock the `Mutex` guarding a connection, then the `Mutex` protecting the `Guarded`.
//
// The view of the network within the `Guarded` is not necessarily always up-to-date with the
// actual state of the connections. For example, the `Guarded` might think that a certain
// substream on a particular connection is open while in reality it has already been closed.
//
// Connections hold a "pending event" field containing an event that has happened on this
// connection but hasn't been delivered to the `Guarded` yet. In other words, the `Guarded`
// doesn't yet take this event into account in its view.
// This field only holds space for a single event. The connection should never be updated as long
// as an event is present in this field, in order to avoid potentially generating a second event.
// Delivering this event to the `Guarded` is expected to be extremely quick, but in case it is
// not, connections will be back-pressured.
//
// This "pending event" field also solves futures-cancellation-related problems. It is possible
// for the user to interrupt any operation at any `await` point without causing a state mismatch.
//
// With all the information above in mind, the flow of a typical operation consists in the
// following steps:
//
// - Lock `Guarded` and inspect the (potentially outdated) state of the connection. Do not modify
// the state within `Guarded` that would require an update of a connection. Instead, we're going
// to modify the connection first, then the `Guarded` later.
// - Unlock `Guarded` then lock the desired connection object.
// - If there exists any "pending event" on that connection, lock the `Guarded` again and apply
// the "pending event" to the `Guarded`.
// - Inspect the state of the connection. If it is found to be inconsistent with the state earlier
// found in `Guarded`, either try again from the beginning or abort the operation. An
// inconsistency can only happen if an event has *just* happened, and considering that connections
// operate in parallel, there shouldn't be any meaningful difference between this event happening
// *just before* or *just after* the attempted operation.
// - Update the state of the connection and set the "pending event" field of that connection to
// match the modification that has just been performed.
// - Lock `Guarded` again, while keeping the connection locked.
// - Remove the "pending event" and apply it to the `Guarded`.
//
// Alternatively, `Guarded` can also be locked before updating the state of the connection. This
// skips the event phase and ensures more consistency, at the cost of keeping `Guarded` locked for
// a longer period of time.
//
// Between the moment when `Guarded` is unlocked and the moment when the connection object is
// locked, *anything* can happen. The connection can close, substream can be destroyed and
// recreated, and so on. Consequently, care must be taken to not have any ambiguity in whether or
// not the state found in `Guarded` actually matches the state found in the connection.
//
// For example, it is important that `SubstreamId`s are never reused. Otherwise, the following
// sequence of events could happen: `Guarded` is locked, a `SubstreamId` is found in `Guarded` and
// copied, `Guarded` is unlocked, the substream with that ID gets destroyed, a new substream that
// reuses the same ID is recreated, the connection is locked, the copied ID is used to reference
// the substream which is thought to be the old one but is actually the new one.
//

use super::connection::{established, handshake, NoiseKey};
use alloc::{collections::BTreeMap, string::String, sync::Arc, vec::Vec};
use core::{
    iter, mem,
    num::NonZeroUsize,
    ops::{Add, Sub},
    pin::Pin,
    task::{Context, Poll},
    time::Duration,
};
use futures::{
    channel::{mpsc, oneshot},
    lock::{Mutex, MutexGuard},
    prelude::*,
}; // TODO: no_std-ize
use rand::Rng as _;
use rand_chacha::{rand_core::SeedableRng as _, ChaCha20Rng};

pub use super::peer_id::PeerId;
pub use super::read_write::ReadWrite;
pub use established::{
    ConfigRequestResponse, ConfigRequestResponseIn, InboundError, NotificationsInClosedErr,
    NotificationsOutErr, SubstreamId,
};
pub use multiaddr::Multiaddr;
#[doc(inline)]
pub use parity_multiaddr as multiaddr;

/// Configuration for a [`Network`].
pub struct Config {
    /// Seed for the randomness within the networking state machine.
    pub randomness_seed: [u8; 32],

    /// Number of connections containers should initially allocate for.
    pub capacity: usize,

    pub notification_protocols: Vec<NotificationProtocolConfig>,

    pub request_response_protocols: Vec<ConfigRequestResponse>,

    /// Amount of time after which a connection handshake is considered to have taken too long
    /// and must be aborted.
    pub handshake_timeout: Duration,

    /// Name of the ping protocol on the network.
    pub ping_protocol: String,

    /// Key used for the encryption layer.
    /// This is a Noise static key, according to the Noise specification.
    /// Signed using the actual libp2p key.
    pub noise_key: NoiseKey,

    /// Number of events that can be buffered internally before connections are back-pressured.
    ///
    /// A good default value is 64.
    ///
    /// # Context
    ///
    /// The [`Network`] maintains an internal buffer of the events returned by
    /// [`Network::next_event`]. When [`Network::read_write`] is called, an event might get pushed
    /// to this buffer. If this buffer is full, back-pressure will be applied to the connections
    /// in order to prevent new events from being pushed.
    ///
    /// This value is important if [`Network::next_event`] is called at a slower than the calls to
    /// [`Network::read_write`] generate events.
    pub pending_api_events_buffer_size: NonZeroUsize,
}

/// Configuration for a specific overlay network.
///
/// See [`Config::notification_protocols`].
pub struct NotificationProtocolConfig {
    /// Name of the protocol negotiated on the wire.
    pub protocol_name: String,

    /// Optional alternative names for this protocol. Can represent different versions.
    ///
    /// Negotiated in order in which they are passed.
    // TODO: presently unused
    pub fallback_protocol_names: Vec<String>,

    /// Maximum size, in bytes, of the handshake that can be received.
    pub max_handshake_size: usize,

    /// Maximum size, in bytes, of a notification that can be received.
    pub max_notification_size: usize,
}

/// Identifier of a connection spawned by the [`Network`].
//
// Identifiers are never reused.
#[derive(Debug, Copy, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct ConnectionId(u64);

impl ConnectionId {
    pub fn min_value() -> Self {
        ConnectionId(u64::min_value())
    }

    pub fn max_value() -> Self {
        ConnectionId(u64::max_value())
    }
}

/// Data structure containing the list of all connections, pending or not, and their latest known
/// state. See also [the module-level documentation](..).
pub struct Network<TConn, TNow> {
    /// Fields behind a mutex.
    guarded: Mutex<Guarded<TConn, TNow>>,

    /// See [`Config::handshake_timeout`].
    handshake_timeout: Duration,

    /// See [`Config::noise_key`].
    noise_key: NoiseKey,

    /// See [`OverlayNetwork`].
    notification_protocols: Arc<[OverlayNetwork]>,

    /// See [`Config::request_response_protocols`].
    request_response_protocols: Vec<ConfigRequestResponse>,

    /// See [`Config::ping_protocol`].
    ping_protocol: String,

    /// Receiver connected to [`Guarded::events_tx`].
    events_rx: Mutex<mpsc::Receiver<Event<TConn>>>,
}

/// State of a specific overlay network.
///
/// This struct is a slight variation to [`NotificationProtocolConfig`].
struct OverlayNetwork {
    /// See [`NotificationProtocolConfig`].
    config: NotificationProtocolConfig,
}

/// Fields of [`Network`] behind a mutex.
struct Guarded<TConn, TNow> {
    /// Sender connected to [`Network::events_rx`].
    events_tx: mpsc::Sender<Event<TConn>>,

    /// Connection id to assign to the next connection.
    next_connection_id: ConnectionId,

    /// Keys are the publicly-exposed [`ConnectionId`], and they map to entries in
    /// [`Guarded::connections`].
    connections_by_id: hashbrown::HashMap<ConnectionId, usize, fnv::FnvBuildHasher>,

    /// List of all connections.
    connections: slab::Slab<Arc<Mutex<Connection<TConn, TNow>>>>,

    /// Container that holds tuples of `(connection_index, direction, substream_id)`.
    ///
    /// While the list of open substreams is also known to the [`Connection`] struct, it is
    /// duplicated here in order to avoid race conditions.
    connection_overlays: BTreeMap<(usize, SubstreamDirection, SubstreamId), SubstreamState>,

    /// Generator for randomness seeds given to the established connections.
    randomness_seeds: ChaCha20Rng,
}

/// See [`Guarded::connection_overlays`].
#[derive(Debug, Copy, Clone, PartialEq, Eq, Ord, PartialOrd, Hash)]
enum SubstreamDirection {
    /// Substream opened by the remote.
    In,
    /// Substream opened by the local node.
    Out,
}

impl SubstreamDirection {
    fn min_value() -> Self {
        SubstreamDirection::In
    }

    fn max_value() -> Self {
        SubstreamDirection::Out
    }
}

/// See [`Guarded::connection_overlays`].
///
/// > **Note**: There is no `Closed` variant, as this corresponds to a lack of entry in the
/// >           hashmap.
#[derive(Debug, Copy, Clone, PartialEq, Eq, Ord, PartialOrd, Hash)]
enum SubstreamState {
    /// Substream hasn't been accepted or refused yet. Contains the requested overlay network
    /// index.
    Pending(usize),
    Open,
}

impl<TConn, TNow> Network<TConn, TNow>
where
    TConn: Clone,
    TNow: Clone + Add<Duration, Output = TNow> + Sub<TNow, Output = Duration> + Ord,
{
    /// Initializes a new network data structure.
    pub fn new(config: Config) -> Self {
        let (events_tx, events_rx) = mpsc::channel(config.pending_api_events_buffer_size.get() - 1);

        let notification_protocols = config
            .notification_protocols
            .into_iter()
            .map(|config| OverlayNetwork { config })
            .collect::<Arc<[_]>>();

        Network {
            handshake_timeout: config.handshake_timeout,
            noise_key: config.noise_key,
            notification_protocols,
            request_response_protocols: config.request_response_protocols,
            ping_protocol: config.ping_protocol,
            events_rx: Mutex::new(events_rx),
            guarded: Mutex::new(Guarded {
                events_tx,
                next_connection_id: ConnectionId(0),
                connections_by_id: hashbrown::HashMap::with_capacity_and_hasher(
                    config.capacity,
                    Default::default(),
                ),
                connections: slab::Slab::with_capacity(config.capacity),
                connection_overlays: BTreeMap::new(),
                randomness_seeds: ChaCha20Rng::from_seed(config.randomness_seed),
            }),
        }
    }

    /// Adds a new connection to the collection.
    ///
    /// Must be passed the moment (as a `TNow`) when the connection as been established, in order
    /// to determine when the handshake timeout expires.
    ///
    /// `is_initiator` must be `true` if the connection has been initiated locally, or `false` if
    /// it has been initiated by the remote.
    pub async fn insert(
        &self,
        when_connected: TNow,
        is_initiator: bool,
        user_data: TConn,
    ) -> ConnectionId {
        let mut guarded = self.guarded.lock().await;
        let guarded = &mut *guarded;

        let connection_id = guarded.next_connection_id;
        guarded.next_connection_id.0 += 1;

        let connection_index = guarded.connections.insert(Arc::new(Mutex::new(Connection {
            connection: ConnectionInner::Handshake {
                handshake: handshake::HealthyHandshake::new(is_initiator),
                randomness_seed: guarded.randomness_seeds.gen(),
                timeout: when_connected + self.handshake_timeout,
            },
            id: connection_id,
            shutting_down: false,
            pending_event: None,
            waker: None,
            user_data,
        })));

        let _previous_value = guarded
            .connections_by_id
            .insert(connection_id, connection_index);
        debug_assert!(_previous_value.is_none());

        connection_id
    }

    /// Switches the connection to a state where it will shut down soon.
    ///
    /// Has no effect if this connection was already shutting down.
    ///
    /// Has no effect if the connection id was invalid, in order to avoid a potential race
    /// condition with the [`Event::Shutdown`] effect.
    // TODO: reconsider this "no effect if connection invalid"
    pub async fn start_shutdown(&self, connection_id: ConnectionId) {
        let connection = {
            let mut guarded = self.guarded.lock().await;
            let guarded = &mut *guarded;

            let connection_index = match guarded.connections_by_id.get(&connection_id) {
                Some(idx) => *idx,
                None => return,
            };

            guarded.connections.get(connection_index).unwrap().clone()
        };

        {
            let mut connection = connection.lock().await;
            connection.shutting_down = true;
        }

        // TODO: return something?
    }

    /// Returns the number of connections in the collection.
    ///
    /// > **Note**: Keep in mind that this method is prone to race conditions, as the user can
    /// >           add and remove connections at the same time as this is being called.
    pub async fn len(&self) -> usize {
        self.guarded.lock().await.connections_by_id.len()
    }

    /// Returns the Noise key originalled passed as [`Config::noise_key`].
    pub fn noise_key(&self) -> &NoiseKey {
        &self.noise_key
    }

    /// Returns the list the overlay networks originally passed as [`Config::notification_protocols`].
    pub fn notification_protocols(
        &self,
    ) -> impl ExactSizeIterator<Item = &NotificationProtocolConfig> {
        self.notification_protocols.iter().map(|v| &v.config)
    }

    /// Returns the list the request-response protocols originally passed as
    /// [`Config::request_response_protocols`].
    pub fn request_response_protocols(
        &self,
    ) -> impl ExactSizeIterator<Item = &ConfigRequestResponse> {
        self.request_response_protocols.iter()
    }

    /// Sends a request to the given peer, and waits for a response.
    ///
    /// This consists in:
    ///
    /// - Opening a substream on an established connection with the target.
    /// - Negotiating the requested protocol (`protocol_index`) on this substream using the
    ///   *multistream-select* protocol.
    /// - Sending the request (`request_data` parameter), prefixed with its length.
    /// - Waiting for the response (prefixed with its length), which is then returned.
    ///
    /// An error happens if the provided [`ConnectionId`] is invalid, if the connection closes
    /// while the request is in progress, if the request or response doesn't respect the protocol
    /// limits (see [`ConfigRequestResponse`]), or if the remote takes too much time to answer.
    ///
    /// As the API of this module is inherently subject to race conditions, it is never possible
    /// to guarantee that this function will succeed. [`RequestError::ConnectionClosed`] should
    /// be handled by retrying the same request again.
    ///
    /// > **Note**: This function doesn't return before the remote has answered. It is strongly
    /// >           recommended to await the returned `Future` in the background, and not block
    /// >           any important task on this.
    ///
    /// # Panic
    ///
    /// Panics if `protocol_index` isn't a valid index in [`Config::request_response_protocols`].
    ///
    pub async fn request(
        &self,
        now: TNow,
        target: ConnectionId,
        protocol_index: usize,
        request_data: Vec<u8>,
    ) -> Result<Vec<u8>, RequestError> {
        // Obtain the connect to use to send the request.
        let connection_arc: Arc<Mutex<Connection<_, _>>> = {
            let guarded = self.guarded.lock().await;

            let connection_index = *guarded
                .connections_by_id
                .get(&target)
                .ok_or(RequestError::InvalidConnection)?;
            guarded.connections[connection_index].clone()
        };

        // The response to the request will be sent on this channel.
        // The sending side is stored as a "user data" alongside with the request in the
        // underlying data structure specific to the connection.
        let (send_back, receive_result) = oneshot::channel();

        // Lock to the connection. This waits for any other call to `request`,
        // `queue_notification` or `read_write` to finish.
        let mut connection_lock = connection_arc.lock().await;

        // Actually start the request by updating the underlying state machine specific to that
        // connection.
        connection_lock
            .connection
            .as_established()
            .ok_or(RequestError::ConnectionClosed)?
            .add_request(now, protocol_index, request_data, send_back);

        // Note that no update of the `Guarded` is necessary. The `Guarded` doesn't track ongoing
        // requests.

        let waker = connection_lock.waker.take();

        // Make sure to unlock the connection before waiting for the result.
        drop(connection_lock);
        // The `Arc` to the connection should also be dropped, in order for everything to be
        // properly cleaned up if the connection closes. In particular, the channel on which
        // the response is sent back should be properly destroyed if the connection closes.
        drop(connection_arc);

        // Wake up the future returned by the latest call to `read_write` on that connection.
        if let Some(waker) = waker {
            let _ = waker.send(());
        }

        // Wait for the result of the request. Can take a long time (i.e. several seconds).
        // TODO: cancel the request if the future is dropped?
        match receive_result.await {
            Ok(r) => r,
            Err(_) => Err(RequestError::ConnectionClosed),
        }
    }

    /// Start opening a notifications substream.
    pub async fn open_notifications_substream(
        &self,
        connection_id: ConnectionId,
        overlay_network_index: usize,
        now: TNow,
        handshake: impl Into<Vec<u8>>,
    ) -> Result<SubstreamId, OpenNotificationsSubstreamError> {
        // Because the user can cancel the future at any `await` point, all the asynchronous
        // operations are performed ahead of any state modification.

        // Obtain the connect to use to send the request.
        let (connection_index, connection_arc): (_, Arc<Mutex<Connection<_, _>>>) = {
            let guarded = self.guarded.lock().await;
            let connection_index = *match guarded.connections_by_id.get(&connection_id) {
                Some(idx) => idx,
                None => return Err(OpenNotificationsSubstreamError::BadConnection),
            };

            (
                connection_index,
                guarded.connections[connection_index].clone(),
            )
        };

        // As explained in the implementation notes at the top, `guarded` must always be locked
        // after the connection.
        let mut connection_lock = connection_arc.lock().await;
        let mut guarded = self.guarded.lock().await;

        // Verify that the connection is still the same as was found before `guarded` got unlocked
        // and locked again.
        match guarded.connections.get_mut(connection_index) {
            Some(c) if Arc::ptr_eq(c, &connection_arc) => {}
            _ => return Err(OpenNotificationsSubstreamError::BadConnection),
        }

        // In order to guarantee a proper ordering of events, any pending event must first be
        // delivered.
        connection_lock.propagate_pending_event(&mut guarded).await;
        debug_assert!(connection_lock.pending_event.is_none());

        // Update the state of the inner connection state machine with the new substream.
        let substream_id = match &mut connection_lock.connection {
            ConnectionInner::Established(established) => established.open_notifications_substream(
                now,
                overlay_network_index,
                handshake.into(),
                overlay_network_index,
            ),
            ConnectionInner::Handshake { .. } => {
                return Err(OpenNotificationsSubstreamError::NotEstablished)
            }
            ConnectionInner::PendingErrorReport { .. } | ConnectionInner::Dead => {
                return Err(OpenNotificationsSubstreamError::BadConnection)
            }
            ConnectionInner::Poisoned => unreachable!(),
        };

        // Wake up the task dedicated to this connection in order for the substream to start
        // opening.
        if let Some(waker) = connection_lock.waker.take() {
            let _ = waker.send(());
        }
        drop::<MutexGuard<_>>(connection_lock);

        // Update the state of `guarded`.
        let _prev_value = guarded.connection_overlays.insert(
            (connection_index, SubstreamDirection::Out, substream_id),
            SubstreamState::Pending(overlay_network_index),
        );
        debug_assert!(_prev_value.is_none());

        Ok(substream_id)
    }

    /// Start closing a previously-open notifications substream, or cancels opening a
    /// notifications substream.
    ///
    /// Calling this method will emit a [`Event::NotificationsOutClose`] if there was indeed a
    /// notifications substream being open or already open.
    ///
    /// The interaction of this method with [`Event::NotificationsOutResult`] and
    /// [`Event::NotificationsOutClose`] is non-trivial.
    ///
    /// It is possible for a [`Event::NotificationsOutResult`] to be emitted *after* this method
    /// has been called. In any situation, if a [`Event::NotificationsOutResult`] regarding this
    /// notifications protocol is emitted in the past, then a corresponding
    /// [`Event::NotificationsOutClose`] will always be emitted.
    ///
    /// Similarly, it is possible for a [`Event::NotificationsOutClose`] to be emitted right
    /// before this method is called. For this reason, this method has no effect if no
    /// notifications substream is open.
    // TODO: review these comments about events
    pub async fn close_notifications_substream(
        &self,
        connection_id: ConnectionId,
        substream_id: SubstreamId,
    ) {
        // Because the user can cancel the future at any `await` point, all the asynchronous
        // operations are performed ahead of any state modification.

        // Obtain the connect to use to send the request.
        let (connection_index, connection_arc): (_, Arc<Mutex<Connection<_, _>>>) = {
            let guarded = self.guarded.lock().await;
            let connection_index = *match guarded.connections_by_id.get(&connection_id) {
                Some(idx) => idx,
                None => return,
            };

            (
                connection_index,
                guarded.connections[connection_index].clone(),
            )
        };

        // As explained in the implementation notes at the top, `guarded` must always be locked
        // after the connection.
        let mut connection_lock = connection_arc.lock().await;
        let mut guarded = self.guarded.lock().await;

        // Verify that the connection is still the same as was found before `guarded` got unlocked
        // and locked again.
        match guarded.connections.get_mut(connection_index) {
            Some(c) if Arc::ptr_eq(c, &connection_arc) => {}
            _ => return,
        }

        // In order to guarantee a proper ordering of events, any pending event must first be
        // delivered.
        connection_lock.propagate_pending_event(&mut guarded).await;
        debug_assert!(connection_lock.pending_event.is_none());

        // Verify that there is indeed a substream being opened or already open.
        if guarded
            .connection_overlays
            .remove(&(connection_index, SubstreamDirection::Out, substream_id))
            .is_none()
        {
            return;
        }

        // Update the state of the inner connection state machine.
        match &mut connection_lock.connection {
            ConnectionInner::Established(established) => {
                established.close_notifications_substream(substream_id);

                // Wake up the task dedicated to this connection in order for the substream to be
                // effectively closed.
                if let Some(waker) = connection_lock.waker.take() {
                    let _ = waker.send(());
                }
            }
            ConnectionInner::PendingErrorReport { .. } | ConnectionInner::Dead => {
                // It is possible that the connection has encountered an error or has been shut
                // down but is still in the state of `Guarded`.
            }
            ConnectionInner::Handshake { .. } | ConnectionInner::Poisoned => unreachable!(),
        }
    }

    /// Adds a notification to the queue of notifications to send to the given peer.
    ///
    /// Each substream maintains a queue of notifications to be sent to the remote. This method
    /// attempts to push a notification to this queue.
    ///
    /// As the API of this module is inherently subject to race conditions, it is possible for
    /// connections and substreams to no longer exist with this peer. This is a normal situation,
    /// and the error returned by this function should simply be ignored apart from diagnostic
    /// purposes.
    ///
    /// An error is also returned if the queue exceeds a certain size in bytes, for two reasons:
    ///
    /// - Since the content of the queue is transferred at a limited rate, each notification
    /// pushed at the end of the queue will take more time than the previous one to reach the
    /// destination. Once the queue reaches a certain size, the time it would take for
    /// newly-pushed notifications to reach the destination would start being unreasonably large.
    ///
    /// - If the remote deliberately applies back-pressure on the substream, it is undesirable to
    /// increase the memory usage of the local node.
    ///
    /// Similarly, the queue being full is a normal situations and notification protocols should
    /// be designed in such a way that discarding notifications shouldn't have a too negative
    /// impact.
    ///
    /// Regardless of the success of this function, no guarantee exists about the successful
    /// delivery of notifications.
    ///
    pub async fn queue_notification(
        &self,
        connection: ConnectionId,
        substream_id: SubstreamId,
        notification: impl Into<Vec<u8>>,
    ) -> Result<(), QueueNotificationError> {
        // Find which connection and substream to use to send the notification.
        // Only existing, established, substreams will be used.
        let connection_arc = {
            // TODO: ideally don't lock
            let guarded = self.guarded.lock().await;

            let connection_index = *guarded
                .connections_by_id
                .get(&connection)
                .ok_or(QueueNotificationError::InvalidConnection)?;

            // Find a substream on this connection.
            let state = *guarded
                .connection_overlays
                .get(&(connection_index, SubstreamDirection::Out, substream_id))
                .ok_or(QueueNotificationError::NoSubstream)?;

            if !matches!(state, SubstreamState::Open) {
                return Err(QueueNotificationError::NoSubstream);
            }

            guarded.connections[connection_index].clone()
        };

        // Lock to the connection. This waits for any other call to `request`,
        // `queue_notification` or `read_write` to finish.
        let mut connection_lock = connection_arc.lock().await;

        let waker = connection_lock.waker.take();

        // TODO: check if substream is still open to avoid a panic in `write_notification_unbounded`
        // TODO: return an error if queue is full
        connection_lock
            .connection
            .as_established()
            .ok_or(QueueNotificationError::InvalidConnection)?
            .write_notification_unbounded(substream_id, notification.into());

        // Note that no update of the `Guarded` is necessary. The `Guarded` doesn't track
        // notifications being sent.

        // Wake up the future returned by the latest call to `read_write` on that connection.
        drop(connection_lock);
        if let Some(waker) = waker {
            let _ = waker.send(());
        }

        Ok(())
    }

    /// Accepts a request for an inbound notifications substream reported by an
    /// [`Event::NotificationsInOpen`].
    ///
    /// Returns `Ok` if the substream has been successfully accepted (and thus opened), or `Err`
    /// if the request was obsolete, for example because the connection has been closed.
    pub async fn accept_notifications_in(
        &self,
        connection: ConnectionId,
        substream_id: SubstreamId,
        handshake: Vec<u8>,
    ) -> Result<(), ()> {
        // Find the connection and substream ID corresponding to the parameters.
        let connection_arc: Arc<Mutex<Connection<_, _>>> = {
            let guarded = self.guarded.lock().await;

            // TODO: we use defensive programming here because the concurrency model is still blurry
            let connection_index = match guarded.connections_by_id.get(&connection) {
                Some(c) => *c,
                None => return Err(()),
            };

            guarded.connections[connection_index].clone()
        };

        let mut connection_lock = connection_arc.lock().await;

        // Because the user can cancel the future at any `await` point, all the asynchronous
        // operations are performed ahead of any state modification.
        let mut guarded = self.guarded.lock().await;

        // In order to guarantee a proper ordering of events, any pending event must first be
        // delivered.
        connection_lock.propagate_pending_event(&mut guarded).await;
        debug_assert!(connection_lock.pending_event.is_none());

        // Checks whether the substream is still valid.
        let connection_index = match guarded.connections_by_id.get(&connection) {
            Some(c) => *c,
            None => return Err(()),
        };

        let overlay_network_index = match guarded.connection_overlays.remove(&(
            connection_index,
            SubstreamDirection::In,
            substream_id,
        )) {
            Some(SubstreamState::Pending(v)) => v,
            _ => return Err(()),
        };

        connection_lock
            .connection
            .as_established()
            .unwrap()
            .accept_in_notifications_substream(substream_id, handshake, overlay_network_index);

        // Wake up the connection in order for the substream confirmation to be sent back to the
        // remote.
        if let Some(waker) = connection_lock.waker.take() {
            let _ = waker.send(());
        }

        // Rather than putting a value into `pending_event`, this function immediately updates
        // `Guarded`. If this was instead done through events, the user could observe that the
        // substream hasn't been confirmed yet for as long as the event hasn't been delivered.
        let _previous_value = guarded.connection_overlays.insert(
            (connection_index, SubstreamDirection::In, substream_id),
            SubstreamState::Open,
        );
        debug_assert!(_previous_value.is_none());

        Ok(())
    }

    /// Rejects a request for an inbound notifications substream reported by an
    /// [`Event::NotificationsInOpen`].
    ///
    /// Returns `Ok` if the substream has been successfully refused, or `Err` if the request was
    /// obsolete, for example because the connection has been closed.
    pub async fn reject_notifications_in(
        &self,
        connection: ConnectionId,
        substream_id: SubstreamId,
    ) -> Result<(), ()> {
        // Find the connection and substream ID corresponding to the parameters.
        let connection_arc: Arc<Mutex<Connection<_, _>>> = {
            let guarded = self.guarded.lock().await;

            // TODO: we use defensive programming here because the concurrency model is still blurry
            let connection_index = match guarded.connections_by_id.get(&connection) {
                Some(c) => *c,
                None => return Err(()),
            };

            guarded.connections[connection_index].clone()
        };

        let mut connection_lock = connection_arc.lock().await;

        // Because the user can cancel the future at any `await` point, all the asynchronous
        // operations are performed ahead of any state modification.
        let mut guarded = self.guarded.lock().await;

        // In order to guarantee a proper ordering of events, any pending event must first be
        // delivered.
        connection_lock.propagate_pending_event(&mut guarded).await;
        debug_assert!(connection_lock.pending_event.is_none());

        // Checks whether the substream is still valid.
        let connection_index = match guarded.connections_by_id.get(&connection) {
            Some(c) => *c,
            None => return Err(()),
        };

        match guarded.connection_overlays.remove(&(
            connection_index,
            SubstreamDirection::In,
            substream_id,
        )) {
            Some(SubstreamState::Pending(_)) => {}
            _ => return Err(()),
        };

        connection_lock
            .connection
            .as_established()
            .unwrap()
            .reject_in_notifications_substream(substream_id);

        // Wake up the connection in order for the rejection to be sent back to the remote.
        if let Some(waker) = connection_lock.waker.take() {
            let _ = waker.send(());
        }

        Ok(())
    }

    /// Responds to an incoming request. Must be called in response to a [`Event::RequestIn`].
    ///
    /// Passing an `Err` corresponds, on the other side, to a
    /// [`established::RequestError::SubstreamClosed`].
    ///
    /// Has no effect if the connection has been closed in the meanwhile.
    pub async fn respond_in_request(
        &self,
        id: ConnectionId,
        substream_id: SubstreamId,
        response: Result<Vec<u8>, ()>,
    ) {
        // Find the connection corresponding to the parameter.
        let connection_arc: Arc<Mutex<Connection<_, _>>> = {
            let guarded = self.guarded.lock().await;
            match guarded.connections_by_id.get(&id) {
                Some(idx) => guarded.connections[*idx].clone(),
                None => return,
            }
        };

        let mut connection_lock = connection_arc.lock().await;

        // In order to guarantee a proper ordering of events, any pending event must first be
        // delivered.
        if connection_lock.pending_event.is_some() {
            let mut guarded = self.guarded.lock().await;
            connection_lock.propagate_pending_event(&mut guarded).await;
            debug_assert!(connection_lock.pending_event.is_none());
        }

        if let Some(connection) = connection_lock.connection.as_established() {
            // As explained in the documentation, ignore the error where the substream has
            // already been closed. This is a normal situation caused by the racy nature of the
            // API.
            match connection.respond_in_request(substream_id, response) {
                Ok(()) => {}
                Err(established::RespondInRequestError::SubstreamClosed) => {}
            }
        } else {
            // The connection no longer exists. This state mismatch is a normal situation. See
            // the implementations notes at the top of the file for more information.
            return;
        }

        // Wake up the connection in order for the substream confirmation to be sent back to the
        // remote.
        if let Some(waker) = connection_lock.waker.take() {
            let _ = waker.send(());
        }
    }

    /// Returns the next event produced by the service.
    ///
    /// This function should be called at a high enough rate that [`Network::read_write`] can
    /// continue pushing events to the internal buffer of events. Failure to call this function
    /// often enough will lead to connections being back-pressured.
    /// See also [`Config::pending_api_events_buffer_size`].
    ///
    /// It is technically possible to call this function multiple times simultaneously, in which
    /// case the events will be distributed amongst the multiple calls in an unspecified way.
    /// Keep in mind that some [`Event`]s have logic attached to the order in which they are
    /// produced, and calling this function multiple times is therefore discouraged.
    pub async fn next_event(&self) -> Event<TConn> {
        let mut events_rx = self.events_rx.lock().await;
        events_rx.next().await.unwrap()
    }

    /// Reads data coming from the connection, updates the internal state machine, and writes data
    /// destined to the connection through the [`ReadWrite`].
    ///
    /// If an error is returned, the connection should be destroyed altogether and the
    /// [`ConnectionId`] is no longer valid. You should continue calling this function until
    /// an error is returned, even if the [`ReadWrite`] indicates a full shutdown.
    ///
    /// # Panic
    ///
    /// Panics if the [`ConnectionId`] isn't a valid connection. Once this function returns an
    /// error, is no longer valid to call this function with this [`ConnectionId`].
    ///
    pub async fn read_write(
        &self,
        connection_id: ConnectionId,
        read_write: &'_ mut ReadWrite<'_, TNow>,
    ) -> Result<(), ConnectionError> {
        let (connection_index, connection_arc): (_, Arc<Mutex<Connection<_, _>>>) = {
            // TODO: ideally we wouldn't need to lock `guarded`, to reduce the possibility of lock contention
            let guarded = self.guarded.lock().await;

            let connection_index = *guarded.connections_by_id.get(&connection_id).unwrap();
            (
                connection_index,
                guarded.connections[connection_index].clone(),
            )
        };

        let mut connection_lock = connection_arc.lock().await;
        let mut connection_lock = &mut *connection_lock;

        let (tx, rx) = oneshot::channel();
        read_write.wake_up_when(ConnectionReadyFuture(rx));
        connection_lock.waker = Some(tx);

        // The code below consists in checking the connection state, calling `read_write`, and
        // checking the connection state again.
        // Since the connection state check is the same twice, this is implemented using a `for`
        // loop that loops twice and only calls `read_write` the first time.
        for is_first in [true, false] {
            // Check whether any event remains to be propagated to the API user.
            if connection_lock.pending_event.is_some() {
                let mut guarded = self.guarded.lock().await;
                connection_lock.propagate_pending_event(&mut guarded).await;
                debug_assert!(connection_lock.pending_event.is_none());
            }

            // If the connection contains an error, remove it from the state of `guarded` and
            // propagating this error.
            // This is done as a separate step for future-cancellation-related reasons.
            if matches!(
                connection_lock.connection,
                ConnectionInner::PendingErrorReport { .. }
            ) {
                let mut guarded = self.guarded.lock().await;
                future::poll_fn(|cx| guarded.events_tx.poll_ready(cx))
                    .await
                    .unwrap();

                let substreams = guarded
                    .connection_overlays
                    .range(
                        (
                            connection_index,
                            SubstreamDirection::min_value(),
                            SubstreamId::min_value(),
                        )
                            ..=(
                                connection_index,
                                SubstreamDirection::max_value(),
                                SubstreamId::max_value(),
                            ),
                    )
                    .map(|(key, _)| *key)
                    .collect::<Vec<_>>();

                let out_notification_protocols_indices = Vec::with_capacity(substreams.len());
                let in_notification_protocols_indices = Vec::with_capacity(substreams.len());

                for (_, direction, substream_id) in substreams {
                    let state = guarded
                        .connection_overlays
                        .remove(&(connection_index, direction, substream_id))
                        .unwrap();

                    match state {
                        SubstreamState::Pending(_) => continue,
                        SubstreamState::Open => {}
                    };

                    // TODO: this isn't working; find a solution
                    /*if let Some(established) = established.as_mut() {
                        let overlay_network_index = *established
                            .notifications_substream_user_data_mut(substream_id)
                            .unwrap();

                        match direction {
                            SubstreamDirection::In => {
                                in_notification_protocols_indices
                                    .push(overlay_network_index);
                            }
                            SubstreamDirection::Out => {
                                out_notification_protocols_indices
                                    .push(overlay_network_index);
                            }
                        }
                    }*/
                }

                guarded
                    .events_tx
                    .try_send(Event::Shutdown {
                        id: connection_lock.id,
                        in_notification_protocols_indices,
                        out_notification_protocols_indices,
                        user_data: connection_lock.user_data.clone(),
                    })
                    .unwrap();

                let _index = guarded.connections_by_id.remove(&connection_id);
                debug_assert_eq!(_index, Some(connection_index));

                guarded.connections.remove(connection_index);
                match mem::replace(&mut connection_lock.connection, ConnectionInner::Dead) {
                    ConnectionInner::PendingErrorReport { error } => {
                        return Err(error);
                    }
                    _ => unreachable!(),
                };
            }

            // As explained, we only call `read_write` the first time.
            if is_first {
                connection_lock.read_write(self, read_write);
            }
        }

        Ok(())
    }

    fn build_connection_config(
        &self,
        now: &TNow,
        randomness_seed: [u8; 32],
    ) -> established::Config<TNow> {
        established::Config {
            notifications_protocols: self
                .notification_protocols
                .iter()
                .flat_map(|net| {
                    let max_handshake_size = net.config.max_handshake_size;
                    let max_notification_size = net.config.max_notification_size;
                    iter::once(&net.config.protocol_name)
                        .chain(net.config.fallback_protocol_names.iter())
                        .map(move |name| {
                            established::ConfigNotifications {
                                name: name.clone(), // TODO: cloning :-/
                                max_handshake_size,
                                max_notification_size,
                            }
                        })
                })
                .collect(),
            request_protocols: self.request_response_protocols.clone(),
            randomness_seed,
            ping_protocol: self.ping_protocol.clone(), // TODO: cloning :-/
            ping_interval: Duration::from_secs(20),    // TODO: hardcoded
            ping_timeout: Duration::from_secs(10),     // TODO: hardcoded
            first_out_ping: now.clone() + Duration::from_secs(2), // TODO: hardcoded
        }
    }
}

/// Event generated by [`Network::next_event`].
#[derive(Debug)]
pub enum Event<TConn> {
    /// Handshake of the given connection has completed.
    ///
    /// This event can only happen once per connection.
    HandshakeFinished {
        /// Identifier of the connection whose handshake is finished.
        id: ConnectionId,
        /// Identity of the peer on the other side of the connection.
        peer_id: PeerId,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    /// A transport-level connection (e.g. a TCP socket) is starting its shutdown.
    ///
    /// No further event related to this connection will be reported. This event is equivalent to
    /// zero or more [`Event::NotificationsOutClose`] and [`Event::NotificationsInClose`] events
    /// grouped into one.
    ///
    /// Keep in mind that this event can happen for connections that haven't finished their
    /// handshake.
    ///
    /// > **Note**: The [`Network::read_write`] method signals the end of the connection
    /// >           orthogonally to this shutdown event. In other words, you must still continue
    /// >           to call [`Network::read_write`] with that [`ConnectionId`] until it is no
    /// >           longer necessary.
    // TODO: add reason for shutdown
    Shutdown {
        id: ConnectionId,
        out_notification_protocols_indices: Vec<usize>,
        in_notification_protocols_indices: Vec<usize>,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    /// Received an incoming substream, but this substream has produced an error.
    ///
    /// > **Note**: This event exists only for diagnostic purposes. No action is expected in
    /// >           return.
    InboundError {
        id: ConnectionId,
        /// Error that happened.
        error: InboundError,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    /// Received a request from a request-response protocol.
    RequestIn {
        id: ConnectionId,
        /// Substream on which the request has been received. Must be passed back when providing
        /// the response.
        substream_id: SubstreamId,
        protocol_index: usize,
        request_payload: Vec<u8>,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    /// Outcome of trying to open a substream with [`Network::open_notifications_substream`].
    ///
    /// If `Ok`, it is now possible to send notifications on this substream.
    /// If `Err`, the substream no longer exists.
    NotificationsOutResult {
        id: ConnectionId,
        // TODO: what if fallback?
        substream_id: SubstreamId,
        notifications_protocol_index: usize,
        /// If `Ok`, contains the handshake sent back by the remote. Its interpretation is out of
        /// scope of this module.
        result: Result<Vec<u8>, NotificationsOutErr>,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    /// A previously open outbound substream has been closed by the remote.
    NotificationsOutClose {
        id: ConnectionId,
        substream_id: SubstreamId,
        notifications_protocol_index: usize,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    ///
    NotificationsInOpen {
        id: ConnectionId,
        /// Substream on which the request has been received. Must be passed back when accepting
        /// or refusing the substream.
        substream_id: SubstreamId,
        notifications_protocol_index: usize,
        remote_handshake: Vec<u8>,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    // TODO: needs a notifications in cancel event? tricky
    /// Received a notification on a notifications substream of a connection.
    NotificationsIn {
        id: ConnectionId,
        substream_id: SubstreamId,
        notifications_protocol_index: usize,
        notification: Vec<u8>,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    NotificationsInClose {
        id: ConnectionId,
        substream_id: SubstreamId,
        notifications_protocol_index: usize,
        outcome: Result<(), NotificationsInClosedErr>,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },

    /// An outgoing ping has succeeded. This event is generated automatically over time for each
    /// connection in the collection.
    PingOutSuccess {
        id: ConnectionId,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },
    /// An outgoing ping has failed. This event is generated automatically over time for each
    /// connection in the collection.
    PingOutFailed {
        id: ConnectionId,
        /// Copy of the user data provided when creating the connection.
        user_data: TConn,
    },
}

impl<TConn> Event<TConn> {
    /// Returns the identifier of the connection concerned by this event.
    pub fn connection_id(&self) -> ConnectionId {
        match self {
            Event::HandshakeFinished { id, .. } => *id,
            Event::Shutdown { id, .. } => *id,
            Event::InboundError { id, .. } => *id,
            Event::RequestIn { id, .. } => *id,
            Event::NotificationsOutResult { id, .. } => *id,
            Event::NotificationsOutClose { id, .. } => *id,
            Event::NotificationsInOpen { id, .. } => *id,
            Event::NotificationsIn { id, .. } => *id,
            Event::NotificationsInClose { id, .. } => *id,
            Event::PingOutSuccess { id, .. } => *id,
            Event::PingOutFailed { id, .. } => *id,
        }
    }

    /// Returns the user data of the connection concerned by this event.
    pub fn user_data(&self) -> &TConn {
        match self {
            Event::HandshakeFinished { user_data, .. } => user_data,
            Event::Shutdown { user_data, .. } => user_data,
            Event::InboundError { user_data, .. } => user_data,
            Event::RequestIn { user_data, .. } => user_data,
            Event::NotificationsOutResult { user_data, .. } => user_data,
            Event::NotificationsOutClose { user_data, .. } => user_data,
            Event::NotificationsInOpen { user_data, .. } => user_data,
            Event::NotificationsIn { user_data, .. } => user_data,
            Event::NotificationsInClose { user_data, .. } => user_data,
            Event::PingOutSuccess { user_data, .. } => user_data,
            Event::PingOutFailed { user_data, .. } => user_data,
        }
    }
}

/// Future ready when a connection has data to give out via [`Network::read_write`].
#[pin_project::pin_project]
pub struct ConnectionReadyFuture(#[pin] oneshot::Receiver<()>);

impl Future for ConnectionReadyFuture {
    type Output = ();

    fn poll(self: Pin<&mut Self>, cx: &mut Context) -> Poll<()> {
        // `Err(Canceled)` is mapped to `Poll::Pending`, meaning that the `oneshot::Sender` can
        // be dropped in order to never notify this future.
        match self.project().0.poll(cx) {
            Poll::Pending => Poll::Pending,
            Poll::Ready(Ok(())) => Poll::Ready(()),
            Poll::Ready(Err(_)) => Poll::Pending,
        }
    }
}

/// Error within the context of a connection. See [`Network::read_write`].
#[derive(Debug, derive_more::Display)]
pub enum ConnectionError {
    /// Protocol error after the connection has been established.
    #[display(fmt = "{}", _0)]
    Established(established::Error),
    /// Eror during the handshake phase.
    #[display(fmt = "{}", _0)]
    Handshake(HandshakeError),
    /// Connection was shut down by calling [`Network::start_shutdown`].
    // TODO: that seems hacky
    LocalShutdown,
    /// Connection was gracefully terminated. Can only happen if the connection was established,
    /// as an EOF during the handshake is an error.
    Eof,
}

/// Protocol error within the context of a connection. See [`Network::read_write`].
#[derive(Debug, derive_more::Display)]
pub enum HandshakeError {
    /// The handshake took too long.
    Timeout,
    /// Protocol error.
    #[display(fmt = "{}", _0)]
    Protocol(handshake::HandshakeError),
}

/// See [`Network::open_notifications_substream`].
#[derive(Debug, derive_more::Display)]
pub enum OpenNotificationsSubstreamError {
    /// The demanded [`ConnectionId`] isn't or is no longer valid.
    ///
    /// Keep in mind that connections can be closed by the remote at any time. This error being
    /// returned doesn't indicate a state mismatch, but rather usually that the connection has
    /// just been closed, and that corresponding events have been queued but not necessarily
    /// processed yet.
    BadConnection,

    /// The connection is still in its handshake phase.
    NotEstablished,
}

/// Data structure holding the state of a single established (i.e. post-handshake) connection.
///
/// This data structure is wrapped around `Arc<Mutex<>>`. As such, its fields do not necessarily
/// match the state in the [`Network`].
struct Connection<TConn, TNow> {
    /// State machine of the underlying connection.
    connection: ConnectionInner<TNow>,

    /// Copy of the id of the connection.
    id: ConnectionId,

    /// `true` is [`Network::start_shutdown`] has been called.
    shutting_down: bool,

    /// Event that has just happened on the connection, but that the [`Guarded`] isn't yet aware
    /// of. See the implementations note at the top of the file for more information.
    pending_event: Option<PendingEvent>,

    /// A sender is stored here when the user calls [`Network::read_write`]. The receiving part
    /// notifies the user that they must call [`Network::read_write`] again.
    /// Send a value on that channel in order to notify that data is potentially available to be
    /// sent on the socket, or that the user should call [`Network::read_write`] in general.
    waker: Option<oneshot::Sender<()>>,

    user_data: TConn,
}

impl<TConn, TNow> Connection<TConn, TNow>
where
    TConn: Clone,
    TNow: Clone + Add<Duration, Output = TNow> + Sub<TNow, Output = Duration> + Ord,
{
    fn read_write(&mut self, parent: &Network<TConn, TNow>, read_write: &mut ReadWrite<TNow>) {
        debug_assert!(self.pending_event.is_none());

        match mem::replace(&mut self.connection, ConnectionInner::Poisoned) {
            ConnectionInner::Established(_) if self.shutting_down => {
                // TODO: shut down should be graceful instead of an error
                debug_assert!(self.pending_event.is_none());
                self.connection = ConnectionInner::PendingErrorReport {
                    error: ConnectionError::LocalShutdown,
                };
            }

            ConnectionInner::Handshake { .. } if self.shutting_down => {
                // TODO: shut down should be graceful instead of an error
                debug_assert!(self.pending_event.is_none());
                self.connection = ConnectionInner::PendingErrorReport {
                    error: ConnectionError::LocalShutdown,
                };
            }

            ConnectionInner::Established(connection) => {
                let rw_before = (read_write.read_bytes, read_write.written_bytes);

                match connection.read_write(read_write) {
                    Ok((connection, event)) => {
                        if read_write.is_dead() && event.is_none() {
                            self.connection = ConnectionInner::PendingErrorReport {
                                error: ConnectionError::Eof,
                            };
                        } else {
                            self.connection = ConnectionInner::Established(connection);
                        }

                        // If the inner `read_write` has processed some bytes or generated an
                        // event, we ask the user to call `read_write` again.
                        if rw_before != (read_write.read_bytes, read_write.written_bytes)
                            || event.is_some()
                        {
                            if let Some(waker) = self.waker.take() {
                                let _ = waker.send(());
                            }
                        }

                        if let Some(event) = event {
                            debug_assert!(self.pending_event.is_none());
                            self.pending_event = Some(PendingEvent::Inner(event));
                        }
                    }
                    Err(err) => {
                        debug_assert!(self.pending_event.is_none());
                        self.connection = ConnectionInner::PendingErrorReport {
                            error: ConnectionError::Established(err),
                        };
                    }
                }
            }

            ConnectionInner::Handshake {
                mut handshake,
                randomness_seed,
                timeout,
            } => {
                // Check that the handshake isn't taking too long.
                if timeout < read_write.now {
                    self.connection = ConnectionInner::PendingErrorReport {
                        error: ConnectionError::Handshake(HandshakeError::Timeout),
                    };
                    return;
                }
                read_write.wake_up_after(&timeout);

                loop {
                    let (read_before, written_before) =
                        (read_write.read_bytes, read_write.written_bytes);

                    let result = match handshake.read_write(read_write) {
                        Ok(rw) => rw,
                        Err(err) => {
                            self.connection = ConnectionInner::PendingErrorReport {
                                error: ConnectionError::Handshake(HandshakeError::Protocol(err)),
                            };
                            return;
                        }
                    };

                    match result {
                        handshake::Handshake::Healthy(updated_handshake)
                            if (read_before, written_before)
                                == (read_write.read_bytes, read_write.written_bytes) =>
                        {
                            self.connection = ConnectionInner::Handshake {
                                handshake: updated_handshake,
                                randomness_seed,
                                timeout,
                            };
                            break;
                        }
                        handshake::Handshake::Healthy(updated_handshake) => {
                            handshake = updated_handshake;
                        }
                        handshake::Handshake::Success {
                            remote_peer_id,
                            connection,
                        } => {
                            debug_assert!(self.pending_event.is_none());
                            self.pending_event =
                                Some(PendingEvent::HandshakeFinished(remote_peer_id));
                            self.connection = ConnectionInner::Established(
                                connection.into_connection(
                                    parent
                                        .build_connection_config(&read_write.now, randomness_seed),
                                ),
                            );
                            break;
                        }
                        handshake::Handshake::NoiseKeyRequired(key) => {
                            handshake = key.resume(&parent.noise_key);
                        }
                    }
                }
            }

            // The `read_write` function shouldn't have been called when in this state.
            ConnectionInner::Dead
            | ConnectionInner::PendingErrorReport { .. }
            | ConnectionInner::Poisoned => unreachable!(),
        }
    }

    /// Removes the pending event stored within that connection and updates the [`Guarded`]
    /// accordingly.
    /// See the implementations notes at the top of the file for more information.
    async fn propagate_pending_event(&mut self, guarded: &mut Guarded<TConn, TNow>) {
        if self.pending_event.is_none() {
            return;
        }

        let connection_index = *guarded.connections_by_id.get(&self.id).unwrap();

        // The body of this function consists in two operations: extracting the event from
        // `pending_event`, then sending a corresponding event on `events_tx`. Because sending an
        // item on `events_tx` might block, and because the user is free to cancel the future at
        // it heir will, it is possible for the event to have been extracted then get lost into
        // oblivion.
        //
        // To prevent this from happening, we first wait for `events_tx` to be ready to accept
        // an item, then use `try_send` to send items on it in a non-blocking way.
        future::poll_fn(|cx| guarded.events_tx.poll_ready(cx))
            .await
            .unwrap();

        // The line below does `pending_event.take()`. After this, no more `await` must be present
        // in the function's body without precautionnary measures.
        match self.pending_event.take().unwrap() {
            PendingEvent::HandshakeFinished(peer_id) => {
                // Check that no substream has been opened yet.
                debug_assert_eq!(
                    guarded
                        .connection_overlays
                        .range(
                            (
                                connection_index,
                                SubstreamDirection::min_value(),
                                SubstreamId::min_value()
                            )
                                ..=(
                                    connection_index,
                                    SubstreamDirection::max_value(),
                                    SubstreamId::max_value()
                                )
                        )
                        .count(),
                    0
                );

                guarded
                    .events_tx
                    .try_send(Event::HandshakeFinished {
                        id: self.id,
                        peer_id,
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::InboundError(error)) => {
                guarded
                    .events_tx
                    .try_send(Event::InboundError {
                        id: self.id,
                        error,
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::RequestIn {
                id: substream_id,
                protocol_index,
                request,
            }) => {
                guarded
                    .events_tx
                    .try_send(Event::RequestIn {
                        id: self.id,
                        substream_id,
                        protocol_index,
                        request_payload: request,
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::Response {
                response,
                user_data: send_back,
                ..
            }) => {
                let _ = send_back.send(response.map_err(RequestError::Connection));
            }
            PendingEvent::Inner(established::Event::NotificationsInOpen {
                id: substream_id,
                protocol_index: overlay_network_index,
                handshake,
            }) => {
                guarded.connection_overlays.insert(
                    (connection_index, SubstreamDirection::In, substream_id),
                    SubstreamState::Pending(overlay_network_index),
                );

                guarded
                    .events_tx
                    .try_send(Event::NotificationsInOpen {
                        id: self.id,
                        substream_id,
                        notifications_protocol_index: overlay_network_index,
                        remote_handshake: handshake,
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::NotificationsInOpenCancel {
                id: substream_id,
                protocol_index: overlay_network_index,
            }) => {
                let _value_in = guarded.connection_overlays.remove(&(
                    connection_index,
                    SubstreamDirection::In,
                    substream_id,
                ));

                debug_assert_eq!(
                    _value_in,
                    Some(SubstreamState::Pending(overlay_network_index))
                );
            }
            PendingEvent::Inner(established::Event::NotificationIn {
                id: substream_id,
                notification,
            }) => {
                let overlay_network_index = *self
                    .connection
                    .as_established()
                    .unwrap() // TODO: shouldn't unwrap here
                    .notifications_substream_user_data_mut(substream_id)
                    .unwrap();

                debug_assert_eq!(
                    *guarded
                        .connection_overlays
                        .get(&(connection_index, SubstreamDirection::In, substream_id,))
                        .unwrap(),
                    SubstreamState::Open,
                );

                guarded
                    .events_tx
                    .try_send(Event::NotificationsIn {
                        id: self.id,
                        substream_id,
                        notifications_protocol_index: overlay_network_index,
                        notification,
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::NotificationsInClose {
                id: substream_id,
                protocol_index: overlay_network_index,
                outcome,
            }) => {
                let _was_in = guarded.connection_overlays.remove(&(
                    connection_index,
                    SubstreamDirection::In,
                    substream_id,
                ));
                debug_assert_eq!(_was_in, Some(SubstreamState::Open));

                guarded
                    .events_tx
                    .try_send(Event::NotificationsInClose {
                        id: self.id,
                        substream_id,
                        notifications_protocol_index: overlay_network_index,
                        user_data: self.user_data.clone(),
                        outcome,
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::NotificationsOutResult {
                id: substream_id,
                result,
            }) => {
                let overlay_network_index = match result {
                    Ok(_) => *self
                        .connection
                        .as_established()
                        .unwrap() // TODO: shouldn't unwrap here
                        .notifications_substream_user_data_mut(substream_id)
                        .unwrap(),
                    Err((_, idx)) => idx,
                };

                let _value_removed = guarded.connection_overlays.remove(&(
                    connection_index,
                    SubstreamDirection::Out,
                    substream_id,
                ));
                debug_assert_eq!(
                    _value_removed,
                    Some(SubstreamState::Pending(overlay_network_index))
                );

                if result.is_ok() {
                    let _prev_value = guarded.connection_overlays.insert(
                        (connection_index, SubstreamDirection::Out, substream_id),
                        SubstreamState::Open,
                    );
                    debug_assert_eq!(_prev_value, None);
                }

                guarded
                    .events_tx
                    .try_send(Event::NotificationsOutResult {
                        id: self.id,
                        substream_id,
                        notifications_protocol_index: overlay_network_index,
                        result: result.map_err(|(err, _)| err),
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::NotificationsOutCloseDemanded { .. }) => {
                todo!()
            }
            PendingEvent::Inner(established::Event::NotificationsOutReset {
                id: substream_id,
                user_data: overlay_network_index,
            }) => {
                let _value_removed = guarded.connection_overlays.remove(&(
                    connection_index,
                    SubstreamDirection::Out,
                    substream_id,
                ));
                debug_assert_eq!(_value_removed, Some(SubstreamState::Open));

                guarded
                    .events_tx
                    .try_send(Event::NotificationsOutClose {
                        id: self.id,
                        substream_id,
                        notifications_protocol_index: overlay_network_index,
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::PingOutSuccess) => {
                guarded
                    .events_tx
                    .try_send(Event::PingOutSuccess {
                        id: self.id,
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
            PendingEvent::Inner(established::Event::PingOutFailed) => {
                guarded
                    .events_tx
                    .try_send(Event::PingOutFailed {
                        id: self.id,
                        user_data: self.user_data.clone(),
                    })
                    .unwrap();
            }
        }
    }
}

enum ConnectionInner<TNow> {
    Handshake {
        handshake: handshake::HealthyHandshake,

        /// Seed that will be used to initialize randomness when building the
        /// [`established::Established`].
        /// This seed is computed during the handshake in order to avoid locking a `Mutex` (and
        /// having to deal with future-cancellation-related issues) when the handshake is over.
        /// While it seems a bit dangerous to leave a randomness seed in plain memory, the
        /// randomness isn't used for anything critical or related to cryptography, but only for
        /// example to avoid hash collision attacks.
        randomness_seed: [u8; 32],

        /// When the handshake times out.
        timeout: TNow,
    },
    Established(
        established::Established<TNow, oneshot::Sender<Result<Vec<u8>, RequestError>>, usize>,
    ),

    /// The connection has generated an error, but this error hasn't been reported to the API
    /// user yet. The connection is still considered as valid for API purposes. After the error
    /// has been reported to the user, the connection switches to the [`ConnectionInner::Dead`]
    /// state.
    PendingErrorReport { error: ConnectionError },

    /// Connection is no longer valid because an error has happened in the connection
    /// or because it has been gracefully shut down. It is invalid for the user to call
    /// `read_write` with this connection again.
    ///
    /// This is the state the connection is in after it has been removed from the [`Guarded`] but
    /// before it is destroyed.
    Dead,

    /// Temporary state. For borrow checking purposes, the state can be transitioned to this state
    /// then back.
    Poisoned,
}

impl<TNow> ConnectionInner<TNow> {
    fn as_established(
        &mut self,
    ) -> Option<
        &mut established::Established<TNow, oneshot::Sender<Result<Vec<u8>, RequestError>>, usize>,
    > {
        if let ConnectionInner::Established(c) = self {
            Some(c)
        } else {
            None
        }
    }
}

enum PendingEvent {
    HandshakeFinished(PeerId),
    Inner(established::Event<oneshot::Sender<Result<Vec<u8>, RequestError>>, usize>),
}

/// Error potentially returned by [`Network::request`].
#[derive(Debug, derive_more::Display)]
pub enum RequestError {
    /// Connection no longer exists.
    ///
    /// > **Note**: Connections can shut down at any point, after which this error will be returned
    /// >           if you try to use it. This error is not (necessarily) a bad usage of the API
    /// >           but can happen in normal situations.
    InvalidConnection,

    /// Connection has been unexpectedly closed by the remote during the request.
    ConnectionClosed,

    /// Error in the context of the connection.
    Connection(established::RequestError),
}

/// Error potentially returned by [`Network::queue_notification`].
#[derive(Debug, derive_more::Display)]
pub enum QueueNotificationError {
    /// Connection no longer exists.
    ///
    /// > **Note**: Connections can shut down at any point, after which this error will be returned
    /// >           if you try to use it. This error is not (necessarily) a bad usage of the API
    /// >           but can happen in normal situations.
    InvalidConnection,

    /// No substream with the given target of the given protocol.
    NoSubstream,

    /// Queue of notifications with that peer is full.
    QueueFull,
}
