//! Generators of elements
//!
//! Generator<T> is a fully specified way to get you an arbitrary T object
//!
//! The general interface works very similarly to an iterator, but instead
//! it never stop generating data, thus instead of returning an Option<Item>
//! it returns the Item directly, and takes an extra random generator
//! to generate the next element.

use super::rand::R;

mod base;
pub mod char;
mod collection;
mod combinators;
mod numerical;
mod product;
pub mod string;

pub use base::*;
pub use collection::*;
pub use combinators::*;
pub use numerical::*;
pub use product::*;

/// Dependent generator where the second items depends on what has been generated by the first generator
pub struct Depends<G, F> {
    src_gen: G,
    dst_gen: F,
}

impl<G1, G2, F> Generator for Depends<G1, F>
where
    G1: Generator,
    G2: Generator,
    F: Fn(&G1::Item) -> G2,
{
    type Item = (G1::Item, G2::Item);
    fn gen(&self, r: &mut R) -> Self::Item {
        let x = self.src_gen.gen(&mut r.sub());
        let g2 = (self.dst_gen)(&x);
        let y = g2.gen(&mut r.sub());
        (x, y)
    }
}

pub fn depends<F, G1, G2>(g1: G1, f: F) -> Depends<G1, F>
where
    G1: Generator,
    G2: Generator,
    F: FnOnce(&G1::Item) -> G2,
{
    Depends {
        src_gen: g1,
        dst_gen: f,
    }
}
