use crate::errors::*;

use crate::base::{ArrayProperties, ValueProperties, NatureContinuous, Nature, Vector1DNull, Jagged, NatureCategorical, DataType};
use crate::utilities::get_common_value;
use itertools::Itertools;
use noisy_float::types::n64;
use num::ToPrimitive;

fn take<T: Clone>(vector: &[T], index: usize) -> Result<T> {
    match vector.get(index) {
        Some(value) => Ok(value.clone()),
        None => Err("property column index is out of bounds".into())
    }
}

pub fn select_properties(properties: &ArrayProperties, index: usize) -> Result<ValueProperties> {
    let mut properties = properties.clone();
    properties.num_columns = Some(1);
    properties.dimensionality = Some(1);
    if let Some(nature) = &properties.nature {
        properties.nature = Some(match nature {
            Nature::Continuous(continuous) => Nature::Continuous(NatureContinuous {
                lower: match &continuous.lower {
                    Vector1DNull::Float(lower) => Vector1DNull::Float(vec![take(lower, index)?]),
                    Vector1DNull::Int(lower) => Vector1DNull::Int(vec![take(lower, index)?]),
                    _ => return Err("lower must be numeric".into())
                },
                upper: match &continuous.upper {
                    Vector1DNull::Float(upper) => Vector1DNull::Float(vec![take(upper, index)?]),
                    Vector1DNull::Int(upper) => Vector1DNull::Int(vec![take(upper, index)?]),
                    _ => return Err("upper must be numeric".into())
                },
            }),
            Nature::Categorical(categorical) => Nature::Categorical(NatureCategorical {
                categories: match &categorical.categories {
                    Jagged::Float(cats) => Jagged::Float(vec![take(&cats, index)?]),
                    Jagged::Int(cats) => Jagged::Int(vec![take(&cats, index)?]),
                    Jagged::Bool(cats) => Jagged::Bool(vec![take(&cats, index)?]),
                    Jagged::Str(cats) => Jagged::Str(vec![take(&cats, index)?]),
                }
            })
        })
    }
    Ok(ValueProperties::Array(properties))
}

pub fn stack_properties(
    all_properties: &[ValueProperties], dimensionality: Option<i64>, node_id: u32
) -> Result<ValueProperties> {
    let all_properties = all_properties.iter()
        .map(|property| Ok(property.array()?.clone()))
        .collect::<Result<Vec<ArrayProperties>>>()?;

    let num_records = get_common_value(&all_properties.iter()
        .map(|prop| prop.num_records).collect()).unwrap_or(None);
    let dataset_id = get_common_value(&all_properties.iter()
        .map(|prop| prop.dataset_id).collect()).unwrap_or(None);

    if num_records.is_none() && dataset_id.is_none() {
        return Err("dataset may not be conformable".into())
    }

    if all_properties.iter().any(|prop| prop.aggregator.is_some()) {
        return Err("indexing is not currently supported on aggregated data".into())
    }

    let data_type = get_common_value(&all_properties.iter().map(|prop| prop.data_type.clone()).collect())
        .ok_or_else(|| Error::from("dataset must have homogeneous type"))?;

    let group_id = get_common_value(&all_properties.iter()
        .map(|v| v.group_id.clone()).collect())
        .ok_or_else(|| "group_id: must be homogeneous")?;

    // TODO: preserve nature when indexing
    let natures = all_properties.iter()
        .map(|prop| prop.nature.as_ref())
        .collect::<Vec<Option<&Nature>>>();

    let nature = get_common_continuous_nature(&natures, data_type.to_owned())
        .or_else(|| get_common_categorical_nature(&natures));

    if !all_properties.iter().all(|prop| prop.naturally_ordered) && dataset_id.is_none() {
        return Err("cannot stack columns that may have been reordered".into())
    }

    let sample_proportion = get_common_value(&all_properties.iter().map(|prop| prop.sample_proportion.map(n64)).collect())
        .ok_or_else(|| Error::from("sample proportions must be shared in common"))?
        .and_then(|v| v.to_f64());

    if sample_proportion.is_some() && dataset_id.is_none() {
        return Err(Error::from("sampled data must come from a common source"))
    }

    Ok(ValueProperties::Array(ArrayProperties {
        num_records,
        num_columns: all_properties.iter()
            .map(|prop| prop.num_columns)
            .try_fold(0, |total, num| num.map(|v| total + v)),
        nullity: get_common_value(&all_properties.iter().map(|prop| prop.nullity).collect()).unwrap_or(true),
        releasable: get_common_value(&all_properties.iter().map(|prop| prop.releasable).collect()).unwrap_or(true),
        c_stability: get_common_value(&all_properties.iter().map(|prop| prop.c_stability).collect())
            .ok_or_else(|| Error::from("c-stabilities must be shared among all arguments"))?,
        aggregator: None,
        nature,
        data_type,
        dataset_id,
        node_id: node_id as i64,
        is_not_empty: all_properties.iter().all(|prop| prop.is_not_empty),
        dimensionality,
        group_id,
        naturally_ordered: true,
        sample_proportion
    }))
}

fn get_common_continuous_nature(natures: &[Option<&Nature>], data_type: DataType) -> Option<Nature> {
    let lower: Vector1DNull = natures.iter().map(|nature| match nature {
        Some(Nature::Continuous(nature)) => Some(nature.lower.clone()),
        Some(Nature::Categorical(_)) => None,
        _ => Some(match data_type {
            DataType::Float => Vector1DNull::Float(vec![None]),
            DataType::Int => Vector1DNull::Int(vec![None]),
            _ => return None
        })
    }).collect::<Option<Vec<Vector1DNull>>>()?.into_iter()
        .map(Ok).fold1(concat_vector1d_null)?.ok()?;

    let upper: Vector1DNull = natures.iter().map(|nature| match nature {
        Some(Nature::Continuous(nature)) => Some(nature.upper.clone()),
        Some(Nature::Categorical(_)) => None,
        None => Some(match data_type {
            DataType::Float => Vector1DNull::Float(vec![None]),
            DataType::Int => Vector1DNull::Int(vec![None]),
            _ => return None
        })
    }).collect::<Option<Vec<Vector1DNull>>>()?.into_iter()
        .map(Ok).fold1(concat_vector1d_null)?.ok()?;

    Some(Nature::Continuous(NatureContinuous {
        lower, upper
    }))
}

fn get_common_categorical_nature(natures: &[Option<&Nature>]) -> Option<Nature> {
    let categories = natures.iter().map(|nature| match nature {
        Some(Nature::Categorical(nature)) => Some(nature.categories.clone()),
        Some(Nature::Continuous(_)) => None,
        None => None
    }).collect::<Option<Vec<Jagged>>>()?.into_iter()
        .map(Ok).fold1(concat_jagged)?.ok()?;

    Some(Nature::Categorical(NatureCategorical {
        categories
    }))
}

fn concat_vector1d_null(a: Result<Vector1DNull>, b: Result<Vector1DNull>) -> Result<Vector1DNull> {
    Ok(match (a?, b?) {
        (Vector1DNull::Float(a), Vector1DNull::Float(b)) =>
            Vector1DNull::Float([&a[..], &b[..]].concat()),
        (Vector1DNull::Int(a), Vector1DNull::Int(b)) =>
            Vector1DNull::Int([&a[..], &b[..]].concat()),
        (Vector1DNull::Bool(a), Vector1DNull::Bool(b)) =>
            Vector1DNull::Bool([&a[..], &b[..]].concat()),
        (Vector1DNull::Str(a), Vector1DNull::Str(b)) =>
            Vector1DNull::Str([&a[..], &b[..]].concat()),
        _ => return Err("attempt to concatenate non-homogenously typed vectors".into())
    })
}

fn concat_jagged(a: Result<Jagged>, b: Result<Jagged>) -> Result<Jagged> {
    Ok(match (a?, b?) {
        (Jagged::Float(a), Jagged::Float(b)) =>
            Jagged::Float([&a[..], &b[..]].concat()),
        (Jagged::Int(a), Jagged::Int(b)) =>
            Jagged::Int([&a[..], &b[..]].concat()),
        (Jagged::Bool(a), Jagged::Bool(b)) =>
            Jagged::Bool([&a[..], &b[..]].concat()),
        (Jagged::Str(a), Jagged::Str(b)) =>
            Jagged::Str([&a[..], &b[..]].concat()),
        _ => return Err("attempt to concatenate non-homogenously typed vectors".into())
    })
}
