use std::{ptr, slice};

use crate::Drain;

/// A splicing iterator for `SmallVec`.
///
/// This struct is created by [`splice`][crate::SmallVec::splice].
/// See its documentation for more.
///
/// # Example
///
/// ```
/// # use small_vec2::{SmallVec, Splice};
/// let mut v: SmallVec<_, 3> = SmallVec::from(vec![0, 1, 2]);
/// let new = [7, 8];
/// let iter: Splice<_, 3> = v.splice(1.., new.iter().cloned());
/// ```
#[derive(Debug)]
pub struct Splice<'a, I: Iterator + 'a, const N: usize> {
    pub(crate) drain: Drain<'a, I::Item, N>,
    pub(crate) replace_with: I,
}

impl<'a, I: Iterator + 'a, const N: usize> Splice<'a, I, N> {
    pub fn new(drain: Drain<'a, I::Item, N>, replace_with: I) -> Self {
        Self {
            drain,
            replace_with,
        }
    }
}

impl<I: Iterator, const N: usize> Iterator for Splice<'_, I, N> {
    type Item = I::Item;

    fn next(&mut self) -> Option<Self::Item> {
        self.drain.next()
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        self.drain.size_hint()
    }
}

impl<I: Iterator, const N: usize> DoubleEndedIterator for Splice<'_, I, N> {
    fn next_back(&mut self) -> Option<Self::Item> {
        self.drain.next_back()
    }
}

impl<I: Iterator, const N: usize> ExactSizeIterator for Splice<'_, I, N> {}

impl<I: Iterator, const N: usize> Drop for Splice<'_, I, N> {
    fn drop(&mut self) {
        self.drain.by_ref().for_each(drop);

        unsafe {
            if self.drain.tail_len == 0 {
                //这块未测试
                self.drain.vec.as_mut().extend(self.replace_with.by_ref());
                return;
            }

            // First fill the range left by drain().
            let fill = self.drain.fill(&mut self.replace_with);
            if !fill {
                return;
            }

            // There may be more elements. Use the lower bound as an estimate.
            // FIXME: Is the upper bound a better guess? Or something else?

            let (lower_bound, _upper_bound) = self.replace_with.size_hint();
            if lower_bound > 0 {
                self.drain.move_tail(lower_bound);
                let fill = self.drain.fill(&mut self.replace_with);
                if !fill {
                    return;
                }
            }
            // Collect any remaining elements.
            // This is a zero-length vector which does not allocate if `lower_bound` was exact.
            let mut collected = self
                .replace_with
                .by_ref()
                .collect::<Vec<I::Item>>()
                .into_iter();
            // Now we have an exact count.
            if collected.len() > 0 {
                self.drain.move_tail(collected.len());
                let filled = self.drain.fill(&mut collected);
                debug_assert!(filled);
                debug_assert_eq!(collected.len(), 0);
            }
        }
        // Let `Drain::drop` move the tail back if necessary and restore `vec.len`.
    }
}
/// Private helper methods for [`Splice::drop`]
impl<T, const N: usize> Drain<'_, T, N> {
    /// The range from `self.vec.len` to `self.tail_start` contains elements
    /// that have been moved out.
    /// Fill that range as much as possible with new elements from the `replace_with` iterator.
    /// Returns `true` if we filled the entire range. (`replace_with.next()` didn’t return `None`.)
    pub(crate) unsafe fn fill<I: Iterator<Item = T>>(&mut self, replace_with: &mut I) -> bool {
        let vec = unsafe { self.vec.as_mut() };

        let range_start = vec.len;
        let range_end = self.tail_start;

        let data = unsafe { vec.as_mut_ptr().add(range_start) };
        let range_len = range_end - range_start;
        let range_slice = unsafe { slice::from_raw_parts_mut(data, range_len) };

        for place in range_slice {
            if let Some(new_item) = replace_with.next() {
                unsafe { ptr::write(place, new_item) };
                vec.len += 1;
            } else {
                return false;
            }
        }
        true
    }

    /// Makes room for inserting more elements before the tail.
    pub(crate) unsafe fn move_tail(&mut self, additional: usize) {
        let small_vec = unsafe { self.vec.as_mut() };
        let len = self.tail_start + self.tail_len;

        let src = unsafe { small_vec.as_ptr().add(self.tail_start) };
        let src_value = unsafe { ptr::read(src) };

        let a = small_vec.spilled();

        small_vec.reserve_exact(additional + len);

        let new_tail_start = self.tail_start + additional;
        let b = small_vec.spilled();
        if a == b {
            unsafe {
                let src = small_vec.as_ptr().add(self.tail_start);
                let dst = small_vec.as_mut_ptr().add(new_tail_start);
                ptr::copy(src, dst, self.tail_len);
            }
        } else {
            unsafe {
                let src = &src_value as *const T;
                let dst = small_vec.as_mut_ptr().add(new_tail_start);
                ptr::copy(src, dst, self.tail_len);
            }
        }
        self.tail_start = new_tail_start;
    }
}
