use std::{fmt, iter::FusedIterator, ptr, slice};

use crate::SmallVec;

/// By-value iterator for `SmallVec`.
pub struct IntoIter<T, const N: usize> {
    pub(crate) index: usize,
    pub(crate) end: usize,
    pub(crate) buf: SmallVec<T, N>,
}

impl<T, const N: usize> IntoIter<T, N> {
    pub fn new(index: usize, end: usize, buf: SmallVec<T, N>) -> Self {
        Self { index, end, buf }
    }

    /// Returns the remaining items of this iterator as a slice.
    pub fn as_slice(&self) -> &[T] {
        let len = self.end - self.index;
        unsafe { slice::from_raw_parts(self.buf.as_ptr().add(self.index), len) }
    }

    /// Returns the remaining items of this iterator as a mutable slice.
    pub fn as_mut_slice(&mut self) -> &mut [T] {
        let len = self.end - self.index;
        unsafe { slice::from_raw_parts_mut(self.buf.as_mut_ptr().add(self.index), len) }
    }
}

impl<T, const N: usize> Iterator for IntoIter<T, N> {
    type Item = T;

    fn next(&mut self) -> Option<Self::Item> {
        if self.index == self.end {
            None
        } else {
            unsafe {
                let index = self.index;
                self.index = index + 1;
                Some(ptr::read(self.buf.as_ptr().add(index)))
            }
        }
    }

    fn size_hint(&self) -> (usize, Option<usize>) {
        let len = self.end - self.index;
        (len, Some(len))
    }
}

impl<T, const N: usize> DoubleEndedIterator for IntoIter<T, N> {
    fn next_back(&mut self) -> Option<Self::Item> {
        if self.index == self.end {
            None
        } else {
            unsafe {
                self.end -= 1;
                Some(ptr::read(self.buf.as_ptr().add(self.end)))
            }
        }
    }
}

impl<T, const N: usize> ExactSizeIterator for IntoIter<T, N> {}
impl<T, const N: usize> FusedIterator for IntoIter<T, N> {}

impl<T, const N: usize> Drop for IntoIter<T, N> {
    fn drop(&mut self) {
        // panic safety: Set length to 0 before dropping elements.
        let index = self.index;
        let len = self.end - index;
        unsafe {
            self.buf.set_len(0);
            let elements = slice::from_raw_parts_mut(self.buf.as_mut_ptr().add(index), len);
            ptr::drop_in_place(elements);
        }
    }
}

impl<T, const N: usize> Clone for IntoIter<T, N>
where
    T: Clone,
{
    fn clone(&self) -> IntoIter<T, N> {
        let mut v = SmallVec::new();
        v.extend_from_slice(self.as_slice());
        v.into_iter()
    }
}

impl<T, const N: usize> fmt::Debug for IntoIter<T, N>
where
    T: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_tuple("IntoIter").field(&self.as_slice()).finish()
    }
}
