use std::mem::{ManuallyDrop, MaybeUninit};
/// An array, either inline (of statically-fixed size) or heap-allocated (of dynamic size).
pub(crate) union Storage<T, const N: usize> {
    pub(crate) inline: ManuallyDrop<[MaybeUninit<T>; N]>,
    pub(crate) heap: ManuallyDrop<Vec<T>>,
}
unsafe impl<T: Send, const N: usize> Send for Storage<T, N> {}
unsafe impl<T: Sync, const N: usize> Sync for Storage<T, N> {}

impl<T, const N: usize> Storage<T, N> {
    pub(crate) fn from_heap(heap: Vec<T>) -> Self {
        Self {
            heap: ManuallyDrop::new(heap),
        }
    }

    pub(crate) fn from_inline(inline: [MaybeUninit<T>; N]) -> Self {
        Self {
            inline: ManuallyDrop::new(inline),
        }
    }

    pub(crate) unsafe fn inline(&self) -> *const T {
        unsafe { self.inline.as_ptr().cast() }
    }

    pub(crate) unsafe fn inline_mut(&mut self) -> *mut T {
        unsafe { self.inline.as_mut_ptr().cast() }
    }

    pub(crate) unsafe fn into_inline(self) -> [MaybeUninit<T>; N] {
        ManuallyDrop::into_inner(unsafe { self.inline })
    }

    pub(crate) unsafe fn heap(&self) -> *const T {
        unsafe { self.heap.as_ptr() }
    }

    pub(crate) unsafe fn heap_mut(&mut self) -> *mut T {
        unsafe { self.heap.as_mut_ptr() }
    }
}
