use crate::{SmallVec, TryReserveError};
use std::{
    fmt::Debug,
    iter::FromIterator,
    mem::{size_of, swap},
    ops::Bound::*,
    panic::{catch_unwind, AssertUnwindSafe},
    ptr,
    rc::Rc,
};

#[test]
fn from_into_inner() {
    let vec: SmallVec<_, 3> = SmallVec::from(vec![1, 2, 3]);
    let ptr = vec.as_ptr();
    let vec = vec.into_iter().collect::<SmallVec<_, 3>>();
    assert_eq!(vec, [1, 2, 3]);
    assert!(ptr != vec.as_ptr());

    let ptr = &vec[1] as *const _;
    let mut it = vec.into_iter();
    it.next().unwrap();
    let vec = it.collect::<SmallVec<_, 3>>();
    assert_eq!(vec, [2, 3]);
    assert!(ptr != vec.as_ptr());

    let vec: SmallVec<_, 2> = SmallVec::from(vec![1, 2, 3]);
    let ptr = vec.as_ptr();
    let vec = vec.into_iter().collect::<SmallVec<_, 2>>();
    assert_eq!(vec, [1, 2, 3]);
    assert_eq!(ptr, vec.as_ptr());

    let ptr = &vec[1] as *const _;
    let mut it = vec.into_iter();
    it.next().unwrap();
    let vec = it.collect::<SmallVec<_, 2>>();
    assert_eq!(vec, [2, 3]);
    assert!(ptr != vec.as_ptr());
}
#[test]
fn test_into_iter_clone() {
    fn iter_equal<I: Iterator<Item = i32>>(it: I, slice: &[i32]) {
        let v: SmallVec<i32, 3> = it.collect();
        assert_eq!(v, slice);
    }
    let mut it = SmallVec::<_, 3>::from(vec![1, 2, 3]).into_iter();
    iter_equal(it.clone(), &[1, 2, 3]);
    assert_eq!(it.next(), Some(1));
    let mut it = it.rev();
    iter_equal(it.clone(), &[3, 2]);
    assert_eq!(it.next(), Some(3));
    iter_equal(it.clone(), &[2]);
    assert_eq!(it.next(), Some(2));
    iter_equal(it.clone(), &[]);
    assert_eq!(it.next(), None);

    let mut it = SmallVec::<_, 2>::from(vec![1, 2, 3]).into_iter();
    iter_equal(it.clone(), &[1, 2, 3]);
    assert_eq!(it.next(), Some(1));
    let mut it = it.rev();
    iter_equal(it.clone(), &[3, 2]);
    assert_eq!(it.next(), Some(3));
    iter_equal(it.clone(), &[2]);
    assert_eq!(it.next(), Some(2));
    iter_equal(it.clone(), &[]);
    assert_eq!(it.next(), None);
}
#[test]
fn test_from_iter_specialization() {
    let src: SmallVec<_, 1> = SmallVec::from(vec![0usize; 1]);
    let srcptr = src.as_ptr();
    let sink = src.into_iter().collect::<SmallVec<_, 1>>();
    let sinkptr = sink.as_ptr();
    assert!(srcptr != sinkptr);

    let src: SmallVec<_, 1> = SmallVec::from(vec![0usize; 2]);
    let srcptr = src.as_ptr();
    let sink = src.into_iter().collect::<SmallVec<_, 1>>();
    let sinkptr = sink.as_ptr();
    assert_eq!(srcptr, sinkptr);
}

#[test]
fn test_from_iter_partially_drained_in_place_specialization() {
    let src: SmallVec<_, 2> = SmallVec::from(vec![0usize; 10]);
    let srcptr = src.as_ptr();
    let mut iter = src.into_iter();
    iter.next();
    iter.next();
    let sink = iter.collect::<SmallVec<_, 2>>();
    let sinkptr = sink.as_ptr();
    assert_eq!(srcptr, sinkptr);

    let src: SmallVec<_, 10> = SmallVec::from(vec![0usize; 10]);
    let srcptr = src.as_ptr();
    let mut iter = src.into_iter();
    iter.next();
    iter.next();
    let sink = iter.collect::<SmallVec<_, 10>>();
    let sinkptr = sink.as_ptr();
    assert!(srcptr != sinkptr);
}

#[test]
fn test_from_iter_specialization_with_iterator_adapters() {
    let src: SmallVec<_, 2> = SmallVec::from(vec![0usize; 256]);
    let srcptr = src.as_ptr();
    let iter = src
        .into_iter()
        .enumerate()
        .map(|i| i.0 + i.1)
        .zip(std::iter::repeat(1usize))
        .map(|(a, b)| a + b)
        .map_while(Option::Some)
        .peekable()
        .skip(1)
        .map(|e| {
            if e != usize::MAX {
                Ok(std::num::NonZeroUsize::new(e))
            } else {
                Err(())
            }
        });
    let sink = iter.collect::<Result<SmallVec<_, 2>, _>>().unwrap();
    let sinkptr = sink.as_ptr();
    assert!(srcptr != sinkptr as *const usize);
    // assert_eq!(srcptr, sinkptr as *const usize);

    let src: SmallVec<_, 256> = SmallVec::from(vec![0usize; 256]);
    let srcptr = src.as_ptr();
    let iter = src
        .into_iter()
        .enumerate()
        .map(|i| i.0 + i.1)
        .zip(std::iter::repeat(1usize))
        .map(|(a, b)| a + b)
        .map_while(Option::Some)
        .peekable()
        .skip(1)
        .map(|e| {
            if e != usize::MAX {
                Ok(std::num::NonZeroUsize::new(e))
            } else {
                Err(())
            }
        });
    let sink = iter.collect::<Result<SmallVec<_, 256>, _>>().unwrap();
    let sinkptr = sink.as_ptr();
    assert!(srcptr != sinkptr as *const usize);
    // assert_eq!(srcptr, sinkptr as *const usize);
}

#[test]
fn test_from_iter_specialization_head_tail_drop() {
    let drop_count: SmallVec<_, 3> = (0..=2).map(|_| Rc::new(())).collect();
    let src: SmallVec<_, 3> = drop_count.iter().cloned().collect();
    let srcptr = src.as_ptr();
    let iter = src.into_iter();
    let sink: SmallVec<_, 3> = iter.skip(1).take(1).collect();
    let sinkptr = sink.as_ptr();
    assert!(srcptr != sinkptr, "specialization was applied");
    assert_eq!(Rc::strong_count(&drop_count[0]), 1, "front was dropped");
    assert_eq!(
        Rc::strong_count(&drop_count[1]),
        2,
        "one element was collected"
    );
    assert_eq!(Rc::strong_count(&drop_count[2]), 1, "tail was dropped");
    assert_eq!(sink.len(), 1);

    let drop_count: SmallVec<_, 2> = (0..=2).map(|_| Rc::new(())).collect();
    let src: SmallVec<_, 2> = drop_count.iter().cloned().collect();
    let srcptr = src.as_ptr();
    let iter = src.into_iter();
    let sink: SmallVec<_, 1> = iter.skip(1).take(1).collect();
    let sinkptr = sink.as_ptr();
    assert!(srcptr != sinkptr, "specialization was applied");
    assert_eq!(Rc::strong_count(&drop_count[0]), 1, "front was dropped");
    assert_eq!(
        Rc::strong_count(&drop_count[1]),
        2,
        "one element was collected"
    );
    assert_eq!(Rc::strong_count(&drop_count[2]), 1, "tail was dropped");
    assert_eq!(sink.len(), 1);
}
#[test]
fn test_from_iter_specialization_panic_during_drop_leaks() {
    static mut DROP_COUNTER: usize = 0;

    #[derive(Debug, Clone)]
    enum Droppable {
        DroppedTwice(Box<i32>),
        PanicOnDrop,
    }

    impl Drop for Droppable {
        fn drop(&mut self) {
            match self {
                Droppable::DroppedTwice(_) => {
                    unsafe {
                        DROP_COUNTER += 1;
                    }
                    println!("Dropping!")
                }
                Droppable::PanicOnDrop => {
                    if !std::thread::panicking() {
                        panic!();
                    }
                }
            }
        }
    }

    let mut to_free: *mut Droppable = core::ptr::null_mut();
    let mut cap = 0;

    let _ = std::panic::catch_unwind(AssertUnwindSafe(|| {
        let mut v: SmallVec<_, 2> = SmallVec::from(vec![
            Droppable::DroppedTwice(Box::new(123)),
            Droppable::PanicOnDrop,
        ]);
        to_free = v.as_mut_ptr();
        cap = v.capacity();
        let _ = v.into_iter().take(0).collect::<SmallVec<_, 2>>();
    }));

    assert_eq!(unsafe { DROP_COUNTER }, 1);
}

#[test]

fn test_extend_from_within_panicing_clone() {
    use std::sync::atomic::{AtomicU32, Ordering};
    struct Panic<'dc> {
        drop_count: &'dc AtomicU32,
        aaaaa: bool,
    }

    impl Clone for Panic<'_> {
        fn clone(&self) -> Self {
            if self.aaaaa {
                panic!("panic! at the clone");
            }

            Self { ..*self }
        }
    }

    impl Drop for Panic<'_> {
        fn drop(&mut self) {
            self.drop_count.fetch_add(1, Ordering::SeqCst);
        }
    }

    let count = AtomicU32::new(0);
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![
        Panic {
            drop_count: &count,
            aaaaa: false,
        },
        Panic {
            drop_count: &count,
            aaaaa: true,
        },
        Panic {
            drop_count: &count,
            aaaaa: false,
        },
    ]);

    // This should clone & append one Panic{..} at the end, and then panic while
    // cloning second Panic{..}. This means that `Panic::drop` should be called
    // 4 times (3 for items already in vector, 1 for just appended).
    //
    // Previously just appended item was leaked, making drop_count = 3, instead of 4.
    catch_unwind(move || vec.extend_from_within(..)).unwrap_err();

    assert_eq!(count.load(Ordering::SeqCst), 4);
}
struct DropCounter<'a> {
    count: &'a mut u32,
}

impl Drop for DropCounter<'_> {
    fn drop(&mut self) {
        *self.count += 1;
    }
}

#[test]
fn test_small_vec_struct() {
    assert_eq!(size_of::<SmallVec<u8, 8>>(), size_of::<usize>() * 5);
    assert_eq!(size_of::<Vec<u8>>(), size_of::<usize>() * 3);
    assert_eq!(size_of::<SmallVec<usize, 8>>(), size_of::<usize>() * 10);
    assert_eq!(size_of::<Vec<usize>>(), size_of::<usize>() * 3);
}

#[test]
fn test_double_drop() {
    struct TwoSmallVec<T, const CAP: usize> {
        x: SmallVec<T, CAP>,
        y: SmallVec<T, CAP>,
    }

    let (mut count_x, mut count_y) = (0, 0);
    {
        let mut tv: TwoSmallVec<DropCounter<'_>, 2> = TwoSmallVec {
            x: SmallVec::new(),
            y: SmallVec::new(),
        };
        tv.x.push(DropCounter {
            count: &mut count_x,
        });
        tv.y.push(DropCounter {
            count: &mut count_y,
        });

        // If SmallVec had a drop flag, here is where it would be zeroed.
        // Instead, it should rely on its internal state to prevent
        // doing anything significant when dropped multiple times.
        drop(tv.x);

        // Here tv goes out of scope, tv.y should be dropped, but not tv.x.
    }

    assert_eq!(count_x, 1);
    assert_eq!(count_y, 1);
}
#[test]
fn test_drain_max_vec_size() {
    let mut v = SmallVec::<(), 5>::with_capacity(usize::MAX);
    unsafe {
        v.set_len(usize::MAX);
    }
    for _ in v.drain(usize::MAX - 1..) {}
    assert_eq!(v.len(), usize::MAX - 1);

    let mut v = SmallVec::<(), 5>::with_capacity(usize::MAX);
    unsafe {
        v.set_len(usize::MAX);
    }
    for _ in v.drain(usize::MAX - 1..=usize::MAX - 1) {}
    assert_eq!(v.len(), usize::MAX - 1);
}
#[test]
fn test_reserve() {
    let mut v: SmallVec<i32, 0> = SmallVec::new();
    assert_eq!(v.capacity(), 0);

    v.reserve(2);
    assert_eq!(v.capacity(), 2);
    assert!(v.capacity() >= 2);

    for i in 0..16 {
        v.push(i);
    }
    assert_eq!(v.capacity(), 16);
    assert!(v.capacity() >= 16);
    v.reserve(16);
    assert_eq!(v.capacity(), 32);
    assert!(v.capacity() >= 32);

    v.push(16);

    v.reserve(16);
    assert_eq!(v.capacity(), 64);
    assert!(v.capacity() >= 33)
}

#[test]
fn test_zst_capacity() {
    assert_eq!(SmallVec::<(), 2>::new().capacity(), usize::MAX);
}

#[test]
fn test_indexing() {
    let v: SmallVec<isize, 2> = SmallVec::from([10, 20]);
    assert_eq!(v[0], 10);
    assert_eq!(v[1], 20);
    let mut x: usize = 0;
    assert_eq!(v[x], 10);
    assert_eq!(v[x + 1], 20);
    x = x + 1;
    assert_eq!(v[x], 20);
    assert_eq!(v[x - 1], 10);

    let v: SmallVec<isize, 2> = SmallVec::from(vec![10, 20, 30]);
    assert_eq!(v[0], 10);
    assert_eq!(v[1], 20);
    let mut x: usize = 0;
    assert_eq!(v[x], 10);
    assert_eq!(v[x + 1], 20);
    x = x + 1;
    assert_eq!(v[x], 20);
    assert_eq!(v[x - 1], 10);
}

#[test]
fn test_debug_fmt() {
    let vec1: SmallVec<isize, 2> = SmallVec::new();
    assert_eq!("[]", format!("{:?}", vec1));

    let vec2: SmallVec<isize, 2> = SmallVec::from([0, 1]);
    assert_eq!("[0, 1]", format!("{:?}", vec2));

    let slice: &[isize] = &[4, 5];
    assert_eq!("[4, 5]", format!("{:?}", slice));
}

#[test]
fn test_push() {
    let mut v: SmallVec<isize, 2> = SmallVec::new();
    v.push(1);
    assert_eq!(v, [1]);
    assert_eq!([1], v);
    v.push(2);
    assert_eq!(v, [1, 2]);
    assert_eq!([1, 2], v);
    v.push(3);
    assert_eq!(v, [1, 2, 3]);
}

#[test]
fn test_extend() {
    let mut v: SmallVec<isize, 2> = SmallVec::new();
    let mut w: SmallVec<isize, 2> = SmallVec::new();

    v.extend(w.clone());
    assert_eq!(v, []);

    v.extend(0..3);
    for i in 0..3 {
        w.push(i)
    }

    assert_eq!(v, w);

    v.extend(3..10);
    for i in 3..10 {
        w.push(i)
    }

    assert_eq!(v, w);

    v.extend(w.clone());
    assert!(v.iter().eq(w.iter().chain(w.iter())));

    // Zero sized types
    #[derive(PartialEq, Debug)]
    struct Foo;

    let mut a: SmallVec<_, 2> = SmallVec::new();
    let b: SmallVec<_, 2> = SmallVec::from(vec![Foo, Foo]);

    a.extend(b);
    assert_eq!(a, [Foo, Foo]);

    // Double drop
    let mut count_x = 0;
    {
        let mut x: SmallVec<_, 2> = SmallVec::new();
        let y = vec![DropCounter {
            count: &mut count_x,
        }];
        x.extend(y);
    }
    assert_eq!(count_x, 1);
}

#[test]
fn test_extend_from_slice() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let b: SmallVec<_, 5> = SmallVec::from(vec![6, 7, 8, 9, 0]);

    v.extend_from_slice(&b);

    assert_eq!(v, [1, 2, 3, 4, 5, 6, 7, 8, 9, 0]);
}

#[test]
fn test_slice_from_ref() {
    let values: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let slice = &values[1..3];

    assert_eq!(slice, [2, 3]);
}

#[test]
fn test_slice_from_mut() {
    let mut values: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    {
        let slice = &mut values[2..];
        assert!(slice == [3, 4, 5]);
        for p in slice {
            *p += 2;
        }
    }

    assert!(values == [1, 2, 5, 6, 7]);
}

#[test]
fn test_slice_to_mut() {
    let mut values: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    {
        let slice = &mut values[..2];
        assert!(slice == [1, 2]);
        for p in slice {
            *p += 1;
        }
    }

    assert!(values == [2, 3, 3, 4, 5]);
}

#[test]
fn test_split_at_mut() {
    let mut values: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    {
        let (left, right) = values.split_at_mut(2);
        {
            let left: &[_] = left;
            assert!(&left[..left.len()] == &[1, 2]);
        }
        for p in left {
            *p += 1;
        }

        {
            let right: &[_] = right;
            assert!(&right[..right.len()] == &[3, 4, 5]);
        }
        for p in right {
            *p += 2;
        }
    }

    assert_eq!(values, [2, 3, 5, 6, 7]);
}

#[test]
fn test_clone() {
    let v: SmallVec<u8, 5> = SmallVec::from(vec![]);
    let w: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);

    assert_eq!(v, v.clone());

    let z = w.clone();
    assert_eq!(w, z);
    // they should be disjoint in memory.
    assert!(w.as_ptr() != z.as_ptr())
}

#[test]
fn test_clone_from() {
    let mut v: SmallVec<Box<i32>, 5> = SmallVec::from(vec![]);
    let three: SmallVec<_, 5> = SmallVec::from(vec![box 1, box 2, box 3]);
    let two: SmallVec<_, 5> = SmallVec::from(vec![box 4, box 5]);
    // zero, long
    v.clone_from(&three);
    assert_eq!(v, three);

    // equal
    v.clone_from(&three);
    assert_eq!(v, three);

    // long, short
    v.clone_from(&two);
    assert_eq!(v, two);

    // short, long
    v.clone_from(&three);
    assert_eq!(v, three)
}

#[test]
fn test_retain() {
    let mut vec: SmallVec<_, 4> = SmallVec::from(vec![1, 2, 3, 4]);
    vec.retain(|x| *x % 2 == 0);
    assert_eq!(vec, [2, 4]);
}

#[test]
fn test_retain_pred_panic_with_hole() {
    let v = (0..5).map(Rc::new).collect::<SmallVec<_, 5>>();
    catch_unwind(AssertUnwindSafe(|| {
        let mut v = v.clone();
        v.retain(|r| match **r {
            0 => true,
            1 => false,
            2 => true,
            _ => panic!(),
        });
    }))
    .unwrap_err();
    // Everything is dropped when predicate panicked.
    assert!(v.iter().all(|r| Rc::strong_count(r) == 1));
}

#[test]
fn test_retain_pred_panic_no_hole() {
    let v = (0..5).map(Rc::new).collect::<SmallVec<_, 5>>();
    catch_unwind(AssertUnwindSafe(|| {
        let mut v = v.clone();
        v.retain(|r| match **r {
            0 | 1 | 2 => true,
            _ => panic!(),
        });
    }))
    .unwrap_err();
    // Everything is dropped when predicate panicked.
    assert!(v.iter().all(|r| Rc::strong_count(r) == 1));
}

#[test]
fn test_retain_drop_panic() {
    #[derive(Clone)]
    struct Wrap(Rc<i32>);

    impl Drop for Wrap {
        fn drop(&mut self) {
            if *self.0 == 3 {
                panic!();
            }
        }
    }

    let v = (0..5).map(|x| Rc::new(x)).collect::<SmallVec<_, 5>>();
    catch_unwind(AssertUnwindSafe(|| {
        let mut v = v
            .iter()
            .map(|r| Wrap(r.clone()))
            .collect::<SmallVec<_, 5>>();
        v.retain(|w| match *w.0 {
            0 => true,
            1 => false,
            2 => true,
            3 => false, // Drop panic.
            _ => true,
        });
    }))
    .unwrap_err();
    // Other elements are dropped when `drop` of one element panicked.
    // The panicked wrapper also has its Rc dropped.
    assert!(v.iter().all(|r| Rc::strong_count(r) == 1));
}

#[test]
fn test_dedup() {
    fn case<const A: usize, const B: usize>(a: SmallVec<u32, A>, b: SmallVec<u32, B>) {
        let mut v = a;
        v.dedup();
        assert_eq!(v, b);
    }
    case(
        SmallVec::<u32, 5>::from(vec![]),
        SmallVec::<u32, 4>::from(vec![]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![1]),
        SmallVec::<u32, 5>::from(vec![1]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![1, 1]),
        SmallVec::<u32, 5>::from(vec![1]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![1, 2, 3]),
        SmallVec::<u32, 5>::from(vec![1, 2, 3]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![1, 1, 2, 3]),
        SmallVec::<u32, 5>::from(vec![1, 2, 3]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![1, 2, 2, 3]),
        SmallVec::<u32, 5>::from(vec![1, 2, 3]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![1, 2, 3, 3]),
        SmallVec::<u32, 5>::from(vec![1, 2, 3]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![1, 1, 2, 2, 2, 3, 3]),
        SmallVec::<u32, 5>::from(vec![1, 2, 3]),
    );
}

#[test]
fn test_dedup_by_key() {
    fn case<const A: usize, const B: usize>(a: SmallVec<u32, A>, b: SmallVec<u32, B>) {
        let mut v = a;
        v.dedup_by_key(|i| *i / 10);
        assert_eq!(v, b);
    }
    case(
        SmallVec::<u32, 4>::from(vec![]),
        SmallVec::<u32, 5>::from(vec![]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![10]),
        SmallVec::<u32, 5>::from(vec![10]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![10, 11]),
        SmallVec::<u32, 5>::from(vec![10]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![10, 20, 30]),
        SmallVec::<u32, 5>::from(vec![10, 20, 30]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![10, 11, 20, 30]),
        SmallVec::<u32, 5>::from(vec![10, 20, 30]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![10, 20, 21, 30]),
        SmallVec::<u32, 5>::from(vec![10, 20, 30]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![10, 20, 30, 31]),
        SmallVec::<u32, 5>::from(vec![10, 20, 30]),
    );
    case(
        SmallVec::<u32, 5>::from(vec![10, 11, 20, 21, 22, 30, 31]),
        SmallVec::<u32, 5>::from(vec![10, 20, 30]),
    );
}

#[test]
fn test_dedup_by() {
    let mut vec: SmallVec<&str, 5> = SmallVec::from(vec!["foo", "bar", "Bar", "baz", "bar"]);
    vec.dedup_by(|a, b| a.eq_ignore_ascii_case(b));

    assert_eq!(vec, ["foo", "bar", "baz", "bar"]);

    let mut vec: SmallVec<(&str, u32), 5> = SmallVec::from(vec![
        ("foo", 1),
        ("foo", 2),
        ("bar", 3),
        ("bar", 4),
        ("bar", 5),
    ]);
    vec.dedup_by(|a, b| {
        a.0 == b.0 && {
            b.1 += a.1;
            true
        }
    });

    assert_eq!(vec, [("foo", 3), ("bar", 12)]);
}

#[test]
fn zero_sized_values() {
    let mut v: SmallVec<(), 5> = SmallVec::from(Vec::new());
    assert_eq!(v.len(), 0);
    v.push(());
    assert_eq!(v.len(), 1);
    v.push(());
    assert_eq!(v.len(), 2);
    assert_eq!(v.pop(), Some(()));
    assert_eq!(v.pop(), Some(()));
    assert_eq!(v.pop(), None);

    assert_eq!(v.iter().count(), 0);
    v.push(());
    assert_eq!(v.iter().count(), 1);
    v.push(());
    assert_eq!(v.iter().count(), 2);

    for &() in &v {}

    assert_eq!(v.iter_mut().count(), 2);
    v.push(());
    assert_eq!(v.iter_mut().count(), 3);
    v.push(());
    assert_eq!(v.iter_mut().count(), 4);

    for &mut () in &mut v {}
    unsafe {
        v.set_len(0);
    }
    assert_eq!(v.iter_mut().count(), 0);
}

#[test]
fn test_partition() {
    assert_eq!(
        SmallVec::<i32, 5>::from(Vec::new())
            .into_iter()
            .partition(|x: &i32| *x < 3),
        (
            SmallVec::<i32, 5>::from(Vec::new()),
            SmallVec::<i32, 5>::from(Vec::new())
        )
    );
    assert_eq!(
        SmallVec::<i32, 5>::from(vec![1, 2, 3])
            .into_iter()
            .partition(|x| *x < 4),
        (
            SmallVec::<i32, 5>::from(vec![1, 2, 3]),
            SmallVec::<i32, 5>::from(Vec::new())
        )
    );
    assert_eq!(
        SmallVec::<i32, 5>::from(vec![1, 2, 3])
            .into_iter()
            .partition(|x| *x < 2),
        (
            SmallVec::<i32, 5>::from(vec![1]),
            SmallVec::<i32, 5>::from(vec![2, 3])
        )
    );
    assert_eq!(
        SmallVec::<i32, 5>::from(vec![1, 2, 3])
            .into_iter()
            .partition(|x| *x < 0),
        (
            SmallVec::<i32, 5>::from(vec![]),
            SmallVec::<i32, 5>::from(vec![1, 2, 3])
        )
    );
}

#[test]
fn test_zip_unzip() {
    let z1: SmallVec<(i32, i32), 5> = SmallVec::from(vec![(1, 4), (2, 5), (3, 6)]);

    let (left, right): (SmallVec<i32, 5>, SmallVec<i32, 5>) = z1.iter().cloned().unzip();

    assert_eq!((1, 4), (left[0], right[0]));
    assert_eq!((2, 5), (left[1], right[1]));
    assert_eq!((3, 6), (left[2], right[2]));
}

#[test]
fn test_cmp() {
    let x: &[isize] = &[1, 2, 3, 4, 5];
    let cmp: &[isize] = &[1, 2, 3, 4, 5];
    assert_eq!(&x[..], cmp);
    let cmp: &[isize] = &[3, 4, 5];
    assert_eq!(&x[2..], cmp);
    let cmp: &[isize] = &[1, 2, 3];
    assert_eq!(&x[..3], cmp);
    let cmp: &[isize] = &[2, 3, 4];
    assert_eq!(&x[1..4], cmp);

    let x: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let cmp: &[isize] = &[1, 2, 3, 4, 5];
    assert_eq!(&x[..], cmp);
    let cmp: &[isize] = &[3, 4, 5];
    assert_eq!(&x[2..], cmp);
    let cmp: &[isize] = &[1, 2, 3];
    assert_eq!(&x[..3], cmp);
    let cmp: &[isize] = &[2, 3, 4];
    assert_eq!(&x[1..4], cmp);
}

#[test]
fn test_vec_truncate_drop() {
    static mut DROPS: u32 = 0;
    struct Elem(i32);
    impl Drop for Elem {
        fn drop(&mut self) {
            unsafe {
                DROPS += 1;
            }
        }
    }

    let mut v: SmallVec<_, 5> = SmallVec::from(vec![Elem(1), Elem(2), Elem(3), Elem(4), Elem(5)]);
    assert_eq!(unsafe { DROPS }, 0);
    v.truncate(3);
    assert_eq!(unsafe { DROPS }, 2);
    v.truncate(0);
    assert_eq!(unsafe { DROPS }, 5);
}

#[test]
#[should_panic]
fn test_vec_truncate_fail() {
    struct BadElem(i32);
    impl Drop for BadElem {
        fn drop(&mut self) {
            let BadElem(ref mut x) = *self;
            if *x == 0xbadbeef {
                panic!("BadElem panic: 0xbadbeef")
            }
        }
    }

    let mut v: SmallVec<_, 5> =
        SmallVec::from(vec![BadElem(1), BadElem(2), BadElem(0xbadbeef), BadElem(4)]);
    v.truncate(0);
}

#[test]
fn test_index() {
    let vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    assert!(vec[1] == 2);
}

#[test]
#[should_panic]
fn test_index_out_of_bounds() {
    let vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    let _ = vec[3];
}

#[test]
#[should_panic]
fn test_slice_out_of_bounds_1() {
    let x: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let _var_name = &x[!0..];
}

#[test]
#[should_panic]
fn test_slice_out_of_bounds_2() {
    let x: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let _var_name = &x[..6];
}

#[test]
#[should_panic]
fn test_slice_out_of_bounds_3() {
    let x: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let _var_name = &x[!0..4];
}

#[test]
#[should_panic]
fn test_slice_out_of_bounds_4() {
    let x: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let _var_name = &x[1..6];
}

#[test]
#[should_panic]
fn test_slice_out_of_bounds_5() {
    let x: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let _var_name = &x[3..2];
}

#[test]
#[should_panic]
fn test_swap_remove_empty() {
    let mut vec = SmallVec::<i32, 1>::new();
    vec.swap_remove(0);
}

#[test]
fn test_move_items() {
    let vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    let mut vec2: SmallVec<_, 5> = SmallVec::from(vec![]);
    for i in vec {
        vec2.push(i);
    }
    assert_eq!(vec2, [1, 2, 3]);
}

#[test]
fn test_move_items_reverse() {
    let vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    let mut vec2: SmallVec<_, 5> = SmallVec::from(vec![]);
    for i in vec.into_iter().rev() {
        vec2.push(i);
    }
    assert_eq!(vec2, [3, 2, 1]);
}

#[test]
fn test_move_items_zero_sized() {
    let vec: SmallVec<_, 5> = SmallVec::from(vec![(), (), ()]);
    let mut vec2: SmallVec<_, 5> = SmallVec::from(vec![]);
    for i in vec {
        vec2.push(i);
    }
    assert_eq!(vec2, [(), (), ()]);
}

#[test]
fn test_drain_empty_vec() {
    let mut vec: SmallVec<i32, 5> = SmallVec::from(vec![]);
    let mut vec2: SmallVec<_, 5> = SmallVec::from(vec![]);
    for i in vec.drain(..) {
        vec2.push(i);
    }
    assert!(vec.is_empty());
    assert!(vec2.is_empty());
}

#[test]
fn test_drain_items() {
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    let mut vec2: SmallVec<_, 5> = SmallVec::from(vec![]);
    for i in vec.drain(..) {
        vec2.push(i);
    }
    assert_eq!(vec, []);
    assert_eq!(vec2, [1, 2, 3]);
}

#[test]
fn test_drain_items_reverse() {
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    let mut vec2: SmallVec<_, 5> = SmallVec::from(vec![]);
    for i in vec.drain(..).rev() {
        vec2.push(i);
    }
    assert_eq!(vec, []);
    assert_eq!(vec2, [3, 2, 1]);
}

#[test]
fn test_drain_items_zero_sized() {
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![(), (), ()]);
    let mut vec2: SmallVec<_, 5> = SmallVec::from(vec![]);
    for i in vec.drain(..) {
        vec2.push(i);
    }
    assert_eq!(vec, []);
    assert_eq!(vec2, [(), (), ()]);
}

#[test]
#[should_panic]
fn test_drain_out_of_bounds() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    v.drain(5..6);
}

#[test]
fn test_drain_range() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    for _ in v.drain(4..) {}
    assert_eq!(v, &[1, 2, 3, 4]);

    let mut v: SmallVec<_, 5> = (1..6).map(|x| x.to_string()).collect();
    for _ in v.drain(1..4) {}
    assert_eq!(v, &[1.to_string(), 5.to_string()]);

    let mut v: SmallVec<_, 5> = (1..6).map(|x| x.to_string()).collect();
    for _ in v.drain(1..4).rev() {}
    assert_eq!(v, &[1.to_string(), 5.to_string()]);

    let mut v: SmallVec<(), 5> = SmallVec::from(vec![(); 5]);
    for _ in v.drain(1..4).rev() {}
    assert_eq!(v, &[(), ()]);
}

#[test]
fn test_drain_inclusive_range() {
    let mut v: SmallVec<char, 5> = SmallVec::from(vec!['a', 'b', 'c', 'd', 'e']);
    for _ in v.drain(1..=3) {}
    assert_eq!(v, &['a', 'e']);

    let mut v: SmallVec<_, 5> = (0..=5).map(|x| x.to_string()).collect();
    for _ in v.drain(1..=5) {}
    assert_eq!(v, &["0".to_string()]);

    let mut v: Vec<String> = (0..=5).map(|x| x.to_string()).collect();
    for _ in v.drain(0..=5) {}
    assert_eq!(v, Vec::<String>::new());

    let mut v: SmallVec<_, 5> = (0..=5).map(|x| x.to_string()).collect();
    for _ in v.drain(0..=3) {}
    assert_eq!(v, &["4".to_string(), "5".to_string()]);

    let mut v: SmallVec<_, 5> = (0..=1).map(|x| x.to_string()).collect();
    for _ in v.drain(..=0) {}
    assert_eq!(v, &["1".to_string()]);
}

#[test]
#[should_panic]
fn test_drain_index_overflow() {
    let mut v = SmallVec::<(), 5>::with_capacity(usize::MAX);
    unsafe {
        v.set_len(usize::MAX);
    }
    v.drain(0..=usize::MAX);
}

#[test]
#[should_panic]
fn test_drain_inclusive_out_of_bounds() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    v.drain(5..=5);
}

#[test]
#[should_panic]
fn test_drain_start_overflow() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    v.drain((Excluded(usize::MAX), Included(0)));
}

#[test]
#[should_panic]
fn test_drain_end_overflow() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    v.drain((Included(0), Included(usize::MAX)));
}

#[test]
fn test_drain_leak() {
    static mut DROPS: i32 = 0;

    #[derive(Debug, PartialEq)]
    struct D(u32, bool);

    impl Drop for D {
        fn drop(&mut self) {
            unsafe {
                DROPS += 1;
            }

            if self.1 {
                panic!("panic in `drop`");
            }
        }
    }

    let mut v: SmallVec<_, 10> = SmallVec::from(vec![
        D(0, false),
        D(1, false),
        D(2, false),
        D(3, false),
        D(4, true),
        D(5, false),
        D(6, false),
    ]);

    catch_unwind(AssertUnwindSafe(|| {
        v.drain(2..=5);
    }))
    .ok();

    assert_eq!(unsafe { DROPS }, 4);
    assert_eq!(v, vec![D(0, false), D(1, false), D(6, false),]);
}

#[test]
fn test_splice() {
    let mut v: SmallVec<_, 3> = SmallVec::from([1, 2, 3]);
    let new = [7, 8];
    let u: SmallVec<_, 2> = v.splice(..2, new.iter().cloned()).collect();
    assert_eq!(v, [7, 8, 3]);
    assert_eq!(u, [1, 2]);

    let mut v: SmallVec<_, 3> = SmallVec::from([1, 2, 3]);
    let new = [7, 8, 9, 10];
    let replace_with = new.iter().cloned();
    let range = ..3;
    let u: SmallVec<_, 2> = v.splice(range, replace_with).collect();
    assert_eq!(u, [1, 2, 3]);
    assert_eq!(v, [7, 8, 9, 10]);

    let mut v: SmallVec<_, 15> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    assert_eq!(v.spilled(), false);
    let a = [10, 11, 12, 13];
    v.splice(2..4, a.iter().cloned());
    assert_eq!(v, &[1, 2, 10, 11, 12, 13, 5]);
    v.splice(1..3, Some(20));
    assert_eq!(v, &[1, 20, 11, 12, 13, 5]);

    let mut v: SmallVec<_, 3> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    assert_eq!(v.spilled(), true);
    let a = [10, 11, 12, 13];
    v.splice(2..4, a.iter().cloned());
    assert_eq!(v, &[1, 2, 10, 11, 12, 13, 5]);
    v.splice(1..3, Some(20));
    assert_eq!(v, &[1, 20, 11, 12, 13, 5]);

    let mut v: SmallVec<_, 4> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    assert_eq!(v.spilled(), true);
    let a = [10, 11, 12, 13];
    v.splice(2..4, a.iter().cloned());
    assert_eq!(v, &[1, 2, 10, 11, 12, 13, 5]);
    v.splice(1..3, Some(20));
    assert_eq!(v, &[1, 20, 11, 12, 13, 5]);
}

#[test]
fn test_splice_inline_to_heap() {
    let mut v: SmallVec<_, 3> = SmallVec::from([1, 2, 3]);
    let new = [7, 8, 9, 10];
    let replace_with = new.iter().cloned();
    let range = ..2;
    let u: SmallVec<_, 2> = v.splice(range, replace_with).collect();
    assert_eq!(u, [1, 2]);
    assert_eq!(v, [7, 8, 9, 10, 3]);

    let mut v: SmallVec<_, 3> = SmallVec::from([1, 2, 3]);
    let new = [7, 8];
    let u: SmallVec<_, 2> = v.splice(2..2, new.iter().cloned()).collect();
    assert_eq!(u, []);
    assert_eq!(v, [1, 2, 7, 8, 3]);

    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12, 13];
    let t1: Vec<_> = v.splice(2..4, a.iter().cloned()).collect();
    assert_eq!(v, &[1, 2, 10, 11, 12, 13, 5]);
    assert_eq!(t1, &[3, 4]);
    v.splice(1..3, Some(20));
    assert_eq!(v, &[1, 20, 11, 12, 13, 5]);

    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    let t1: Vec<_> = v.splice(2..=3, a.iter().cloned()).collect();
    assert_eq!(v, &[1, 2, 10, 11, 12, 5]);
    assert_eq!(t1, &[3, 4]);
    let t2: Vec<_> = v.splice(1..=2, Some(20)).collect();
    assert_eq!(v, &[1, 20, 11, 12, 5]);
    assert_eq!(t2, &[2, 10]);
}

#[test]
fn test_splice_inclusive_range() {
    let mut v: SmallVec<_, 10> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    let t1: Vec<_> = v.splice(2..=3, a.iter().cloned()).collect();
    assert_eq!(v, &[1, 2, 10, 11, 12, 5]);
    assert_eq!(t1, &[3, 4]);
    let t2: Vec<_> = v.splice(1..=2, Some(20)).collect();
    assert_eq!(v, &[1, 20, 11, 12, 5]);
    assert_eq!(t2, &[2, 10]);
}

#[test]
#[should_panic]
fn test_splice_out_of_bounds() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    v.splice(5..6, a.iter().cloned());
}

#[test]
#[should_panic]
fn test_splice_inclusive_out_of_bounds() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    v.splice(5..=5, a.iter().cloned());
}

#[test]
fn test_splice_items_zero_sized() {
    let mut vec: SmallVec<(), 5> = SmallVec::from(vec![(), (), ()]);
    let vec2: SmallVec<(), 5> = SmallVec::from(vec![]);
    let t: SmallVec<(), 5> = vec.splice(1..2, vec2.iter().cloned()).collect();
    assert_eq!(vec, &[(), ()]);
    assert_eq!(t, &[()]);
}

#[test]
fn test_splice_unbounded() {
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let t: SmallVec<_, 5> = vec.splice(.., None).collect();
    assert_eq!(vec, &[]);
    assert_eq!(t, &[1, 2, 3, 4, 5]);
}

#[test]
fn test_splice_forget() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    std::mem::forget(v.splice(2..4, a.iter().cloned()));
    assert_eq!(v, &[1, 2]);
}

#[test]
fn test_into_boxed_slice() {
    let xs: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    let ys = xs.into_boxed_slice();
    assert_eq!(&*ys, [1, 2, 3]);
}

#[test]

fn test_append() {
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3]);
    let mut vec2: SmallVec<_, 5> = SmallVec::from(vec![4, 5, 6]);
    vec.append(&mut vec2);
    assert_eq!(vec, [1, 2, 3, 4, 5, 6]);
    assert_eq!(vec2, []);
}

#[test]
fn test_split_off() {
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5, 6]);
    let orig_capacity = vec.capacity();
    let vec2 = vec.split_off(4);
    assert_eq!(vec, [1, 2, 3, 4]);
    assert_eq!(vec2, [5, 6]);
    assert_eq!(vec.capacity(), orig_capacity);
}

#[test]
fn test_split_off_take_all() {
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![1, 2, 3, 4, 5, 6]);
    let orig_ptr = vec.as_ptr();
    let orig_capacity = vec.capacity();
    let vec2 = vec.split_off(0);
    assert_eq!(vec, []);
    assert_eq!(vec2, [1, 2, 3, 4, 5, 6]);
    assert_eq!(vec.capacity(), orig_capacity);
    assert_eq!(vec2.as_ptr(), orig_ptr);
}

#[test]
fn test_into_iter_as_slice() {
    let vec: SmallVec<_, 5> = SmallVec::from(vec!['a', 'b', 'c']);
    let mut into_iter = vec.into_iter();
    assert_eq!(into_iter.as_slice(), &['a', 'b', 'c']);
    let _ = into_iter.next().unwrap();
    assert_eq!(into_iter.as_slice(), &['b', 'c']);
    let _ = into_iter.next().unwrap();
    let _ = into_iter.next().unwrap();
    assert_eq!(into_iter.as_slice(), &[]);
}

#[test]
fn test_into_iter_as_mut_slice() {
    let vec: SmallVec<_, 5> = SmallVec::from(vec!['a', 'b', 'c']);
    let mut into_iter = vec.into_iter();
    assert_eq!(into_iter.as_slice(), &['a', 'b', 'c']);
    into_iter.as_mut_slice()[0] = 'x';
    into_iter.as_mut_slice()[1] = 'y';
    assert_eq!(into_iter.next().unwrap(), 'x');
    assert_eq!(into_iter.as_slice(), &['y', 'c']);
}

#[test]
fn test_into_iter_debug() {
    let vec: SmallVec<_, 5> = SmallVec::from(vec!['a', 'b', 'c']);
    let into_iter = vec.into_iter();
    let debug = format!("{:?}", into_iter);
    assert_eq!(debug, "IntoIter(['a', 'b', 'c'])");
}

#[test]
fn test_into_iter_count() {
    assert_eq!(SmallVec::<_, 5>::from(vec![1, 2, 3]).into_iter().count(), 3);
}

#[test]
fn test_into_iter_leak() {
    static mut DROPS: i32 = 0;

    struct D(bool);

    impl Drop for D {
        fn drop(&mut self) {
            unsafe {
                DROPS += 1;
            }

            if self.0 {
                panic!("panic in `drop`");
            }
        }
    }

    let v: SmallVec<_, 5> = SmallVec::from(vec![D(false), D(true), D(false)]);

    catch_unwind(move || drop(v.into_iter())).ok();

    assert_eq!(unsafe { DROPS }, 3);
}

#[test]
fn test_from_iter_specialization_panic_during_iteration_drops() {
    let drop_count: SmallVec<_, 3> = (0..=2).map(|_| Rc::new(())).collect();
    let src: SmallVec<_, 3> = drop_count.iter().cloned().collect();
    let iter = src.into_iter();

    let _ = std::panic::catch_unwind(AssertUnwindSafe(|| {
        let _ = iter
            .enumerate()
            .filter_map(|(i, e)| {
                if i == 1 {
                    std::panic!("aborting iteration");
                }
                Some(e)
            })
            .collect::<SmallVec<_, 3>>();
    }));

    assert!(
        drop_count
            .iter()
            .map(Rc::strong_count)
            .all(|count| count == 1),
        "all items were dropped once"
    );
}

#[test]
fn test_extend_ref() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![1, 2]);
    v.extend(&[3, 4, 5]);

    assert_eq!(v.len(), 5);
    assert_eq!(v, [1, 2, 3, 4, 5]);

    let w: SmallVec<_, 5> = SmallVec::from(vec![6, 7]);
    v.extend(&w);

    assert_eq!(v.len(), 7);
    assert_eq!(v, [1, 2, 3, 4, 5, 6, 7]);
}

#[test]
fn overaligned_allocations() {
    #[repr(align(256))]
    struct Foo(usize);
    let mut v: SmallVec<_, 3> = SmallVec::from(vec![Foo(273)]);
    for i in 0..0x1000 {
        v.reserve_exact(i);
        assert!(v[0].0 == 273);
        assert!(v.as_ptr() as usize & 0xff == 0);
        v.shrink_to_fit();
        assert!(v[0].0 == 273);
        assert!(v.as_ptr() as usize & 0xff == 0);
    }
}

#[test]
fn drain_filter_empty() {
    let mut vec: SmallVec<i32, 3> = SmallVec::from(vec![]);

    {
        let mut iter = vec.drain_filter(|_| true);
        assert_eq!(iter.size_hint(), (0, Some(0)));
        assert_eq!(iter.next(), None);
        assert_eq!(iter.size_hint(), (0, Some(0)));
        assert_eq!(iter.next(), None);
        assert_eq!(iter.size_hint(), (0, Some(0)));
    }
    assert_eq!(vec.len(), 0);
    assert_eq!(vec, vec![]);
}

#[test]
fn drain_filter_zst() {
    let mut vec: SmallVec<_, 5> = SmallVec::from(vec![(), (), (), (), ()]);
    let initial_len = vec.len();
    let mut count = 0;
    {
        let mut iter = vec.drain_filter(|_| true);
        assert_eq!(iter.size_hint(), (0, Some(initial_len)));
        while let Some(_) = iter.next() {
            count += 1;
            assert_eq!(iter.size_hint(), (0, Some(initial_len - count)));
        }
        assert_eq!(iter.size_hint(), (0, Some(0)));
        assert_eq!(iter.next(), None);
        assert_eq!(iter.size_hint(), (0, Some(0)));
    }

    assert_eq!(count, initial_len);
    assert_eq!(vec.len(), 0);
    assert_eq!(vec, vec![]);
}

#[test]
fn drain_filter_false() {
    let mut vec: SmallVec<_, 10> = SmallVec::from(vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

    let initial_len = vec.len();
    let mut count = 0;
    {
        let mut iter = vec.drain_filter(|_| false);
        assert_eq!(iter.size_hint(), (0, Some(initial_len)));
        for _ in iter.by_ref() {
            count += 1;
        }
        assert_eq!(iter.size_hint(), (0, Some(0)));
        assert_eq!(iter.next(), None);
        assert_eq!(iter.size_hint(), (0, Some(0)));
    }

    assert_eq!(count, 0);
    assert_eq!(vec.len(), initial_len);
    assert_eq!(vec, vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);
}

#[test]
fn drain_filter_true() {
    let mut vec: SmallVec<_, 10> = SmallVec::from(vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

    let initial_len = vec.len();
    let mut count = 0;
    {
        let mut iter = vec.drain_filter(|_| true);
        assert_eq!(iter.size_hint(), (0, Some(initial_len)));
        while let Some(_) = iter.next() {
            count += 1;
            assert_eq!(iter.size_hint(), (0, Some(initial_len - count)));
        }
        assert_eq!(iter.size_hint(), (0, Some(0)));
        assert_eq!(iter.next(), None);
        assert_eq!(iter.size_hint(), (0, Some(0)));
    }

    assert_eq!(count, initial_len);
    assert_eq!(vec.len(), 0);
    assert_eq!(vec, vec![]);
}

#[test]
fn drain_filter_complex() {
    {
        //                [+xxx++++++xxxxx++++x+x++]
        let mut vec: SmallVec<_, 20> = SmallVec::from(vec![
            1, 2, 4, 6, 7, 9, 11, 13, 15, 17, 18, 20, 22, 24, 26, 27, 29, 31, 33, 34, 35, 36, 37,
            39,
        ]);

        let removed = vec
            .drain_filter(|x| *x % 2 == 0)
            .collect::<SmallVec<_, 20>>();
        assert_eq!(removed.len(), 10);
        assert_eq!(removed, vec![2, 4, 6, 18, 20, 22, 24, 26, 34, 36]);

        assert_eq!(vec.len(), 14);
        assert_eq!(
            vec,
            vec![1, 7, 9, 11, 13, 15, 17, 27, 29, 31, 33, 35, 37, 39]
        );
    }

    {
        //                [xxx++++++xxxxx++++x+x++]
        let mut vec: SmallVec<_, 20> = SmallVec::from(vec![
            2, 4, 6, 7, 9, 11, 13, 15, 17, 18, 20, 22, 24, 26, 27, 29, 31, 33, 34, 35, 36, 37, 39,
        ]);

        let removed = vec
            .drain_filter(|x| *x % 2 == 0)
            .collect::<SmallVec<_, 20>>();
        assert_eq!(removed.len(), 10);
        assert_eq!(removed, vec![2, 4, 6, 18, 20, 22, 24, 26, 34, 36]);

        assert_eq!(vec.len(), 13);
        assert_eq!(vec, vec![7, 9, 11, 13, 15, 17, 27, 29, 31, 33, 35, 37, 39]);
    }

    {
        //                [xxx++++++xxxxx++++x+x]
        let mut vec: SmallVec<_, 20> = SmallVec::from(vec![
            2, 4, 6, 7, 9, 11, 13, 15, 17, 18, 20, 22, 24, 26, 27, 29, 31, 33, 34, 35, 36,
        ]);

        let removed = vec
            .drain_filter(|x| *x % 2 == 0)
            .collect::<SmallVec<_, 20>>();
        assert_eq!(removed.len(), 10);
        assert_eq!(removed, vec![2, 4, 6, 18, 20, 22, 24, 26, 34, 36]);

        assert_eq!(vec.len(), 11);
        assert_eq!(vec, vec![7, 9, 11, 13, 15, 17, 27, 29, 31, 33, 35]);
    }

    {
        //                [xxxxxxxxxx+++++++++++]
        let mut vec: SmallVec<_, 20> = SmallVec::from(vec![
            2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 1, 3, 5, 7, 9, 11, 13, 15, 17, 19,
        ]);

        let removed = vec
            .drain_filter(|x| *x % 2 == 0)
            .collect::<SmallVec<_, 20>>();
        assert_eq!(removed.len(), 10);
        assert_eq!(removed, vec![2, 4, 6, 8, 10, 12, 14, 16, 18, 20]);

        assert_eq!(vec.len(), 10);
        assert_eq!(vec, vec![1, 3, 5, 7, 9, 11, 13, 15, 17, 19]);
    }

    {
        //                [+++++++++++xxxxxxxxxx]
        let mut vec: SmallVec<_, 20> = SmallVec::from(vec![
            1, 3, 5, 7, 9, 11, 13, 15, 17, 19, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20,
        ]);

        let removed = vec
            .drain_filter(|x| *x % 2 == 0)
            .collect::<SmallVec<_, 20>>();
        assert_eq!(removed.len(), 10);
        assert_eq!(removed, vec![2, 4, 6, 8, 10, 12, 14, 16, 18, 20]);

        assert_eq!(vec.len(), 10);
        assert_eq!(vec, vec![1, 3, 5, 7, 9, 11, 13, 15, 17, 19]);
    }
}

#[test]
fn drain_filter_unconsumed() {
    let mut vec: SmallVec<_, 3> = SmallVec::from(vec![1, 2, 3, 4]);
    let drain = vec.drain_filter(|&mut x| x % 2 != 0);
    drop(drain);
    assert_eq!(vec, [2, 4]);
}

#[test]
fn test_reserve_exact() {
    // This is all the same as test_reserve

    let mut v: SmallVec<_, 0> = SmallVec::from(Vec::new());
    assert_eq!(v.capacity(), 0);

    v.reserve_exact(2);
    assert!(v.capacity() >= 2);

    for i in 0..16 {
        v.push(i);
    }

    assert!(v.capacity() >= 16);
    v.reserve_exact(16);
    assert!(v.capacity() >= 32);

    v.push(16);

    v.reserve_exact(16);
    assert!(v.capacity() >= 33)
}

macro_rules! generate_assert_eq_vec_and_prim {
    ($name:ident<$B:ident>($type:ty)) => {
        fn $name<A: PartialEq<$B> + Debug, $B: Debug, const C: usize>(a: SmallVec<A, C>, b: $type) {
            assert!(a == b);
            assert_eq!(a, b);
        }
    };
}

generate_assert_eq_vec_and_prim! { assert_eq_vec_and_slice  <B>(&[B])   }
generate_assert_eq_vec_and_prim! { assert_eq_vec_and_array_3<B>([B; 3]) }

#[test]
fn partialeq_vec_and_prim() {
    assert_eq_vec_and_slice(SmallVec::<_, 3>::from(vec![1, 2, 3]), &[1, 2, 3]);
    assert_eq_vec_and_array_3(SmallVec::<_, 3>::from(vec![1, 2, 3]), [1, 2, 3]);
}

macro_rules! assert_partial_eq_valid {
    ($a2:expr, $a3:expr; $b2:expr, $b3: expr) => {
        assert!($a2 == $b2);
        assert!($a2 != $b3);
        assert!($a3 != $b2);
        assert!($a3 == $b3);
        assert_eq!($a2, $b2);
        assert_ne!($a2, $b3);
        assert_ne!($a3, $b2);
        assert_eq!($a3, $b3);
    };
}

#[test]
fn partialeq_vec_full() {
    let vec2: SmallVec<_, 2> = SmallVec::from(vec![1, 2]);
    let vec3: SmallVec<_, 3> = SmallVec::from(vec![1, 2, 3]);
    let slice2: &[_] = &[1, 2];
    let slice3: &[_] = &[1, 2, 3];
    let slicemut2: &[_] = &mut [1, 2];
    let slicemut3: &[_] = &mut [1, 2, 3];
    let array2: [_; 2] = [1, 2];
    let array3: [_; 3] = [1, 2, 3];
    let arrayref2: &[_; 2] = &[1, 2];
    let arrayref3: &[_; 3] = &[1, 2, 3];

    assert_partial_eq_valid!(vec2,vec3; vec2,vec3);
    assert_partial_eq_valid!(vec2,vec3; slice2,slice3);
    assert_partial_eq_valid!(vec2,vec3; slicemut2,slicemut3);
    assert_partial_eq_valid!(slice2,slice3; vec2,vec3);
    assert_partial_eq_valid!(slicemut2,slicemut3; vec2,vec3);
    assert_partial_eq_valid!(vec2,vec3; array2,array3);
    assert_partial_eq_valid!(vec2,vec3; arrayref2,arrayref3);
    assert_partial_eq_valid!(vec2,vec3; arrayref2[..],arrayref3[..]);
}

#[test]
fn test_zero_sized_vec_push() {
    const N: usize = 8;

    for len in 0..N {
        let mut tester: SmallVec<_, 10> = SmallVec::from(Vec::with_capacity(len));
        assert_eq!(tester.len(), 0);
        assert!(tester.capacity() >= len);
        for _ in 0..len {
            tester.push(());
        }
        assert_eq!(tester.len(), len);
        assert_eq!(tester.iter().count(), len);
        tester.clear();
    }
}

#[test]
fn test_vec_macro_repeat() {
    assert_eq!(
        SmallVec::<_, 3>::from(vec![1; 3]),
        SmallVec::<_, 3>::from(vec![1, 1, 1])
    );
    assert_eq!(
        SmallVec::<_, 3>::from(vec![1; 2]),
        SmallVec::<_, 3>::from(vec![1, 1])
    );
    assert_eq!(
        SmallVec::<_, 3>::from(vec![1; 1]),
        SmallVec::<_, 3>::from(vec![1])
    );
    assert_eq!(
        SmallVec::<_, 3>::from(vec![1; 0]),
        SmallVec::<_, 3>::from(vec![])
    );

    let el = Box::new(1);
    let n = 3;
    assert_eq!(
        SmallVec::<_, 3>::from(vec![el; n]),
        SmallVec::<_, 3>::from(vec![Box::new(1), Box::new(1), Box::new(1)])
    );
}

#[test]
fn test_vec_swap() {
    let mut a: SmallVec<_, 10> = SmallVec::from(vec![0, 1, 2, 3, 4, 5, 6]);
    a.swap(2, 4);
    assert_eq!(a[2], 4);
    assert_eq!(a[4], 2);
    let mut n = 42;
    swap(&mut n, &mut a[0]);
    assert_eq!(a[0], 42);
    assert_eq!(n, 0);
}

#[test]
fn test_extend_from_within_spec() {
    #[derive(Copy)]
    struct CopyOnly;

    impl Clone for CopyOnly {
        fn clone(&self) -> Self {
            panic!("extend_from_within must use specialization on copy");
        }
    }

    SmallVec::<_, 2>::from(vec![CopyOnly, CopyOnly]).extend_from_within(..);
}

#[test]
fn test_extend_from_within_clone() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![
        String::from("sssss"),
        String::from("12334567890"),
        String::from("c"),
    ]);
    v.extend_from_within(1..);

    assert_eq!(v, ["sssss", "12334567890", "c", "12334567890", "c"]);
}

#[test]

fn test_extend_from_within_complete_rande() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![0, 1, 2, 3]);
    v.extend_from_within(..);

    assert_eq!(v, [0, 1, 2, 3, 0, 1, 2, 3]);
}

#[test]
fn test_extend_from_within_empty_rande() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![0, 1, 2, 3]);
    v.extend_from_within(1..1);

    assert_eq!(v, [0, 1, 2, 3]);
}

#[test]
#[should_panic]
fn test_extend_from_within_out_of_rande() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![0, 1]);
    v.extend_from_within(..3);
}

#[test]
fn test_extend_from_within_zst() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![(); 8]);
    v.extend_from_within(3..7);

    assert_eq!(v, [(); 12]);
}

#[test]
fn test_extend_from_within_empty_vec() {
    let mut v: SmallVec<i32, 0> = SmallVec::from(Vec::new());
    v.extend_from_within(..);

    assert_eq!(v, []);
}

#[test]
fn test_extend_from_within() {
    let mut v: SmallVec<_, 5> = SmallVec::from(vec![
        String::from("a"),
        String::from("b"),
        String::from("c"),
    ]);
    v.extend_from_within(1..=2);
    v.extend_from_within(..=1);

    assert_eq!(v, ["a", "b", "c", "b", "c", "a", "b"]);
}

#[test]
fn test_vec_dedup_by() {
    let mut vec: SmallVec<i32, 10> = SmallVec::from(vec![1, -1, 2, 3, 1, -5, 5, -2, 2]);

    vec.dedup_by(|a, b| a.abs() == b.abs());

    assert_eq!(vec, [1, 2, 3, 1, -5, -2]);
}

#[test]
fn test_vec_dedup_empty() {
    let mut vec: SmallVec<i32, 0> = SmallVec::from(Vec::new());

    vec.dedup();

    assert_eq!(vec, []);
}

#[test]
fn test_vec_dedup_one() {
    let mut vec: SmallVec<i32, 10> = SmallVec::from(vec![12i32]);

    vec.dedup();

    assert_eq!(vec, [12]);
}

#[test]
fn test_vec_dedup_multiple_ident() {
    let mut vec: SmallVec<i32, 10> =
        SmallVec::from(vec![12, 12, 12, 12, 12, 11, 11, 11, 11, 11, 11]);

    vec.dedup();

    assert_eq!(vec, [12, 11]);
}

#[test]
fn test_vec_dedup_partialeq() {
    #[derive(Debug)]
    struct Foo(i32, i32);

    impl PartialEq for Foo {
        fn eq(&self, other: &Foo) -> bool {
            self.0 == other.0
        }
    }

    let mut vec: SmallVec<_, 10> = SmallVec::from(vec![Foo(0, 1), Foo(0, 5), Foo(1, 7), Foo(1, 9)]);

    vec.dedup();
    assert_eq!(vec, [Foo(0, 1), Foo(1, 7)]);
}

#[test]
fn test_vec_dedup() {
    let mut vec: SmallVec<bool, 10> = SmallVec::with_capacity(8);
    let mut template = vec.clone();

    for x in 0u8..255u8 {
        vec.clear();
        template.clear();

        let iter = (0..8).map(move |bit| (x >> bit) & 1 == 1);
        vec.extend(iter);
        template.extend_from_slice(&vec);

        let (dedup, _) = template.partition_dedup();
        vec.dedup();

        assert_eq!(vec, dedup);
    }
}

#[test]
fn test_zero_size_items() {
    let mut v = SmallVec::<(), 0>::new();
    v.push(());
    assert!(!v.spilled());

    let mut v: SmallVec<_, 3> = SmallVec::from(vec![(), (), (), ()]);
    assert!(!v.spilled());
    v.pop();
    v.pop();
    assert!(!v.spilled());
    v.grow_to(4);
    assert!(!v.spilled());
    assert_eq!(v.len(), 2);

    let mut v: SmallVec<_, 3> = SmallVec::from(vec![(), (), (), ()]);
    assert!(!v.spilled());
    v.pop();
    v.pop();
    assert!(!v.spilled());
    v.shrink_to(3);
    assert!(!v.spilled());
    assert_eq!(v.len(), 2);
}

#[test]
pub fn test_zero() {
    let mut v = SmallVec::<usize, 0>::new();
    assert!(!v.spilled());
    v.push(0usize);
    assert!(v.spilled());
    assert_eq!(v, [0]);
}

#[test]
pub fn test_inline() {
    let mut v = SmallVec::<String, 16>::new();
    v.push("hello".to_owned());
    v.push("there".to_owned());
    assert_eq!(v, ["hello".to_owned(), "there".to_owned(),]);
}

#[test]

pub fn test_spill() {
    let mut v = SmallVec::<String, 2>::new();
    v.push("hello".to_owned());
    assert_eq!(v[0], "hello");
    v.push("there".to_owned());
    v.push("burma".to_owned());
    assert_eq!(v[0], "hello");
    v.push("shave".to_owned());
    assert_eq!(
        v,
        [
            "hello".to_owned(),
            "there".to_owned(),
            "burma".to_owned(),
            "shave".to_owned(),
        ]
    );
}

#[test]

pub fn test_double_spill() {
    let mut v = SmallVec::<String, 2>::new();
    v.push("hello".to_owned());
    v.push("there".to_owned());
    v.push("burma".to_owned());
    v.push("shave".to_owned());
    v.push("hello".to_owned());
    v.push("there".to_owned());
    v.push("burma".to_owned());
    v.push("shave".to_owned());
    assert_eq!(
        v,
        [
            "hello".to_owned(),
            "there".to_owned(),
            "burma".to_owned(),
            "shave".to_owned(),
            "hello".to_owned(),
            "there".to_owned(),
            "burma".to_owned(),
            "shave".to_owned(),
        ]
    );
}

#[test]
fn test_with_capacity() {
    let v: SmallVec<(), 3> = SmallVec::with_capacity(1);
    assert!(v.is_empty());
    assert!(!v.spilled());

    let v: SmallVec<u8, 3> = SmallVec::with_capacity(1);
    assert!(v.is_empty());
    assert!(!v.spilled());
    assert_eq!(v.capacity(), 3);

    let v: SmallVec<u8, 3> = SmallVec::with_capacity(10);
    assert!(v.is_empty());
    assert!(v.spilled());
    assert_eq!(v.capacity(), 10);
}

#[test]
fn test_drain() {
    let mut v: SmallVec<u8, 2> = SmallVec::new();
    v.push(3);
    assert_eq!(v.drain(..).collect::<Vec<_>>(), [3]);

    // spilling the vec
    v.push(3);
    v.push(4);
    v.push(5);
    let old_capacity = v.capacity();
    assert_eq!(v.drain(1..).collect::<Vec<_>>(), [4, 5]);
    // drain should not change the capacity
    assert_eq!(v.capacity(), old_capacity);
}

#[test]
fn test_drain_rev() {
    let mut v: SmallVec<u8, 2> = SmallVec::new();
    v.push(3);
    assert_eq!(v.drain(..).rev().collect::<Vec<_>>(), [3]);

    // spilling the vec
    v.push(3);
    v.push(4);
    v.push(5);
    assert_eq!(v.drain(..).rev().collect::<Vec<_>>(), [5, 4, 3]);
}

#[test]
fn test_drain_forget() {
    let mut v: SmallVec<u8, 1> = SmallVec::from(vec![0, 1, 2, 3, 4, 5, 6, 7]);
    std::mem::forget(v.drain(2..5));
    assert_eq!(v.len(), 2);
}

#[test]
fn test_into_iter() {
    let mut v: SmallVec<u8, 2> = SmallVec::new();
    v.push(3);
    assert_eq!(v.into_iter().collect::<Vec<_>>(), [3]);

    // spilling the vec
    let mut v: SmallVec<u8, 2> = SmallVec::new();
    v.push(3);
    v.push(4);
    v.push(5);
    assert_eq!(v.into_iter().collect::<Vec<_>>(), [3, 4, 5]);
}

#[test]
fn test_into_iter_rev() {
    let mut v: SmallVec<u8, 2> = SmallVec::new();
    v.push(3);
    assert_eq!(v.into_iter().rev().collect::<Vec<_>>(), [3]);

    // spilling the vec
    let mut v: SmallVec<u8, 2> = SmallVec::new();
    v.push(3);
    v.push(4);
    v.push(5);
    assert_eq!(v.into_iter().rev().collect::<Vec<_>>(), [5, 4, 3]);
}

#[test]
fn test_into_iter_drop() {
    use std::cell::Cell;

    struct DropCounter<'a>(&'a Cell<i32>);

    impl<'a> Drop for DropCounter<'a> {
        fn drop(&mut self) {
            self.0.set(self.0.get() + 1);
        }
    }

    {
        let cell = Cell::new(0);
        let mut v: SmallVec<DropCounter<'_>, 2> = SmallVec::new();
        v.push(DropCounter(&cell));
        v.into_iter();
        assert_eq!(cell.get(), 1);
    }

    {
        let cell = Cell::new(0);
        let mut v: SmallVec<DropCounter<'_>, 2> = SmallVec::new();
        v.push(DropCounter(&cell));
        v.push(DropCounter(&cell));
        assert!(v.into_iter().next().is_some());
        assert_eq!(cell.get(), 2);
    }

    {
        let cell = Cell::new(0);
        let mut v: SmallVec<DropCounter<'_>, 2> = SmallVec::new();
        v.push(DropCounter(&cell));
        v.push(DropCounter(&cell));
        v.push(DropCounter(&cell));
        assert!(v.into_iter().next().is_some());
        assert_eq!(cell.get(), 3);
    }
    {
        let cell = Cell::new(0);
        let mut v: SmallVec<DropCounter<'_>, 2> = SmallVec::new();
        v.push(DropCounter(&cell));
        v.push(DropCounter(&cell));
        v.push(DropCounter(&cell));
        {
            let mut it = v.into_iter();
            assert!(it.next().is_some());
            assert!(it.next_back().is_some());
        }
        assert_eq!(cell.get(), 3);
    }
}

#[test]
fn test_capacity() {
    let vec = vec![1, 2, 3, 4, 5];
    let v: SmallVec<u32, 3> = SmallVec::from_vec(vec);
    assert_eq!(v.capacity(), 5);

    let vec = [1, 2, 3, 4, 5];
    let v: SmallVec<u32, 5> = SmallVec::from_array(vec);
    assert_eq!(v.capacity(), 5);

    let mut v: SmallVec<u8, 2> = SmallVec::new();
    v.reserve(1);
    assert_eq!(v.capacity(), 2);
    assert!(!v.spilled());

    v.reserve_exact(10);
    assert!(v.capacity() >= 10);
    assert!(v.spilled());
    v.push(0);
    v.push(1);
    v.push(2);
    v.push(3);

    v.shrink_to(4);
    assert!(v.spilled());
    assert!(v.capacity() < 10);
}

#[test]
fn test_truncate() {
    let mut v: SmallVec<Box<u8>, 8> = SmallVec::new();

    for x in 0..8 {
        v.push(Box::new(x));
    }
    v.truncate(4);

    assert_eq!(v.len(), 4);
    assert!(!v.spilled());

    assert_eq!(Box::into_inner(v.swap_remove(1)), 1);
    assert_eq!(Box::into_inner(v.remove(1)), 3);
    v.insert(1, Box::new(3));

    assert_eq!(v.iter().map(|v| **v).collect::<Vec<_>>(), [0, 3, 2]);
}

#[test]
#[should_panic]
fn test_drain_overflow() {
    let mut v: SmallVec<u8, 8> = SmallVec::from(vec![0]);
    v.drain(..=usize::MAX);
}

#[test]
fn test_extend_from_slice2() {
    let mut v: SmallVec<u8, 8> = SmallVec::new();
    for x in 0..4 {
        v.push(x);
    }
    assert_eq!(v.len(), 4);
    v.extend_from_slice(&[5, 6]);
    assert_eq!(v.iter().map(|v| *v).collect::<Vec<_>>(), [0, 1, 2, 3, 5, 6]);
}

#[test]
#[should_panic]
fn test_drop_panic_smallvec() {
    // This test should only panic once, and not double panic,
    // which would mean a double drop
    struct DropPanic;

    impl Drop for DropPanic {
        fn drop(&mut self) {
            panic!("drop");
        }
    }

    let mut v = SmallVec::<_, 1>::new();
    v.push(DropPanic);
}

#[test]
fn test_eq() {
    let mut a: SmallVec<u32, 2> = SmallVec::new();
    let mut b: SmallVec<u32, 2> = SmallVec::new();
    let mut c: SmallVec<u32, 2> = SmallVec::new();
    // a = [1, 2]
    a.push(1);
    a.push(2);
    // b = [1, 2]
    b.push(1);
    b.push(2);
    // c = [3, 4]
    c.push(3);
    c.push(4);

    assert!(a == b);
    assert!(a != c);
}

#[test]
fn test_ord() {
    let mut a: SmallVec<u32, 2> = SmallVec::new();
    let mut b: SmallVec<u32, 2> = SmallVec::new();
    let mut c: SmallVec<u32, 2> = SmallVec::new();
    // a = [1]
    a.push(1);
    // b = [1, 1]
    b.push(1);
    b.push(1);
    // c = [1, 2]
    c.push(1);
    c.push(2);

    assert!(a < b);
    assert!(b > a);
    assert!(b < c);
    assert!(c > b);
}

#[test]
fn test_hash() {
    use std::{collections::hash_map::DefaultHasher, hash::Hash};

    {
        let mut a: SmallVec<u32, 2> = SmallVec::new();
        let b = [1, 2];
        a.extend(b.iter().cloned());
        let mut hasher = DefaultHasher::new();
        assert_eq!(a.hash(&mut hasher), b.hash(&mut hasher));
    }
    {
        let mut a: SmallVec<u32, 2> = SmallVec::new();
        let b = [1, 2, 11, 12];
        a.extend(b.iter().cloned());
        let mut hasher = DefaultHasher::new();
        assert_eq!(a.hash(&mut hasher), b.hash(&mut hasher));
    }
}

#[test]

fn test_as_ref() {
    let mut a: SmallVec<u32, 2> = SmallVec::new();
    a.push(1);
    assert_eq!(a.as_ref(), [1]);
    a.push(2);
    assert_eq!(a.as_ref(), [1, 2]);
    a.push(3);
    assert_eq!(a.as_ref(), [1, 2, 3]);
}

#[test]

fn test_as_mut() {
    let mut a: SmallVec<u32, 2> = SmallVec::new();
    a.push(1);
    assert_eq!(a.as_mut(), [1]);
    a.push(2);
    assert_eq!(a.as_mut(), [1, 2]);
    a.push(3);
    assert_eq!(a.as_mut(), [1, 2, 3]);
    a.as_mut()[1] = 4;
    assert_eq!(a.as_mut(), [1, 4, 3]);
}

#[test]
fn test_borrow() {
    use std::borrow::Borrow;

    let mut a: SmallVec<u32, 2> = SmallVec::new();
    a.push(1);
    assert_eq!(a.borrow(), [1]);
    a.push(2);
    assert_eq!(a.borrow(), [1, 2]);
    a.push(3);
    assert_eq!(a.borrow(), [1, 2, 3]);
}

#[test]
fn test_borrow_mut() {
    use std::borrow::BorrowMut;

    let mut a: SmallVec<u32, 2> = SmallVec::new();
    a.push(1);
    assert_eq!(a.borrow_mut(), [1]);
    a.push(2);
    assert_eq!(a.borrow_mut(), [1, 2]);
    a.push(3);
    assert_eq!(a.borrow_mut(), [1, 2, 3]);
    BorrowMut::<[u32]>::borrow_mut(&mut a)[1] = 4;
    assert_eq!(a.borrow_mut(), [1, 4, 3]);
}

#[test]
fn test_from() {
    let a: SmallVec<u32, 2> = SmallVec::from(&[1][..]);
    let b: SmallVec<u32, 2> = SmallVec::from(&[1, 2, 3][..]);
    assert_eq!(a, [1]);
    assert_eq!(b, [1, 2, 3]);

    let vec = vec![];
    let small_vec: SmallVec<u8, 3> = SmallVec::from(vec);
    assert_eq!(&small_vec, &[]);

    let vec = vec![1, 2, 3, 4, 5];
    let small_vec: SmallVec<u8, 3> = SmallVec::from(vec);
    assert_eq!(small_vec, [1, 2, 3, 4, 5]);

    let vec = vec![1, 2, 3, 4, 5];
    let small_vec: SmallVec<u8, 1> = SmallVec::from(vec);
    assert_eq!(small_vec, [1, 2, 3, 4, 5]);

    let array = [1];
    let small_vec: SmallVec<u8, 1> = SmallVec::from(array);
    assert_eq!(small_vec, [1]);

    let array = [99; 128];
    let small_vec: SmallVec<u8, 128> = SmallVec::from(array);
    assert_eq!(&small_vec, vec![99u8; 128].as_slice());
}

#[test]
fn test_exact_size_iterator() {
    let mut vec = SmallVec::<u32, 2>::from(&[1, 2, 3][..]);
    assert_eq!(vec.clone().into_iter().len(), 3);
    assert_eq!(vec.drain(..2).len(), 2);
    assert_eq!(vec.into_iter().len(), 1);
}

#[test]
fn test_into_iter_as_slice2() {
    let vec = SmallVec::<u32, 2>::from(&[1, 2, 3][..]);
    let mut iter = vec.clone().into_iter();
    assert_eq!(iter.as_slice(), &[1, 2, 3]);
    assert_eq!(iter.as_mut_slice(), &[1, 2, 3]);
    iter.next();
    assert_eq!(iter.as_slice(), &[2, 3]);
    assert_eq!(iter.as_mut_slice(), &[2, 3]);
    iter.next_back();
    assert_eq!(iter.as_slice(), &[2]);
    assert_eq!(iter.as_mut_slice(), &[2]);
}

#[test]
fn test_into_iter_clone_empty_smallvec() {
    let mut iter = SmallVec::<u8, 2>::new().into_iter();
    let mut clone_iter = iter.clone();
    assert_eq!(iter.next(), None);
    assert_eq!(clone_iter.next(), None);
}

#[test]
fn test_from_vec() {
    let vec = vec![];
    let small_vec: SmallVec<u8, 3> = SmallVec::from_vec(vec);
    assert_eq!(&small_vec, &[]);

    let vec = vec![];
    let small_vec: SmallVec<u8, 1> = SmallVec::from_vec(vec);
    assert_eq!(&small_vec, &[]);

    let vec = vec![1];
    let small_vec: SmallVec<u8, 3> = SmallVec::from_vec(vec);
    assert_eq!(&small_vec, &[1]);

    let vec = vec![1, 2, 3];
    let small_vec: SmallVec<u8, 3> = SmallVec::from_vec(vec);
    assert_eq!(&small_vec, &[1, 2, 3]);

    let vec = vec![1, 2, 3, 4, 5];
    let small_vec: SmallVec<u8, 3> = SmallVec::from_vec(vec);
    assert_eq!(&small_vec, &[1, 2, 3, 4, 5]);

    let vec = vec![1, 2, 3, 4, 5];
    let small_vec: SmallVec<u8, 1> = SmallVec::from_vec(vec);
    assert_eq!(&small_vec, &[1, 2, 3, 4, 5]);
}

#[test]
fn test_retain2() {
    // Test inline data storate
    let mut sv: SmallVec<i32, 5> = SmallVec::from(&[1, 2, 3, 3, 4][..]);
    sv.retain(|&mut i| i != 3);
    assert_eq!(sv.pop(), Some(4));
    assert_eq!(sv.pop(), Some(2));
    assert_eq!(sv.pop(), Some(1));
    assert_eq!(sv.pop(), None);

    // Test spilled data storage
    let mut sv: SmallVec<i32, 3> = SmallVec::from(&[1, 2, 3, 3, 4][..]);
    sv.retain(|&mut i| i != 3);
    assert_eq!(sv.pop(), Some(4));
    assert_eq!(sv.pop(), Some(2));
    assert_eq!(sv.pop(), Some(1));
    assert_eq!(sv.pop(), None);

    // Test that drop implementations are called for inline.
    let one = Rc::new(1);
    let mut sv: SmallVec<Rc<i32>, 3> = SmallVec::new();
    sv.push(Rc::clone(&one));
    assert_eq!(Rc::strong_count(&one), 2);
    sv.retain(|_| false);
    assert_eq!(Rc::strong_count(&one), 1);

    // Test that drop implementations are called for spilled data.
    let mut sv: SmallVec<Rc<i32>, 1> = SmallVec::new();
    sv.push(Rc::clone(&one));
    sv.push(Rc::new(2));
    assert_eq!(Rc::strong_count(&one), 2);
    sv.retain(|_| false);
    assert_eq!(Rc::strong_count(&one), 1);
}

#[test]
fn test_dedup2() {
    let mut dupes: SmallVec<i32, 5> = SmallVec::from(&[1, 1, 2, 3, 3][..]);
    dupes.dedup();
    assert_eq!(&dupes, &[1, 2, 3]);

    let mut empty: SmallVec<i32, 5> = SmallVec::new();
    empty.dedup();
    assert!(empty.is_empty());

    let mut all_ones: SmallVec<i32, 5> = SmallVec::from(&[1, 1, 1, 1, 1][..]);
    all_ones.dedup();
    assert_eq!(all_ones.len(), 1);

    let mut no_dupes: SmallVec<i32, 5> = SmallVec::from(&[1, 2, 3, 4, 5][..]);
    no_dupes.dedup();
    assert_eq!(no_dupes.len(), 5);
}

#[test]
fn test_resize() {
    let mut v: SmallVec<i32, 8> = SmallVec::new();
    v.push(1);
    v.resize(5, 0);
    assert_eq!(v[..], [1, 0, 0, 0, 0]);

    v.resize(2, 0);
    assert_eq!(v[..], [1, 0]);
}

#[test]
fn test_write() {
    use std::io::Write;

    let data = [1, 2, 3, 4, 5];

    let mut small_vec: SmallVec<u8, 2> = SmallVec::new();
    let len = small_vec.write(&data).unwrap();
    assert_eq!(len, 5);
    assert_eq!(small_vec.as_ref(), data.as_ref());

    let mut small_vec: SmallVec<u8, 2> = SmallVec::new();
    small_vec.write_all(&data).unwrap();
    assert_eq!(small_vec.as_ref(), data.as_ref());
}

#[test]
fn test_grow_to_inline() {
    let mut v: SmallVec<u8, 5> = SmallVec::from(vec![1, 2]);
    assert!(!v.spilled());
    // // Shrink to inline.
    v.grow_to(3);
    assert!(!v.spilled());
    assert_eq!(v.capacity(), 5);
    assert_eq!(v.len(), 2);
    v.push(3);
    assert_eq!(v, [1, 2, 3]);

    let mut v: SmallVec<u8, 5> = SmallVec::from(vec![1, 2, 3]);
    assert!(!v.spilled());
    // // Shrink to inline.
    v.grow_to(3);
    assert!(!v.spilled());
    assert_eq!(v.capacity(), 5);
    assert_eq!(v.len(), 3);
    v.push(4);
    assert_eq!(v, [1, 2, 3, 4]);

    let mut v: SmallVec<u8, 2> = SmallVec::from(vec![1, 2]);
    assert!(!v.spilled());
    v.grow_to(3);
    assert!(v.spilled());
    assert_eq!(v.capacity(), 3);
    assert_eq!(v.len(), 2);
    v.push(3);
    assert_eq!(v, [1, 2, 3]);

    let mut v: SmallVec<u8, 5> = SmallVec::from(vec![1, 2]);
    assert!(!v.spilled());
    v.grow_to(8);
    assert!(v.spilled());
    assert_eq!(v.capacity(), 8);
    assert_eq!(v.len(), 2);
    v.push(3);
    assert_eq!(v, [1, 2, 3]);
}

#[test]
fn test_grow_to_heap() {
    let mut v: SmallVec<u8, 2> = SmallVec::from(vec![1, 2, 3]);
    assert!(v.spilled());
    v.grow_to(3);
    assert!(v.spilled());
    assert_eq!(v.capacity(), 3);
    assert_eq!(v.len(), 3);
    v.push(4);
    assert_eq!(v, [1, 2, 3, 4]);

    let mut v: SmallVec<u8, 2> = SmallVec::from(vec![1, 2, 3]);
    assert!(v.spilled());
    v.grow_to(4);
    assert!(v.spilled());
    assert_eq!(v.capacity(), 4);
    assert_eq!(v.len(), 3);
    v.push(4);
    assert_eq!(v, [1, 2, 3, 4]);

    let mut v: SmallVec<u8, 3> = SmallVec::from(vec![1, 2, 3, 4]);
    assert!(v.spilled());
    v.pop();
    assert!(v.spilled());
    assert_eq!(v.capacity(), 4);
    assert_eq!(v.len(), 3);
    v.grow_to(3);
    assert!(!v.spilled());
    assert_eq!(v.capacity(), 3);
    assert_eq!(v.len(), 3);
    assert_eq!(v, [1, 2, 3]);
    v.push(4);
    assert_eq!(v, [1, 2, 3, 4]);

    let mut v: SmallVec<u8, 3> = SmallVec::from(vec![1, 2, 3, 4]);
    assert!(v.spilled());
    v.pop();
    v.pop();
    v.grow_to(2);
    assert!(!v.spilled());
    assert_eq!(v.capacity(), 3);
    assert_eq!(v.len(), 2);
    assert_eq!(v, [1, 2]);
    v.push(3);
    assert_eq!(v, [1, 2, 3]);
}

#[test]
fn test_grow_spilled_same_size() {
    let mut v: SmallVec<u8, 2> = SmallVec::from(vec![1, 2, 3]);
    assert!(v.spilled());
    assert_eq!(v.capacity(), 3);
    // grow with the same capacity
    v.grow_to(3);
    assert!(v.spilled());
    assert_eq!(v.capacity(), 3);
    assert_eq!(v.len(), 3);
    v.push(4);
    assert_eq!(v, [1, 2, 3, 4]);

    let mut v: SmallVec<u8, 2> = SmallVec::new();
    assert_eq!(v.capacity(), 2);
    assert!(!v.spilled());
    v.push(0);
    assert_eq!(v.capacity(), 2);
    v.push(1);
    assert_eq!(v.capacity(), 2);
    assert_eq!(v.len(), 2);
    v.push(2);
    assert_eq!(v.capacity(), 4);
    assert_eq!(v.len(), 3);
    assert!(v.spilled());
    // grow with the same capacity
    v.grow_to(4);
    assert_eq!(v.capacity(), 4);
    assert_eq!(v, [0, 1, 2]);
}

#[test]
fn test_shrink() {
    let mut vec: SmallVec<_, 4> = SmallVec::with_capacity(10);
    assert_eq!(vec.capacity(), 10);
    vec.extend([1, 2, 3].iter().cloned());
    assert_eq!(vec.capacity(), 10);
    vec.shrink_to(4);
    assert_eq!(vec.capacity(), 4);
    vec.shrink_to(0);
    assert_eq!(vec.capacity(), 4);
    assert!(vec.capacity() >= 3);

    let mut vec: SmallVec<_, 3> = SmallVec::with_capacity(100);
    assert_eq!(vec.capacity(), 100);
    vec.extend([1, 2, 3].iter().cloned());
    assert_eq!(vec.capacity(), 100);
    vec.shrink_to(40);
    assert_eq!(vec.capacity(), 40);
    vec.shrink_to(0);
    assert_eq!(vec.capacity(), 3);
    assert!(vec.capacity() >= 3);

    let mut v: SmallVec<u8, 5> = SmallVec::from(vec![1, 2]);
    assert!(!v.spilled());
    v.shrink_to(5);
    assert!(!v.spilled());
    assert_eq!(v.capacity(), 5);
    assert_eq!(v.len(), 2);
    v.push(3);
    assert_eq!(v, [1, 2, 3]);

    let mut v: SmallVec<u8, 3> = SmallVec::from(vec![1, 2, 3, 4]);
    assert!(v.spilled());
    v.pop();
    v.pop();
    assert!(v.spilled());
    v.shrink_to(4);
    assert!(v.spilled());
    assert_eq!(v.capacity(), 4);
    assert_eq!(v.len(), 2);
    assert_eq!(v, [1, 2]);
    v.push(3);
    assert_eq!(v, [1, 2, 3]);
}

#[test]
fn test_default() {
    let _v = SmallVec::<i32, 987>::default();
}

#[test]
fn test_resumable_extend() {
    let s = "a b c";
    // This iterator yields: (Some('a'), None, Some('b'), None, Some('c')), None
    let it = s
        .chars()
        .scan(0, |_, ch| if ch.is_whitespace() { None } else { Some(ch) });
    let mut v: SmallVec<char, 4> = SmallVec::new();
    v.extend(it);
    assert_eq!(v, ['a']);
}

#[test]
fn test_into_vec() {
    let vec: SmallVec<u8, 2> = SmallVec::from(vec![0, 1]);
    assert_eq!(vec.into_vec(), vec![0, 1]);

    let vec: SmallVec<u8, 2> = SmallVec::from(vec![0, 1, 2]);
    assert_eq!(vec.into_vec(), vec![0, 1, 2]);
}

#[test]
fn test_into_inner() {
    let vec: SmallVec<u8, 2> = SmallVec::from(vec![0, 1]);
    assert_eq!(vec.into_inner(), Ok([0, 1]));

    let vec: SmallVec<u8, 2> = SmallVec::from(vec![0]);
    assert_eq!(vec.clone().into_inner(), Err(vec));

    let vec: SmallVec<u8, 2> = SmallVec::from(vec![0, 1, 2]);
    assert_eq!(vec.clone().into_inner(), Err(vec));
}

#[test]
fn test_splice2() {
    let mut v: SmallVec<_, 1> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    v.splice(2..4, a.iter().cloned());
    assert_eq!(v, [1, 2, 10, 11, 12, 5]);
    v.splice(1..3, Some(20));
    assert_eq!(v, [1, 20, 11, 12, 5]);

    let mut v: SmallVec<_, 3> = SmallVec::from([1, 2, 3]);
    let new = [7, 8];
    let u: SmallVec<_, 2> = v.splice(..2, new.iter().cloned()).collect();
    assert_eq!(v, [7, 8, 3]);
    assert_eq!(u, [1, 2]);
}

#[test]
fn test_splice_inclusive_range2() {
    let mut v: SmallVec<_, 1> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    let t1: SmallVec<_, 2> = v.splice(2..=3, a.iter().cloned()).collect();
    assert_eq!(v, [1, 2, 10, 11, 12, 5]);
    assert_eq!(t1, [3, 4]);
    let t2: SmallVec<_, 2> = v.splice(1..=2, Some(20)).collect();
    assert_eq!(v, [1, 20, 11, 12, 5]);
    assert_eq!(t2, [2, 10]);
}

#[test]
#[should_panic]
fn test_splice_out_of_bounds2() {
    let mut v: SmallVec<_, 1> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    v.splice(5..6, a.iter().cloned());
}

#[test]
#[should_panic]
fn test_splice_inclusive_out_of_bounds2() {
    let mut v: SmallVec<_, 1> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    v.splice(5..=5, a.iter().cloned());
}

#[test]
fn test_splice_items_zero_sized2() {
    let mut smallvec: SmallVec<_, 1> = SmallVec::from(vec![(), (), ()]);
    let smallvec2 = vec![];
    let t: SmallVec<_, 2> = smallvec.splice(1..2, smallvec2.iter().cloned()).collect();
    assert_eq!(smallvec, [(), ()]);
    assert_eq!(t, [()]);
}

#[test]
fn test_splice_unbounded2() {
    let mut smallvec: SmallVec<_, 1> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let t: SmallVec<_, 2> = smallvec.splice(.., None).collect();
    assert_eq!(smallvec, []);
    assert_eq!(t, [1, 2, 3, 4, 5]);
}

#[test]
fn test_splice_forget2() {
    let mut v: SmallVec<_, 1> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let a = [10, 11, 12];
    std::mem::forget(v.splice(2..4, a.iter().cloned()));
    assert_eq!(v, [1, 2]);
}

#[test]
fn shrink_to_fit_unspill() {
    let mut vec = SmallVec::<u8, 2>::from_iter(0..3);
    vec.pop();
    assert!(vec.spilled());
    vec.shrink_to_fit();
    assert!(!vec.spilled(), "shrink_to_fit will un-spill if possible");
}

#[test]
#[should_panic]
fn test_invalid_grow() {
    let mut v: SmallVec<u8, 8> = SmallVec::new();
    v.extend(0..8);
    v.grow_to(5);
}
#[test]
fn test_from_slice() {
    assert_eq!(SmallVec::<u32, 2>::from_slice(&[1]), [1]);
    assert_eq!(SmallVec::<u32, 2>::from_slice(&[1, 2, 3]), [1, 2, 3]);
}

#[test]
fn test_into_iter_clone2() {
    // Test that the cloned iterator yields identical elements and that it owns its own copy
    // (i.e. no use after move errors).
    let mut iter = SmallVec::<u8, 2>::from_iter(0..3).into_iter();
    let mut clone_iter = iter.clone();
    while let Some(x) = iter.next() {
        assert_eq!(x, clone_iter.next().unwrap());
    }
    assert_eq!(clone_iter.next(), None);
}

#[test]
fn test_into_iter_clone_partially_consumed_iterator() {
    // Test that the cloned iterator only contains the remaining elements of the original iterator.
    let mut iter = SmallVec::<u8, 2>::from_iter(0..3).into_iter().skip(1);
    let mut clone_iter = iter.clone();
    while let Some(x) = iter.next() {
        assert_eq!(x, clone_iter.next().unwrap());
    }
    assert_eq!(clone_iter.next(), None);
}
#[test]
fn test_from_and_into() {
    let v: SmallVec<_, 3> = SmallVec::from(vec![1, 2, 3, 4, 5]);
    let spilled = v.spilled();
    let (p, len, cap) = v.into_raw_parts();
    unsafe {
        // Overwrite memory with [4, 5, 6, 7, 8].
        //
        // This is only safe if `spilled` is true! Otherwise, we are
        // writing into the old `SmallVec`'s inline storage on the
        // stack.
        assert!(spilled);
        for i in 0..len {
            let dst = p.add(i);
            ptr::write(dst, 4 + i);
        }
        // Put everything back together into a SmallVec with a different
        // amount of inline storage, but which is still less than `cap`.
        let rebuilt = SmallVec::<_, 3>::from_raw_parts(p, len, cap);
        assert_eq!(rebuilt, [4, 5, 6, 7, 8]);
    }

    let v: SmallVec<i32, 2> = SmallVec::from(vec![-1, 0, 1]);
    let (ptr, len, cap) = v.into_raw_parts();
    let rebuilt: SmallVec<u32, 2> = unsafe {
        // We can now make changes to the components, such as
        // transmuting the raw pointer to a compatible type.
        let ptr = ptr as *mut u32;
        SmallVec::from_raw_parts(ptr, len, cap)
    };
    assert_eq!(rebuilt, &[4294967295, 0, 1]);
}
#[test]
#[ignore]
fn test_from_and_into_spec() {
    let v: SmallVec<_, 10> = SmallVec::from(vec![1, 2, 3]);
    let spilled = v.spilled();
    let (p, len, cap) = v.into_raw_parts();
    unsafe {
        // Overwrite memory with [4, 5, 6].
        //
        // This is only safe if `spilled` is true! Otherwise, we are
        // writing into the old `SmallVec`'s inline storage on the
        // stack.
        assert!(!spilled);
        for i in 0..len {
            let dst = p.add(i);
            ptr::write(dst, 4 + i);
        }
        // Put everything back together into a SmallVec with a different
        // amount of inline storage, but which is still less than `cap`.
        let rebuilt = SmallVec::<_, 10>::from_raw_parts(p, len, cap);
        assert_eq!(rebuilt, [4, 5, 6]);
    }
}
#[test]
fn test_spare_capacity_mut() {
    let mut v: SmallVec<i32, 10> = SmallVec::with_capacity(10);

    // Fill in the first 3 elements.
    let uninit = v.spare_capacity_mut();
    uninit[0].write(0);
    uninit[1].write(1);
    uninit[2].write(2);

    // Mark the first 3 elements of the vector as being initialized.
    unsafe {
        v.set_len(3);
    }

    assert_eq!(&v, &[0, 1, 2]);
}
#[test]
fn test_stable_pointers() {
    /// Pull an element from the iterator, then drop it.
    /// Useful to cover both the `next` and `drop` paths of an iterator.
    fn next_then_drop<I: Iterator>(mut i: I) {
        i.next().unwrap();
        drop(i);
    }

    // Test that, if we reserved enough space, adding and removing elements does not
    // invalidate references into the vector (such as `v0`).  This test also
    // runs in Miri, which would detect such problems.
    // Note that this test does *not* constitute a stable guarantee that all these functions do not
    // reallocate! Only what is explicitly documented at
    // <https://doc.rust-lang.org/nightly/std/vec/struct.Vec.html#guarantees> is stably guaranteed.
    let mut v: SmallVec<i32, 3> = SmallVec::with_capacity(128);
    v.push(13);

    // Laundering the lifetime -- we take care that `v` does not reallocate, so that's okay.
    let mut v0 = v[0];
    // let v0 = unsafe { &mut *(v0 as *mut _) };
    let v0 = ptr::addr_of_mut!(v0);
    // Now do a bunch of things and occasionally use `v0` again to assert it is still valid.

    // Pushing/inserting and popping/removing
    v.push(1);
    v.push(2);
    v.insert(1, 1);
    assert_eq!(unsafe { *v0 }, 13);
    v.remove(1);
    v.pop().unwrap();
    assert_eq!(unsafe { *v0 }, 13);
    v.push(1);
    v.swap_remove(1);
    assert_eq!(v.len(), 2);
    v.swap_remove(1); // swap_remove the last element
    assert_eq!(unsafe { *v0 }, 13); // miri error

    // Appending
    v.append(&mut SmallVec::<_, 2>::from(vec![27, 19]));
    assert_eq!(unsafe { *v0 }, 13);

    // Extending
    v.extend_from_slice(&[1, 2]);
    v.extend(&[1, 2]); // `slice::Iter` (with `T: Copy`) specialization
    v.extend(SmallVec::<_, 2>::from(vec![2, 3])); // `vec::IntoIter` specialization
    v.extend(std::iter::once(3)); // `TrustedLen` specialization
    v.extend(std::iter::empty::<i32>()); // `TrustedLen` specialization with empty iterator
    v.extend(std::iter::once(3).filter(|_| true)); // base case
    v.extend(std::iter::once(&3)); // `cloned` specialization
    assert_eq!(unsafe { *v0 }, 13);

    // Truncation
    v.truncate(2);
    assert_eq!(unsafe { *v0 }, 13);

    // Resizing
    v.resize_with(v.len() + 10, || 42);
    assert_eq!(unsafe { *v0 }, 13);
    v.resize_with(2, || panic!());
    assert_eq!(unsafe { *v0 }, 13);

    // No-op reservation
    v.reserve(32);
    v.reserve_exact(32);
    assert_eq!(unsafe { *v0 }, 13);

    // Partial draining
    v.resize_with(10, || 42);
    next_then_drop(v.drain(5..));
    assert_eq!(unsafe { *v0 }, 13);

    // Splicing
    v.resize_with(10, || 42);
    next_then_drop(v.splice(5.., SmallVec::<_, 2>::from(vec![1, 2, 3, 4, 5]))); // empty tail after range
    assert_eq!(unsafe { *v0 }, 13);
    next_then_drop(v.splice(5..8, SmallVec::<_, 2>::from(vec![1]))); // replacement is smaller than original range
    assert_eq!(unsafe { *v0 }, 13);
    next_then_drop(
        v.splice(
            5..6,
            SmallVec::<_, 2>::from(vec![1; 10])
                .into_iter()
                .filter(|_| true),
        ),
    ); // lower bound not exact
    assert_eq!(unsafe { *v0 }, 13);

    // spare_capacity_mut
    v.spare_capacity_mut();
    assert_eq!(unsafe { *v0 }, 13);

    // Smoke test that would fire even outside Miri if an actual relocation happened.
    unsafe { *v0 -= 13 };
    assert_eq!(unsafe { *v0 }, 0);
}
#[test]
#[cfg_attr(miri, ignore)] // Miri does not support signalling OOM
fn test_try_reserve() {
    // These are the interesting cases:
    // * exactly isize::MAX should never trigger a CapacityOverflow (can be OOM)
    // * > isize::MAX should always fail
    //    * On 16/32-bit should CapacityOverflow
    //    * On 64-bit should OOM
    // * overflow may trigger when adding `len` to `cap` (in number of elements)
    // * overflow may trigger when multiplying `new_cap` by size_of::<T> (to get bytes)

    const MAX_CAP: usize = isize::MAX as usize;
    const MAX_USIZE: usize = usize::MAX;

    // On 16/32-bit, we check that allocations don't exceed isize::MAX,
    // on 64-bit, we assume the OS will give an OOM for such a ridiculous size.
    // Any platform that succeeds for these requests is technically broken with
    // ptr::offset because LLVM is the worst.
    let guards_against_isize = usize::BITS < 64;

    {
        // Note: basic stuff is checked by test_reserve
        let mut empty_bytes: SmallVec<u8, 0> = SmallVec::new();

        // Check isize::MAX doesn't count as an overflow
        if let Err(TryReserveError::CapacityOverflow) = empty_bytes.try_reserve(MAX_CAP) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        // Play it again, frank! (just to be sure)
        if let Err(TryReserveError::CapacityOverflow) = empty_bytes.try_reserve(MAX_CAP) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }

        if guards_against_isize {
            // Check isize::MAX + 1 does count as overflow
            if let Err(TryReserveError::CapacityOverflow) = empty_bytes.try_reserve(MAX_CAP + 1) {
            } else {
                panic!("isize::MAX + 1 should trigger an overflow!")
            }

            // Check usize::MAX does count as overflow
            if let Err(TryReserveError::CapacityOverflow) = empty_bytes.try_reserve(MAX_USIZE) {
            } else {
                panic!("usize::MAX should trigger an overflow!")
            }
        } else {
            // Check isize::MAX + 1 is an OOM
            if let Err(TryReserveError::AllocError { .. }) = empty_bytes.try_reserve(MAX_CAP + 1) {
            } else {
                panic!("isize::MAX + 1 should trigger an OOM!")
            }

            // Check usize::MAX is an OOM
            if let Err(TryReserveError::AllocError { .. }) = empty_bytes.try_reserve(MAX_USIZE) {
                panic!("usize::MAX should trigger an OOM!")
            } else {
            }
        }
    }

    {
        // Same basic idea, but with non-zero len
        let mut ten_bytes: SmallVec<u8, 2> = SmallVec::from(vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        if let Err(TryReserveError::CapacityOverflow) = ten_bytes.try_reserve(MAX_CAP - 10) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if let Err(TryReserveError::CapacityOverflow) = ten_bytes.try_reserve(MAX_CAP - 10) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if guards_against_isize {
            if let Err(TryReserveError::CapacityOverflow) = ten_bytes.try_reserve(MAX_CAP - 9) {
            } else {
                panic!("isize::MAX + 1 should trigger an overflow!");
            }
        } else {
            if let Err(TryReserveError::AllocError { .. }) = ten_bytes.try_reserve(MAX_CAP - 9) {
            } else {
                panic!("isize::MAX + 1 should trigger an OOM!")
            }
        }
        // Should always overflow in the add-to-len
        if let Err(TryReserveError::CapacityOverflow) = ten_bytes.try_reserve(MAX_USIZE) {
        } else {
            panic!("usize::MAX should trigger an overflow!")
        }
    }

    {
        // Same basic idea, but with interesting type size
        let mut ten_u32s: SmallVec<u32, 2> = SmallVec::from(vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        if let Err(TryReserveError::CapacityOverflow) = ten_u32s.try_reserve(MAX_CAP / 4 - 10) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if let Err(TryReserveError::CapacityOverflow) = ten_u32s.try_reserve(MAX_CAP / 4 - 10) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if guards_against_isize {
            if let Err(TryReserveError::CapacityOverflow) = ten_u32s.try_reserve(MAX_CAP / 4 - 9) {
            } else {
                panic!("isize::MAX + 1 should trigger an overflow!");
            }
        } else {
            if let Err(TryReserveError::AllocError { .. }) = ten_u32s.try_reserve(MAX_CAP / 4 - 9) {
            } else {
                panic!("isize::MAX + 1 should trigger an OOM!")
            }
        }
        // Should fail in the mul-by-size
        if let Err(TryReserveError::CapacityOverflow) = ten_u32s.try_reserve(MAX_USIZE - 20) {
        } else {
            panic!("usize::MAX should trigger an overflow!");
        }
    }
}

#[test]
#[cfg_attr(miri, ignore)] // Miri does not support signalling OOM
fn test_try_reserve_exact() {
    // This is exactly the same as test_try_reserve with the method changed.
    // See that test for comments.

    const MAX_CAP: usize = isize::MAX as usize;
    const MAX_USIZE: usize = usize::MAX;

    let guards_against_isize = size_of::<usize>() < 8;

    {
        let mut empty_bytes: SmallVec<u8, 0> = SmallVec::new();

        if let Err(TryReserveError::CapacityOverflow) = empty_bytes.try_reserve_exact(MAX_CAP) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if let Err(TryReserveError::CapacityOverflow) = empty_bytes.try_reserve_exact(MAX_CAP) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }

        if guards_against_isize {
            if let Err(TryReserveError::CapacityOverflow) =
                empty_bytes.try_reserve_exact(MAX_CAP + 1)
            {
            } else {
                panic!("isize::MAX + 1 should trigger an overflow!")
            }

            if let Err(TryReserveError::CapacityOverflow) = empty_bytes.try_reserve_exact(MAX_USIZE)
            {
            } else {
                panic!("usize::MAX should trigger an overflow!")
            }
        } else {
            if let Err(TryReserveError::AllocError { .. }) =
                empty_bytes.try_reserve_exact(MAX_CAP + 1)
            {
            } else {
                panic!("isize::MAX + 1 should trigger an OOM!")
            }

            if let Err(TryReserveError::AllocError { .. }) =
                empty_bytes.try_reserve_exact(MAX_USIZE)
            {
            } else {
                panic!("usize::MAX should trigger an OOM!")
            }
        }
    }

    {
        let mut ten_bytes: SmallVec<u8, 2> = SmallVec::from(vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        if let Err(TryReserveError::CapacityOverflow) = ten_bytes.try_reserve_exact(MAX_CAP - 10) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if let Err(TryReserveError::CapacityOverflow) = ten_bytes.try_reserve_exact(MAX_CAP - 10) {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if guards_against_isize {
            if let Err(TryReserveError::CapacityOverflow) = ten_bytes.try_reserve_exact(MAX_CAP - 9)
            {
            } else {
                panic!("isize::MAX + 1 should trigger an overflow!");
            }
        } else {
            if let Err(TryReserveError::AllocError { .. }) =
                ten_bytes.try_reserve_exact(MAX_CAP - 9)
            {
            } else {
                panic!("isize::MAX + 1 should trigger an OOM!")
            }
        }
        if let Err(TryReserveError::CapacityOverflow) = ten_bytes.try_reserve_exact(MAX_USIZE) {
        } else {
            panic!("usize::MAX should trigger an overflow!")
        }
    }

    {
        let mut ten_u32s: SmallVec<u32, 2> = SmallVec::from(vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

        if let Err(TryReserveError::CapacityOverflow) = ten_u32s.try_reserve_exact(MAX_CAP / 4 - 10)
        {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if let Err(TryReserveError::CapacityOverflow) = ten_u32s.try_reserve_exact(MAX_CAP / 4 - 10)
        {
            panic!("isize::MAX shouldn't trigger an overflow!");
        }
        if guards_against_isize {
            if let Err(TryReserveError::CapacityOverflow) =
                ten_u32s.try_reserve_exact(MAX_CAP / 4 - 9)
            {
            } else {
                panic!("isize::MAX + 1 should trigger an overflow!");
            }
        } else {
            if let Err(TryReserveError::AllocError { .. }) =
                ten_u32s.try_reserve_exact(MAX_CAP / 4 - 9)
            {
            } else {
                panic!("isize::MAX + 1 should trigger an OOM!")
            }
        }
        if let Err(TryReserveError::CapacityOverflow) = ten_u32s.try_reserve_exact(MAX_USIZE - 20) {
        } else {
            panic!("usize::MAX should trigger an overflow!")
        }
    }
}
#[test]
fn test_push_growth_strategy() {
    // If the element size is 1, we jump from 0 to 8, then double.
    {
        let mut v1: SmallVec<u8, 0> = SmallVec::new();
        assert_eq!(v1.capacity(), 0);

        for _ in 0..8 {
            v1.push(0);
        }
        assert_eq!(v1.capacity(), 8);

        for _ in 8..16 {
            v1.push(0);
            assert_eq!(v1.capacity(), 16);
        }

        for _ in 16..32 {
            v1.push(0);
            assert_eq!(v1.capacity(), 32);
        }

        for _ in 32..64 {
            v1.push(0);
            assert_eq!(v1.capacity(), 64);
        }
    }

    // If the element size is 2..=1024, we jump from 0 to 4, then double.
    {
        let mut v2: SmallVec<u16, 0> = SmallVec::new();
        let mut v1024: SmallVec<[u8; 1024], 0> = SmallVec::new();
        assert_eq!(v2.capacity(), 0);
        assert_eq!(v1024.capacity(), 0);

        for _ in 0..4 {
            v2.push(0);
            v1024.push([0; 1024]);
        }
        assert_eq!(v2.capacity(), 4);
        assert_eq!(v1024.capacity(), 4);

        for _ in 4..8 {
            v2.push(0);
            v1024.push([0; 1024]);
            assert_eq!(v2.capacity(), 8);
            assert_eq!(v1024.capacity(), 8);
        }

        for _ in 8..16 {
            v2.push(0);
            v1024.push([0; 1024]);
            assert_eq!(v2.capacity(), 16);
            assert_eq!(v1024.capacity(), 16);
        }

        for _ in 16..32 {
            v2.push(0);
            v1024.push([0; 1024]);
            assert_eq!(v2.capacity(), 32);
            assert_eq!(v1024.capacity(), 32);
        }

        for _ in 32..64 {
            v2.push(0);
            v1024.push([0; 1024]);
            assert_eq!(v2.capacity(), 64);
            assert_eq!(v1024.capacity(), 64);
        }
    }

    // If the element size is > 1024, we jump from 0 to 1, then double.
    {
        let mut v1025: SmallVec<[u8; 1025], 0> = SmallVec::new();
        assert_eq!(v1025.capacity(), 0);

        for _ in 0..1 {
            v1025.push([0; 1025]);
            assert_eq!(v1025.capacity(), 1);
        }

        for _ in 1..2 {
            v1025.push([0; 1025]);
            assert_eq!(v1025.capacity(), 2);
        }

        for _ in 2..4 {
            v1025.push([0; 1025]);
            assert_eq!(v1025.capacity(), 4);
        }

        for _ in 4..8 {
            v1025.push([0; 1025]);
            assert_eq!(v1025.capacity(), 8);
        }

        for _ in 8..16 {
            v1025.push([0; 1025]);
            assert_eq!(v1025.capacity(), 16);
        }

        for _ in 16..32 {
            v1025.push([0; 1025]);
            assert_eq!(v1025.capacity(), 32);
        }

        for _ in 32..64 {
            v1025.push([0; 1025]);
            assert_eq!(v1025.capacity(), 64);
        }
    }
}
