/// `small_vec!` is a macro similar in spirit to the stdlib's `vec!`.
///
/// It supports the creation of `SmallVec` with:
/// * `small_vec![]`
/// * `small_vec![val1, val2, val3, ...]`
/// * `small_vec![val; num_elems]`
///
/// # Examples
///
/// ```
/// use small_vec2::{SmallVec, small_vec};
/// let v: SmallVec<i32, 5> = small_vec![];
/// assert_eq!(v.capacity(), 5);
/// assert_eq!(v.as_slice(), &[]);
/// ```
///
/// - Creates a [`crate::SmallVec`] containing a given list of elements:
///
/// ```
/// use small_vec2::{SmallVec, small_vec};
/// let v: SmallVec<i32, 5> = small_vec![1, 2, 3];
/// assert_eq!(v.capacity(), 5);
/// assert_eq!(v.as_slice(), &[1, 2, 3]);
/// ```
///
/// - Creates a [`crate::SmallVec`] from a given element and size:
///
/// ```
/// use small_vec2::{SmallVec, small_vec};
/// let v: SmallVec<i32, 5> = small_vec![1i32; 3];
/// assert_eq!(v.capacity(), 5);
/// assert_eq!(v.as_slice(), &[1, 1, 1]);
/// ```
pub macro small_vec {
    () => {
        $crate::SmallVec::new()
    },
    ($elem:expr; $n:expr) => {
        $crate::SmallVec::from_elem($elem, $n)
    },
    ($($x:expr),+ $(,)?) => {
        {
            let mut tmp = $crate::SmallVec::new();
            $(
                tmp.push($x);
            )*
            tmp
        }
    }
}
