use crate::{IntoIter, SmallVec, SpecExtend, SpecFromIterNested};
use std::{mem::ManuallyDrop, ptr};
/// Specialization trait used for SmallVec::from_iter
///
/// ## The delegation graph:
///
/// ```text
/// +-------------+
/// |FromIterator |
/// +-+-----------+
///   |
///   v
/// +-+-------------------------------+  +---------------------+
/// |SpecFromIter                  +---->+SpecFromIterNested   |
/// |where I:                      |  |  |where I:             |
/// |  Iterator (default)----------+  |  |  Iterator (default) |
/// |  vec::IntoIter                  |  |  TrustedLen         |
/// +---------------------------------+  +---------------------+
/// ```
pub(crate) trait SpecFromIter<T, I> {
    fn from_iter(iter: I) -> Self;
}

impl<T, I, const N: usize> SpecFromIter<T, I> for SmallVec<T, N>
where
    I: Iterator<Item = T>,
{
    default fn from_iter(iterator: I) -> Self {
        SpecFromIterNested::from_iter(iterator)
    }
}

impl<T: Copy, const N: usize, const M: usize> SpecFromIter<T, IntoIter<T, N>> for SmallVec<T, M> {
    fn from_iter(iterator: IntoIter<T, N>) -> Self {
        // A common case is passing a vector into a function which immediately
        // re-collects into a vector. We can short circuit this if the IntoIter
        // has not been advanced at all.
        // When it has been advanced We can also reuse the memory and move the data to the front.
        // But we only do so when the resulting Vec wouldn't have more unused capacity
        // than creating it through the generic FromIterator implementation would. That limitation
        // is not strictly necessary as Vec's allocation behavior is intentionally unspecified.
        // But it is a conservative choice.
        let not_advanced = iterator.index == 0;
        if not_advanced {
            unsafe {
                let mut it = ManuallyDrop::new(iterator);
                return SmallVec::from_raw_parts(it.buf.as_mut_ptr(), it.len(), it.buf.capacity());
            }
        }
        let a = iterator.len() >= iterator.buf.capacity() / 2;
        if a && !not_advanced {
            unsafe {
                let mut it = ManuallyDrop::new(iterator);

                ptr::copy(it.buf.as_ptr().add(it.index), it.buf.as_mut_ptr(), it.len());

                return SmallVec::from_raw_parts(it.buf.as_mut_ptr(), it.len(), it.buf.capacity());
            }
        }

        let mut vec = SmallVec::new();
        // must delegate to spec_extend() since extend() itself delegates
        // to spec_from for empty Vecs
        vec.spec_extend(iterator);
        vec
    }
}
