use crate::{IntoIter, SmallVec};
use std::{iter::TrustedLen, ptr, slice};

// Specialization trait used for SmallVec::extend
pub(crate) trait SpecExtend<T, I> {
    fn spec_extend(&mut self, iter: I);
}

impl<T, I, const N: usize> SpecExtend<T, I> for SmallVec<T, N>
where
    I: Iterator<Item = T>,
{
    default fn spec_extend(&mut self, iter: I) {
        self.extend_desugared(iter)
    }
}

impl<T, I, const N: usize> SpecExtend<T, I> for SmallVec<T, N>
where
    I: TrustedLen<Item = T>,
{
    default fn spec_extend(&mut self, iterator: I) {
        // This is the case for a TrustedLen iterator.
        let (low, high) = iterator.size_hint();
        if let Some(high_value) = high {
            debug_assert_eq!(
                low,
                high_value,
                "TrustedLen iterator's size hint is not exact: {:?}",
                (low, high)
            );
        }
        if let Some(additional) = high {
            self.reserve(additional);
            unsafe {
                let mut ptr = self.as_mut_ptr().add(self.len());

                iterator.for_each(move |element| {
                    ptr::write(ptr, element);
                    ptr = ptr.offset(1);
                    // NB can't overflow since we would have had to alloc the address space
                    self.len += 1;
                });
            }
        } else {
            // Per TrustedLen contract a `None` upper bound means that the iterator length
            // truly exceeds usize::MAX, which would eventually lead to a capacity overflow anyway.
            // Since the other branch already panics eagerly (via `reserve()`) we do the same here.
            // This avoids additional codegen for a fallback code path which would eventually
            // panic anyway.
            panic!("capacity overflow");
        }
    }
}
// 注意
impl<T, const N: usize, const M: usize> SpecExtend<T, IntoIter<T, N>> for SmallVec<T, M> {
    fn spec_extend(&mut self, mut iterator: IntoIter<T, N>) {
        unsafe {
            self.append_elements(iterator.as_slice() as _);
        }
        iterator.index = iterator.end;
    }
}

impl<'a, T: 'a, I, const N: usize> SpecExtend<&'a T, I> for SmallVec<T, N>
where
    I: Iterator<Item = &'a T>,
    T: Clone,
{
    default fn spec_extend(&mut self, iterator: I) {
        self.spec_extend(iterator.cloned())
    }
}
impl<'a, T: 'a, const N: usize> SpecExtend<&'a T, slice::Iter<'a, T>> for SmallVec<T, N>
where
    T: Copy,
{
    fn spec_extend(&mut self, iterator: slice::Iter<'a, T>) {
        let slice = iterator.as_slice();
        unsafe { self.append_elements(slice) };
    }
}
