use std::{
    alloc::{self, Layout, LayoutError},
    fmt,
};

/// The error type for `try_reserve` methods.
#[derive(Clone, PartialEq, Eq, Debug)]
pub enum TryReserveError {
    /// Error due to the computed capacity exceeding the collection's maximum
    /// (usually `isize::MAX` bytes).
    CapacityOverflow,

    /// The memory allocator returned an error
    AllocError {
        /// The layout of allocation request that failed
        layout: Layout,
    },
}

impl From<LayoutError> for TryReserveError {
    #[inline]
    fn from(_: LayoutError) -> Self {
        TryReserveError::CapacityOverflow
    }
}

impl fmt::Display for TryReserveError {
    fn fmt(&self, fmt: &mut fmt::Formatter<'_>) -> Result<(), fmt::Error> {
        fmt.write_str("memory allocation failed")?;
        let reason = match &self {
            TryReserveError::CapacityOverflow => {
                " because the computed capacity exceeded the collection's maximum"
            }
            TryReserveError::AllocError { .. } => " because the memory allocator returned a error",
        };
        fmt.write_str(reason)
    }
}
pub(crate) fn infallible<T>(result: Result<T, TryReserveError>) -> T {
    match result {
        Ok(x) => x,
        Err(TryReserveError::CapacityOverflow) => panic!("capacity overflow"),
        Err(TryReserveError::AllocError { layout }) => alloc::handle_alloc_error(layout),
    }
}
