use std::path::Path;
use std::process::Command;
use std::{env, fs};

fn main() {
    let out_dir = env::var_os("CARGO_MANIFEST_DIR").unwrap();

    let git_cmd = Command::new("git")
        .arg("rev-parse")
        .arg("--show-toplevel")
        .current_dir(&Path::new(&out_dir))
        .output();

    let output = match git_cmd {
        Ok(output) => output,
        Err(_) => {
            println!("Could not find a git repository.\n");
            return;
        }
    };

    let git_repo_path = String::from_utf8_lossy(&output.stdout);
    let git_repo_path = git_repo_path.trim(); // redefine the var, cleanup stdout from newlines.
    println!("{}", git_repo_path);

    let hooked = Command::new("git")
        .arg("config")
        .arg("core.hooksPath")
        .arg(".sloughi")
        .current_dir(&Path::new(&out_dir))
        .status();

    let git_config_exit_status = match hooked {
        Ok(exit_status) => exit_status,
        Err(_) => {
            println!("Could not setup a custom git hook path!\n");
            return;
        }
    };

    if git_config_exit_status.success() {
        let dest_path = Path::new(&git_repo_path.trim()).join(".sloughi").join("_");
        println!("{}", dest_path.display());
        fs::create_dir_all(dest_path.clone()).unwrap();
        fs::write(dest_path.join(".gitignore"), "*").map_err(|_| {
            println!("Could not write to .gitignore file!\n");
        }).unwrap();

        fs::write(
            &dest_path.join("sloughi.sh"),
            r#"#!/bin/sh

if [ -z "$husky_skip_init" ]; then
  debug () {
    if [ "$HUSKY_DEBUG" = "1" ]; then
      echo "husky (debug) - $1"
    fi
  }

  readonly hook_name="$(basename "$0")"
  debug "starting $hook_name..."

  if [ "$HUSKY" = "0" ]; then
    debug "HUSKY env variable is set to 0, skipping hook"
    exit 0
  fi

  export readonly husky_skip_init=1
  sh -e "$0" "$@"
  exitCode="$?"

  if [ $exitCode != 0 ]; then
    echo "husky - $hook_name hook exited with code $exitCode (error)"
  fi

  exit $exitCode
fi

"#,
        )
        .unwrap();
    }

    println!("cargo:rerun-if-changed=build.rs");
}
