typedef ulong limb_t;

typedef limb_t vec256[256/(8*sizeof(limb_t))];

struct vec256_t {
    // vec256.l to treat it as 4 64-bit limbed structure
    // vec256.v to treat it as 8 32-bit limbed structure

    // it uses the same memory space but it can be
    // indexed in 2 different ways
    union { vec256 l;
            uint v[sizeof(vec256)/sizeof(uint)];
    };
    
};

struct vec256_shared_t {
    
    // first limb of 8 limbs is stored in a register (i.e v0)
    uint v0;
    // remaining 7 limbs are stored in shared memory
    // v_off marks the start of the shared memory region
    // where this variable's limbs are stored
    uint v_off;
};

// Pseudocode of the function below

// r = a
// for (int i = 0; i < n; i++) {
//     r = r * r;
// }
// if (val == 0)
//     r = r * a;
// else if (val == 1)
//     r = r * b;
//
// return r;
inline struct vec256_t sqr_n_mul_mod_256_189_val_or_self(const struct vec256_t a,
                                                         unsigned int n,
                                                         int val,
                                                         const struct vec256_shared_t b,
                                                         __local unsigned int scratchpad[]) {

    struct vec256_t ret;
    
    asm volatile("{\n\t"
        "   .reg.b32    cnt, vptr;\t\n"
        "   .reg.b32    a<8>, b<8>, carry;\t\n"
        "   .reg.b32    sqr<16>;\t\n"
        "   .reg.pred   p;\t\n"
        "\t\n"
        "   mov.b32     vptr, %5;\t\n"
        "   ld.b32    cnt, [%4+4*1];\t\n"
        "   add.u32     vptr, vptr, cnt;\t\n"
        "   mov.b32 carry, %3;\t\n"
        "   setp.ne.u32 p, carry, 0;\t\n"
        "\t\n"
        "   ld.b32  a0, [%1+4*0];\t\n"
        "@!p    mov.b32     b0, a0;\t\n"
        "@p ld.b32  b0, [%4];\t\n"
        "   ld.b32  a1, [%1+4*1];\t\n"
        "@!p    mov.b32     b1, a1;\t\n"
        "@p ld.shared.b32   b1, [vptr+4*(1-1)];\t\n"
        "   ld.b32  a2, [%1+4*2];\t\n"
        "@!p    mov.b32     b2, a2;\t\n"
        "@p ld.shared.b32   b2, [vptr+4*(2-1)];\t\n"
        "   ld.b32  a3, [%1+4*3];\t\n"
        "@!p    mov.b32     b3, a3;\t\n"
        "@p ld.shared.b32   b3, [vptr+4*(3-1)];\t\n"
        "   ld.b32  a4, [%1+4*4];\t\n"
        "@!p    mov.b32     b4, a4;\t\n"
        "@p ld.shared.b32   b4, [vptr+4*(4-1)];\t\n"
        "   ld.b32  a5, [%1+4*5];\t\n"
        "@!p    mov.b32     b5, a5;\t\n"
        "@p ld.shared.b32   b5, [vptr+4*(5-1)];\t\n"
        "   ld.b32  a6, [%1+4*6];\t\n"
        "@!p    mov.b32     b6, a6;\t\n"
        "@p ld.shared.b32   b6, [vptr+4*(6-1)];\t\n"
        "   ld.b32  a7, [%1+4*7];\t\n"
        "@!p    mov.b32     b7, a7;\t\n"
        "@p ld.shared.b32   b7, [vptr+4*(7-1)];\t\n"
        "   mov.u32 cnt, %2;\t\n"
        "   mov.u32 carry, 0;\t\n"
        "Loop:\t\n"
        "    {  // square\t\n"
        "   .reg.b32    odd<14>;\t\n"
        "\t\n"
        "   mul.lo.u32  sqr2, a2, a0;\t\n"
        "   mul.hi.u32  sqr3, a2, a0;\t\n"
        "   mul.lo.u32  sqr4, a4, a0;\t\n"
        "   mul.hi.u32  sqr5, a4, a0;\t\n"
        "   mul.lo.u32  sqr6, a6, a0;\t\n"
        "   mul.hi.u32  sqr7, a6, a0;\t\n"
        "   mul.lo.u32  odd0, a1, a0;\t\n"
        "   mul.hi.u32  odd1, a1, a0;\t\n"
        "   mul.lo.u32  odd2, a3, a0;\t\n"
        "   mul.hi.u32  odd3, a3, a0;\t\n"
        "   mul.lo.u32  odd4, a5, a0;\t\n"
        "   mul.hi.u32  odd5, a5, a0;\t\n"
        "   mul.lo.u32  odd6, a7, a0;\t\n"
        "   mul.hi.u32  odd7, a7, a0;\t\n"
        "\t\n"
        "   mad.lo.u32  a0, carry, 189, a0;\t\n"
        "   mad.lo.cc.u32   sqr4, a3, a1, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, a3, a1, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, a5, a1, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a5, a1, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a7, a1, 0;\t\n"
        "   madc.hi.u32 sqr9, a7, a1, 0;\t\n"
        "   mad.lo.cc.u32   odd2, a2, a1, odd2;\t\n"
        "   madc.hi.cc.u32  odd3, a2, a1, odd3;\t\n"
        "   madc.lo.cc.u32  odd4, a4, a1, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, a4, a1, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, a6, a1, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a6, a1, odd7;\t\n"
        "   addc.u32    sqr9, sqr9, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   odd4, a3, a2, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, a3, a2, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, a5, a2, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a5, a2, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, a7, a2, 0;\t\n"
        "   madc.hi.u32 odd9, a7, a2, 0;\t\n"
        "   mad.lo.cc.u32   sqr6, a4, a2, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a4, a2, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a6, a2, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a6, a2, sqr9;\t\n"
        "   addc.u32    odd9, odd9, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr8, a5, a3, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a5, a3, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, a7, a3, 0;\t\n"
        "   madc.hi.u32 sqr11, a7, a3, 0;\t\n"
        "   mad.lo.cc.u32   odd6, a4, a3, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a4, a3, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, a6, a3, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, a6, a3, odd9;\t\n"
        "   addc.u32    sqr11, sqr11, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   odd8, a5, a4, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, a5, a4, odd9;\t\n"
        "   madc.lo.cc.u32  odd10, a7, a4, 0;\t\n"
        "   madc.hi.u32 odd11, a7, a4, 0;\t\n"
        "   mad.lo.cc.u32   sqr10, a6, a4, sqr10;\t\n"
        "   madc.hi.cc.u32  sqr11, a6, a4, sqr11;\t\n"
        "\t\n"
        "   madc.lo.cc.u32  sqr12, a7, a5, 0;\t\n"
        "   madc.hi.u32 sqr13, a7, a5, 0;\t\n"
        "   mad.lo.cc.u32   odd10, a6, a5, odd10;\t\n"
        "   madc.hi.cc.u32  odd11, a6, a5, odd11;\t\n"
        "\t\n"
        "   madc.lo.cc.u32  odd12, a7, a6, 0;\t\n"
        "   madc.hi.u32 odd13, a7, a6, 0;\t\n"
        "\t\n"
        "   add.cc.u32  sqr2, sqr2, odd1;\t\n"
        "   addc.cc.u32 sqr3, sqr3, odd2;\t\n"
        "   addc.cc.u32 sqr4, sqr4, odd3;\t\n"
        "   addc.cc.u32 sqr5, sqr5, odd4;\t\n"
        "   addc.cc.u32 sqr6, sqr6, odd5;\t\n"
        "   addc.cc.u32 sqr7, sqr7, odd6;\t\n"
        "   addc.cc.u32 sqr8, sqr8, odd7;\t\n"
        "   addc.cc.u32 sqr9, sqr9, odd8;\t\n"
        "   addc.cc.u32 sqr10, sqr10, odd9;\t\n"
        "   addc.cc.u32 sqr11, sqr11, odd10;\t\n"
        "   addc.cc.u32 sqr12, sqr12, odd11;\t\n"
        "   addc.cc.u32 sqr13, sqr13, odd12;\t\n"
        "   addc.u32    sqr14, odd13, 0;\t\n"
        "   mov.u32 sqr0, 0;\t\n"
        "   add.cc.u32  sqr1, odd0, odd0;\t\n"
        "   addc.cc.u32 sqr2, sqr2, sqr2;\t\n"
        "   addc.cc.u32 sqr3, sqr3, sqr3;\t\n"
        "   addc.cc.u32 sqr4, sqr4, sqr4;\t\n"
        "   addc.cc.u32 sqr5, sqr5, sqr5;\t\n"
        "   addc.cc.u32 sqr6, sqr6, sqr6;\t\n"
        "   addc.cc.u32 sqr7, sqr7, sqr7;\t\n"
        "   addc.cc.u32 sqr8, sqr8, sqr8;\t\n"
        "   addc.cc.u32 sqr9, sqr9, sqr9;\t\n"
        "   addc.cc.u32 sqr10, sqr10, sqr10;\t\n"
        "   addc.cc.u32 sqr11, sqr11, sqr11;\t\n"
        "   addc.cc.u32 sqr12, sqr12, sqr12;\t\n"
        "   addc.cc.u32 sqr13, sqr13, sqr13;\t\n"
        "   addc.cc.u32 sqr14, sqr14, sqr14;\t\n"
        "   addc.u32    sqr15, 0, 0;\t\n"
        "   mad.lo.cc.u32   sqr0, a0, a0, sqr0;\t\n"
        "   madc.hi.cc.u32  sqr1, a0, a0, sqr1;\t\n"
        "   madc.lo.cc.u32  sqr2, a1, a1, sqr2;\t\n"
        "   madc.hi.cc.u32  sqr3, a1, a1, sqr3;\t\n"
        "   madc.lo.cc.u32  sqr4, a2, a2, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, a2, a2, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, a3, a3, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a3, a3, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a4, a4, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a4, a4, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, a5, a5, sqr10;\t\n"
        "   madc.hi.cc.u32  sqr11, a5, a5, sqr11;\t\n"
        "   madc.lo.cc.u32  sqr12, a6, a6, sqr12;\t\n"
        "   madc.hi.cc.u32  sqr13, a6, a6, sqr13;\t\n"
        "   madc.lo.cc.u32  sqr14, a7, a7, sqr14;\t\n"
        "   madc.hi.cc.u32  sqr15, a7, a7, sqr15;\t\n"
        "    }\t\n"
        "\t\n"
        "    {  // reduce\t\n"
        "   .reg.b32    top;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr0, sqr8, 189, sqr0;\t\n"
        "   madc.hi.cc.u32  sqr1, sqr8, 189, sqr1;\t\n"
        "   madc.lo.cc.u32  sqr2, sqr10, 189, sqr2;\t\n"
        "   madc.hi.cc.u32  sqr3, sqr10, 189, sqr3;\t\n"
        "   madc.lo.cc.u32  sqr4, sqr12, 189, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, sqr12, 189, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, sqr14, 189, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, sqr14, 189, sqr7;\t\n"
        "   addc.u32    carry, 0, 0;\t\n"
        "   mul.lo.u32  sqr8, sqr9, 189;\t\n"
        "   mul.hi.u32  sqr9, sqr9, 189;\t\n"
        "   mul.lo.u32  sqr10, sqr11, 189;\t\n"
        "   mul.hi.u32  sqr11, sqr11, 189;\t\n"
        "   mul.lo.u32  sqr12, sqr13, 189;\t\n"
        "   mul.hi.u32  sqr13, sqr13, 189;\t\n"
        "   mul.lo.u32  sqr14, sqr15, 189;\t\n"
        "   mul.hi.u32  sqr15, sqr15, 189;\t\n"
        "   add.cc.u32  sqr1, sqr1, sqr8;\t\n"
        "   addc.cc.u32 sqr2, sqr2, sqr9;\t\n"
        "   addc.cc.u32 sqr3, sqr3, sqr10;\t\n"
        "   addc.cc.u32 sqr4, sqr4, sqr11;\t\n"
        "   addc.cc.u32 sqr5, sqr5, sqr12;\t\n"
        "   addc.cc.u32 sqr6, sqr6, sqr13;\t\n"
        "   addc.cc.u32 sqr7, sqr7, sqr14;\t\n"
        "   addc.u32    carry, carry, sqr15;\t\n"
        "   mad.lo.cc.u32   a0, carry, 189, sqr0;\t\n"
        "   addc.cc.u32 a1, sqr1, 0;\t\n"
        "   addc.cc.u32 a2, sqr2, 0;\t\n"
        "   addc.cc.u32 a3, sqr3, 0;\t\n"
        "   addc.cc.u32 a4, sqr4, 0;\t\n"
        "   addc.cc.u32 a5, sqr5, 0;\t\n"
        "   addc.cc.u32 a6, sqr6, 0;\t\n"
        "   addc.cc.u32 a7, sqr7, 0;\t\n"
        "   addc.u32    carry, 0, 0;\t\n"
        "    }\t\n"
        "\t\n"
        "   sub.u32     cnt, cnt, 1;\t\n"
        "   setp.ne.u32 p, cnt, 0;\t\n"
        "@p bra.uni     Loop;\t\n"
        "\t\n"
        "   mad.lo.u32  a0, carry, 189, a0;\t\n"
        "    {  // multiply\t\n"
        "   .reg.b32    odd<14>;\t\n"
        "   .reg.b32    bi;\t\n"
        "\t\n"
        "   mul.lo.u32  sqr0, a0, b0;\t\n"
        "   mul.hi.u32  sqr1, a0, b0;\t\n"
        "   mul.lo.u32  sqr2, a2, b0;\t\n"
        "   mul.hi.u32  sqr3, a2, b0;\t\n"
        "   mul.lo.u32  sqr4, a4, b0;\t\n"
        "   mul.hi.u32  sqr5, a4, b0;\t\n"
        "   mul.lo.u32  sqr6, a6, b0;\t\n"
        "   mul.hi.u32  sqr7, a6, b0;\t\n"
        "   mul.lo.u32  odd0, a1, b0;\t\n"
        "   mul.hi.u32  odd1, a1, b0;\t\n"
        "   mul.lo.u32  odd2, a3, b0;\t\n"
        "   mul.hi.u32  odd3, a3, b0;\t\n"
        "   mul.lo.u32  odd4, a5, b0;\t\n"
        "   mul.hi.u32  odd5, a5, b0;\t\n"
        "   mul.lo.u32  odd6, a7, b0;\t\n"
        "   mul.hi.u32  odd7, a7, b0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr2, a1, b1, sqr2;\t\n"
        "   madc.hi.cc.u32  sqr3, a1, b1, sqr3;\t\n"
        "   madc.lo.cc.u32  sqr4, a3, b1, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, a3, b1, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, a5, b1, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a5, b1, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a7, b1, 0;\t\n"
        "   madc.hi.u32 sqr9, a7, b1, 0;\t\n"
        "   mad.lo.cc.u32   odd0, a0, b1, odd0;\t\n"
        "   madc.hi.cc.u32  odd1, a0, b1, odd1;\t\n"
        "   madc.lo.cc.u32  odd2, a2, b1, odd2;\t\n"
        "   madc.hi.cc.u32  odd3, a2, b1, odd3;\t\n"
        "   madc.lo.cc.u32  odd4, a4, b1, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, a4, b1, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, a6, b1, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a6, b1, odd7;\t\n"
        "   addc.u32    sqr9, sqr9, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   odd2, a1, b2, odd2;\t\n"
        "   madc.hi.cc.u32  odd3, a1, b2, odd3;\t\n"
        "   madc.lo.cc.u32  odd4, a3, b2, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, a3, b2, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, a5, b2, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a5, b2, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, a7, b2, 0;\t\n"
        "   madc.hi.u32 odd9, a7, b2, 0;\t\n"
        "   mad.lo.cc.u32   sqr2, a0, b2, sqr2;\t\n"
        "   madc.hi.cc.u32  sqr3, a0, b2, sqr3;\t\n"
        "   madc.lo.cc.u32  sqr4, a2, b2, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, a2, b2, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, a4, b2, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a4, b2, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a6, b2, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a6, b2, sqr9;\t\n"
        "   addc.u32    odd9, odd9, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr4, a1, b3, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, a1, b3, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, a3, b3, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a3, b3, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a5, b3, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a5, b3, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, a7, b3, 0;\t\n"
        "   madc.hi.u32 sqr11, a7, b3, 0;\t\n"
        "   mad.lo.cc.u32   odd2, a0, b3, odd2;\t\n"
        "   madc.hi.cc.u32  odd3, a0, b3, odd3;\t\n"
        "   madc.lo.cc.u32  odd4, a2, b3, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, a2, b3, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, a4, b3, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a4, b3, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, a6, b3, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, a6, b3, odd9;\t\n"
        "   addc.u32    sqr11, sqr11, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   odd4, a1, b4, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, a1, b4, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, a3, b4, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a3, b4, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, a5, b4, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, a5, b4, odd9;\t\n"
        "   madc.lo.cc.u32  odd10, a7, b4, 0;\t\n"
        "   madc.hi.u32 odd11, a7, b4, 0;\t\n"
        "   mad.lo.cc.u32   sqr4, a0, b4, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, a0, b4, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, a2, b4, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a2, b4, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a4, b4, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a4, b4, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, a6, b4, sqr10;\t\n"
        "   madc.hi.cc.u32  sqr11, a6, b4, sqr11;\t\n"
        "   addc.u32    odd11, odd11, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr6, a1, b5, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a1, b5, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a3, b5, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a3, b5, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, a5, b5, sqr10;\t\n"
        "   madc.hi.cc.u32  sqr11, a5, b5, sqr11;\t\n"
        "   madc.lo.cc.u32  sqr12, a7, b5, 0;\t\n"
        "   madc.hi.u32 sqr13, a7, b5, 0;\t\n"
        "   mad.lo.cc.u32   odd4, a0, b5, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, a0, b5, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, a2, b5, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a2, b5, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, a4, b5, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, a4, b5, odd9;\t\n"
        "   madc.lo.cc.u32  odd10, a6, b5, odd10;\t\n"
        "   madc.hi.cc.u32  odd11, a6, b5, odd11;\t\n"
        "   addc.u32    sqr13, sqr13, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   odd6, a1, b6, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a1, b6, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, a3, b6, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, a3, b6, odd9;\t\n"
        "   madc.lo.cc.u32  odd10, a5, b6, odd10;\t\n"
        "   madc.hi.cc.u32  odd11, a5, b6, odd11;\t\n"
        "   madc.lo.cc.u32  odd12, a7, b6, 0;\t\n"
        "   madc.hi.u32 odd13, a7, b6, 0;\t\n"
        "   mad.lo.cc.u32   sqr6, a0, b6, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, a0, b6, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, a2, b6, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a2, b6, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, a4, b6, sqr10;\t\n"
        "   madc.hi.cc.u32  sqr11, a4, b6, sqr11;\t\n"
        "   madc.lo.cc.u32  sqr12, a6, b6, sqr12;\t\n"
        "   madc.hi.cc.u32  sqr13, a6, b6, sqr13;\t\n"
        "   addc.u32    odd13, odd13, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr8, a1, b7, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, a1, b7, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, a3, b7, sqr10;\t\n"
        "   madc.hi.cc.u32  sqr11, a3, b7, sqr11;\t\n"
        "   madc.lo.cc.u32  sqr12, a5, b7, sqr12;\t\n"
        "   madc.hi.cc.u32  sqr13, a5, b7, sqr13;\t\n"
        "   madc.lo.cc.u32  sqr14, a7, b7, 0;\t\n"
        "   madc.hi.u32 sqr15, a7, b7, 0;\t\n"
        "   mad.lo.cc.u32   odd6, a0, b7, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, a0, b7, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, a2, b7, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, a2, b7, odd9;\t\n"
        "   madc.lo.cc.u32  odd10, a4, b7, odd10;\t\n"
        "   madc.hi.cc.u32  odd11, a4, b7, odd11;\t\n"
        "   madc.lo.cc.u32  odd12, a6, b7, odd12;\t\n"
        "   madc.hi.cc.u32  odd13, a6, b7, odd13;\t\n"
        "   addc.u32    sqr15, sqr15, 0;\t\n"
        "\t\n"
        "   add.cc.u32  sqr1, sqr1, odd0;\t\n"
        "   addc.cc.u32 sqr2, sqr2, odd1;\t\n"
        "   addc.cc.u32 sqr3, sqr3, odd2;\t\n"
        "   addc.cc.u32 sqr4, sqr4, odd3;\t\n"
        "   addc.cc.u32 sqr5, sqr5, odd4;\t\n"
        "   addc.cc.u32 sqr6, sqr6, odd5;\t\n"
        "   addc.cc.u32 sqr7, sqr7, odd6;\t\n"
        "   addc.cc.u32 sqr8, sqr8, odd7;\t\n"
        "   addc.cc.u32 sqr9, sqr9, odd8;\t\n"
        "   addc.cc.u32 sqr10, sqr10, odd9;\t\n"
        "   addc.cc.u32 sqr11, sqr11, odd10;\t\n"
        "   addc.cc.u32 sqr12, sqr12, odd11;\t\n"
        "   addc.cc.u32 sqr13, sqr13, odd12;\t\n"
        "   addc.cc.u32 sqr14, sqr14, odd13;\t\n"
        "   addc.u32    sqr15, sqr15, 0;\t\n"
        "    }\t\n"
        "\t\n"
        "    {  // reduce\t\n"
        "   .reg.b32    top;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr0, sqr8, 189, sqr0;\t\n"
        "   madc.hi.cc.u32  sqr1, sqr8, 189, sqr1;\t\n"
        "   madc.lo.cc.u32  sqr2, sqr10, 189, sqr2;\t\n"
        "   madc.hi.cc.u32  sqr3, sqr10, 189, sqr3;\t\n"
        "   madc.lo.cc.u32  sqr4, sqr12, 189, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, sqr12, 189, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, sqr14, 189, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, sqr14, 189, sqr7;\t\n"
        "   addc.u32    top, 0, 0;\t\n"
        "   mul.lo.u32  sqr8, sqr9, 189;\t\n"
        "   mul.hi.u32  sqr9, sqr9, 189;\t\n"
        "   mul.lo.u32  sqr10, sqr11, 189;\t\n"
        "   mul.hi.u32  sqr11, sqr11, 189;\t\n"
        "   mul.lo.u32  sqr12, sqr13, 189;\t\n"
        "   mul.hi.u32  sqr13, sqr13, 189;\t\n"
        "   mul.lo.u32  sqr14, sqr15, 189;\t\n"
        "   mul.hi.u32  sqr15, sqr15, 189;\t\n"
        "   add.cc.u32  sqr1, sqr1, sqr8;\t\n"
        "   addc.cc.u32 sqr2, sqr2, sqr9;\t\n"
        "   addc.cc.u32 sqr3, sqr3, sqr10;\t\n"
        "   addc.cc.u32 sqr4, sqr4, sqr11;\t\n"
        "   addc.cc.u32 sqr5, sqr5, sqr12;\t\n"
        "   addc.cc.u32 sqr6, sqr6, sqr13;\t\n"
        "   addc.cc.u32 sqr7, sqr7, sqr14;\t\n"
        "   addc.u32    top, top, sqr15;\t\n"
        "   mad.lo.cc.u32   a0, top, 189, sqr0;\t\n"
        "   addc.cc.u32 a1, sqr1, 0;\t\n"
        "   addc.cc.u32 a2, sqr2, 0;\t\n"
        "   addc.cc.u32 a3, sqr3, 0;\t\n"
        "   addc.cc.u32 a4, sqr4, 0;\t\n"
        "   addc.cc.u32 a5, sqr5, 0;\t\n"
        "   addc.cc.u32 a6, sqr6, 0;\t\n"
        "   addc.cc.u32 a7, sqr7, 0;\t\n"
        "   addc.u32    top, 0, 0;\t\n"
        "   mad.lo.u32  a0, top, 189, a0;\t\n"
        "    }\t\n"
        "\t\n"
        "   st.u32  [%0+4*0], a0;\t\n"
        "   st.u32  [%0+4*1], a1;\t\n"
        "   st.u32  [%0+4*2], a2;\t\n"
        "   st.u32  [%0+4*3], a3;\t\n"
        "   st.u32  [%0+4*4], a4;\t\n" 
        "   st.u32  [%0+4*5], a5;\t\n"
        "   st.u32  [%0+4*6], a6;\t\n"
        "   st.u32  [%0+4*7], a7;\t\n"
        "}"
        :: "l" (ret.v), "l" (a.v), "r" (n), "r" (val), "l" (&b.v0), "r" (scratchpad) : "memory"
        );
        
    return ret;
}

inline struct vec256_t vec256_load_global(const limb_t* aptr) {
    
    struct vec256_t ret;
    
    asm volatile(
        "{\t\n"
        "   .reg.b64    aptr1, aptr2, tptr1, tptr2;\t\n"
        "   .reg.b32    a<13>, l;\t\n"
        "   .reg.pred   odd;\t\n"
        "\t\n"
        "   mov.b64     aptr1,  %1;\t\n"
        "   cvta.to.global.u64  aptr1, aptr1;   // a 'nop'\t\n"
        "   // copy neighbour's pointer\t\n"
        "   mov.b64         {a0, a1}, aptr1;\t\n"
        "   shfl.sync.bfly.b32  a2, a0, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a3, a1, 1, 0x1f, -1;\t\n"
        "   mov.b64         aptr2, {a2, a3};\t\n"
        "   mov.b32     l, %%laneid;\t\n"
        "   and.b32     l, l, 1;\t\n"
        "   setp.ne.u32 odd, l, 0;\t\n"
        "\t\n"
        "   add.u64 tptr1, aptr1, 16;\t\n"
        "   add.u64 tptr2, aptr2, 16;\t\n"
        "@odd   mov.b64 aptr2, tptr1;\t\n"
        "@odd   mov.b64 aptr1, tptr2;\t\n"
        "\t\n"
        "   // coalesce 256 bits from even thread\t\n"
        "   ld.global.cg.v4.b32 {a0,a1,a2,a3}, [aptr1];\t\n"
        "   // coalesce 256 bits from odd thread\t\n"
        "   ld.global.cg.v4.b32 {a4,a5,a6,a7}, [aptr2];\t\n"
        "\t\n"
        "   prefetch.global.L2  [tptr1+16];\t\n"
        "\t\n"
        "   // transpose 128-bit halves\t\n"
        "   shfl.sync.bfly.b32  a8,  a0, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a9,  a1, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a10, a2, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a11, a3, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a12, a4, 1, 0x1f, -1;\t\n"
        "@!odd  mov.b32         a4,  a8;\t\n"
        "   shfl.sync.bfly.b32  a8,  a5, 1, 0x1f, -1;\t\n"
        "@!odd  mov.b32         a5,  a9;\t\n"
        "   shfl.sync.bfly.b32  a9,  a6, 1, 0x1f, -1;\t\n"
        "@!odd  mov.b32         a6,  a10;\t\n"
        "   shfl.sync.bfly.b32  a10, a7, 1, 0x1f, -1;\t\n"
        "@!odd  mov.b32         a7,  a11;\t\n"
        "@odd   mov.b32         a0,  a12;\t\n"
        "@odd   mov.b32         a1,  a8;\t\n"
        "@odd   mov.b32         a2,  a9;\t\n"
        "@odd   mov.b32         a3,  a10;\t\n"
        "\t\n"
        "   st.b32      [%0+4*0], a0;\t\n"
        "   st.b32      [%0+4*1], a1;\t\n"
        "   st.b32      [%0+4*2], a2;\t\n"
        "   st.b32      [%0+4*3], a3;\t\n"
        "   st.b32      [%0+4*4], a4;\t\n"
        "   st.b32      [%0+4*5], a5;\t\n"
        "   st.b32      [%0+4*6], a6;\t\n"
        "   st.b32      [%0+4*7], a7;\t\n"
        "\t\n"
        "}"
        :: "l" (ret.v), "l" (aptr)
        );
        
    return ret;
}

inline void vec256_store_global(struct vec256_t inp, const limb_t* rptr) {

    asm volatile(
        "{\t\n"
        "   .reg.b64    rptr1, rptr2, tptr1, tptr2;\t\n"
        "   .reg.b32    a<13>, l;\t\n"
        "   .reg.pred   even;\t\n"
        "\t\n"
        "   mov.b64     rptr1, %1;\t\n"
        "   cvta.to.global.u64  rptr1, rptr1;   // a 'nop'\t\n"
        "   // copy neighbour's pointer\t\n"
        "   mov.b64         {a0, a1}, rptr1;\t\n"
        "   shfl.sync.bfly.b32  a2, a0, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a3, a1, 1, 0x1f, -1;\t\n"
        "   mov.b64         rptr2, {a2, a3};\t\n"
        "   ld.u32  a0, [%0+4*0];\t\n"
        "   ld.u32  a1, [%0+4*1];\t\n"
        "   ld.u32  a2, [%0+4*2];\t\n"
        "   ld.u32  a3, [%0+4*3];\t\n"
        "   ld.u32  a4, [%0+4*4];\t\n"
        "   ld.u32  a5, [%0+4*5];\t\n"
        "   ld.u32  a6, [%0+4*6];\t\n"
        "   ld.u32  a7, [%0+4*7];\t\n"
        "   mov.b32     l, %%laneid;\t\n"
        "   and.b32     l, l, 1;\t\n"
        "   setp.eq.u32 even, l, 0;\t\n"
        "\t\n"
        "   add.u64 tptr1, rptr1, 16;\t\n"
        "   add.u64 tptr2, rptr2, 16;\t\n"
        "@!even mov.b64 rptr2, tptr1;\t\n"
        "@!even mov.b64 rptr1, tptr2;\t\n"
        "\t\n"
        "   // transpose 128-bit halves\t\n"
        "   shfl.sync.bfly.b32  a8,  a0, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a9,  a1, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a10, a2, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a11, a3, 1, 0x1f, -1;\t\n"
        "   shfl.sync.bfly.b32  a12, a4, 1, 0x1f, -1;\t\n"
        "@even  mov.b32         a4,  a8;\t\n"
        "   shfl.sync.bfly.b32  a8,  a5, 1, 0x1f, -1;\t\n"
        "@even  mov.b32         a5,  a9;\t\n"
        "   shfl.sync.bfly.b32  a9,  a6, 1, 0x1f, -1;\t\n"
        "@even  mov.b32         a6,  a10;\t\n"
        "   shfl.sync.bfly.b32  a10, a7, 1, 0x1f, -1;\t\n"
        "@even  mov.b32         a7,  a11;\t\n"
        "@!even mov.b32         a0,  a12;\t\n"
        "@!even mov.b32         a1,  a8;\t\n"
        "@!even mov.b32         a2,  a9;\t\n"
        "@!even mov.b32         a3,  a10;\t\n"
        "\t\n"
        "   st.global.cs.v4.b32 [rptr1], {a0, a1, a2, a3};\t\n"
        "   st.global.cs.v4.b32 [rptr2], {a4, a5, a6, a7};\t\n"
        "\t\n"
        "}"
        :: "l" (inp.v), "l" (rptr)
        );  
}

inline struct vec256_t redc_mod_256_189_val(const struct vec256_t inp) {

    struct vec256_t ret;

    asm volatile(
        "{\t\n"
        "   .reg.b64    rptr, aptr;\t\n"
        "   .reg.b32    a<8>, b<9>;\t\n"
        "   .reg.pred   p;\t\n"
        "\t\n"
        "   ld.u32  a0, [%1+4*0];\t\n"
        "   ld.u32  a1, [%1+4*1];\t\n"
        "   ld.u32  a2, [%1+4*2];\t\n"
        "   ld.u32  a3, [%1+4*3];\t\n"
        "   ld.u32  a4, [%1+4*4];\t\n"
        "   ld.u32  a5, [%1+4*5];\t\n"
        "   ld.u32  a6, [%1+4*6];\t\n"
        "   ld.u32  a7, [%1+4*7];\t\n"
        "   add.cc.u32  b0, a0, 189;\t\n"
        "   addc.cc.u32 b1, a1, 0;\t\n"
        "   addc.cc.u32 b2, a2, 0;\t\n"
        "   addc.cc.u32 b3, a3, 0;\t\n"
        "   addc.cc.u32 b4, a4, 0;\t\n"
        "   addc.cc.u32 b5, a5, 0;\t\n"
        "   addc.cc.u32 b6, a6, 0;\t\n"
        "   addc.cc.u32 b7, a7, 0;\t\n"
        "   addc.u32    b8, 0, 0;\t\n"
        "   setp.ne.u32 p, b8, 0;\t\n"
        "@p mov.b32 a0, b0;\t\n"
        "   st.u32  [%0+4*0], a0;\t\n"
        "@p mov.b32 a1, b1;\t\n"
        "   st.u32  [%0+4*1], a1;\t\n"
        "@p mov.b32 a2, b2;\t\n"
        "   st.u32  [%0+4*2], a2;\t\n"
        "@p mov.b32 a3, b3;\t\n"
        "   st.u32  [%0+4*3], a3;\t\n"
        "@p mov.b32 a4, b4;\t\n"
        "   st.u32  [%0+4*4], a4;\t\n"
        "@p mov.b32 a5, b5;\t\n"
        "   st.u32  [%0+4*5], a5;\t\n"
        "@p mov.b32 a6, b6;\t\n"
        "   st.u32  [%0+4*6], a6;\t\n"
        "@p mov.b32 a7, b7;\t\n"
        "   st.u32  [%0+4*7], a7;\t\n"
        "}"
        :: "l" (ret.v), "l" (inp.v)
        );

    return ret;
}

inline struct vec256_t sqr_n_cneg_mod_256_189_val(const struct vec256_t sqrt,
                                                  const struct vec256_shared_t inp,
                                                  __local unsigned int scratchpad[]) {

    struct vec256_t ret;

    asm volatile(
        "{\t\n"
        "   .reg.b32    vptr, voff, flag;\t\n"
        "   .reg.b32    in<8>;\t\n"
        "   .reg.b32    sqr<16>;\t\n"
        "\t\n"
        "   ld.u32  in0, [%1+4*0];\t\n"
        "   ld.u32  in1, [%1+4*1];\t\n"
        "   ld.u32  in2, [%1+4*2];\t\n"
        "   ld.u32  in3, [%1+4*3];\t\n"
        "   ld.u32  in4, [%1+4*4];\t\n"
        "   ld.u32  in5, [%1+4*5];\t\n"
        "   ld.u32  in6, [%1+4*6];\t\n"
        "   ld.u32  in7, [%1+4*7];\t\n"
        "    {  // square\t\n"
        "   .reg.b32    odd<14>;\t\n"
        "\t\n"
        "   mul.lo.u32  sqr2, in2, in0;\t\n"
        "   mul.hi.u32  sqr3, in2, in0;\t\n"
        "   mul.lo.u32  sqr4, in4, in0;\t\n"
        "   mul.hi.u32  sqr5, in4, in0;\t\n"
        "   mul.lo.u32  sqr6, in6, in0;\t\n"
        "   mul.hi.u32  sqr7, in6, in0;\t\n"
        "   mul.lo.u32  odd0, in1, in0;\t\n"
        "   mul.hi.u32  odd1, in1, in0;\t\n"
        "   mul.lo.u32  odd2, in3, in0;\t\n"
        "   mul.hi.u32  odd3, in3, in0;\t\n"
        "   mul.lo.u32  odd4, in5, in0;\t\n"
        "   mul.hi.u32  odd5, in5, in0;\t\n"
        "   mul.lo.u32  odd6, in7, in0;\t\n"
        "   mul.hi.u32  odd7, in7, in0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr4, in3, in1, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, in3, in1, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, in5, in1, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, in5, in1, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, in7, in1, 0;\t\n"
        "   madc.hi.u32 sqr9, in7, in1, 0;\t\n"
        "   mad.lo.cc.u32   odd2, in2, in1, odd2;\t\n"
        "   madc.hi.cc.u32  odd3, in2, in1, odd3;\t\n"
        "   madc.lo.cc.u32  odd4, in4, in1, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, in4, in1, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, in6, in1, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, in6, in1, odd7;\t\n"
        "   addc.u32    sqr9, sqr9, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   odd4, in3, in2, odd4;\t\n"
        "   madc.hi.cc.u32  odd5, in3, in2, odd5;\t\n"
        "   madc.lo.cc.u32  odd6, in5, in2, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, in5, in2, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, in7, in2, 0;\t\n"
        "   madc.hi.u32 odd9, in7, in2, 0;\t\n"
        "   mad.lo.cc.u32   sqr6, in4, in2, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, in4, in2, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, in6, in2, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, in6, in2, sqr9;\t\n"
        "   addc.u32    odd9, odd9, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr8, in5, in3, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, in5, in3, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, in7, in3, 0;\t\n"
        "   madc.hi.u32 sqr11, in7, in3, 0;\t\n"
        "   mad.lo.cc.u32   odd6, in4, in3, odd6;\t\n"
        "   madc.hi.cc.u32  odd7, in4, in3, odd7;\t\n"
        "   madc.lo.cc.u32  odd8, in6, in3, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, in6, in3, odd9;\t\n"
        "   addc.u32    sqr11, sqr11, 0;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   odd8, in5, in4, odd8;\t\n"
        "   madc.hi.cc.u32  odd9, in5, in4, odd9;\t\n"
        "   madc.lo.cc.u32  odd10, in7, in4, 0;\t\n"
        "   madc.hi.u32 odd11, in7, in4, 0;\t\n"
        "   mad.lo.cc.u32   sqr10, in6, in4, sqr10;\t\n"
        "   madc.hi.cc.u32  sqr11, in6, in4, sqr11;\t\n"
        "\t\n"
        "   madc.lo.cc.u32  sqr12, in7, in5, 0;\t\n"
        "   madc.hi.u32 sqr13, in7, in5, 0;\t\n"
        "   mad.lo.cc.u32   odd10, in6, in5, odd10;\t\n"
        "   madc.hi.cc.u32  odd11, in6, in5, odd11;\t\n"
        "\t\n"
        "   madc.lo.cc.u32  odd12, in7, in6, 0;\t\n"
        "   madc.hi.u32 odd13, in7, in6, 0;\t\n"
        "\t\n"
        "   add.cc.u32  sqr2, sqr2, odd1;\t\n"
        "   addc.cc.u32 sqr3, sqr3, odd2;\t\n"
        "   addc.cc.u32 sqr4, sqr4, odd3;\t\n"
        "   addc.cc.u32 sqr5, sqr5, odd4;\t\n"
        "   addc.cc.u32 sqr6, sqr6, odd5;\t\n"
        "   addc.cc.u32 sqr7, sqr7, odd6;\t\n"
        "   addc.cc.u32 sqr8, sqr8, odd7;\t\n"
        "   addc.cc.u32 sqr9, sqr9, odd8;\t\n"
        "   addc.cc.u32 sqr10, sqr10, odd9;\t\n"
        "   addc.cc.u32 sqr11, sqr11, odd10;\t\n"
        "   addc.cc.u32 sqr12, sqr12, odd11;\t\n"
        "   addc.cc.u32 sqr13, sqr13, odd12;\t\n"
        "   addc.u32    sqr14, odd13, 0;\t\n"
        "   mov.u32 sqr0, 0;\t\n"
        "   add.cc.u32  sqr1, odd0, odd0;\t\n"
        "   addc.cc.u32 sqr2, sqr2, sqr2;\t\n"
        "   addc.cc.u32 sqr3, sqr3, sqr3;\t\n"
        "   addc.cc.u32 sqr4, sqr4, sqr4;\t\n"
        "   addc.cc.u32 sqr5, sqr5, sqr5;\t\n"
        "   addc.cc.u32 sqr6, sqr6, sqr6;\t\n"
        "   addc.cc.u32 sqr7, sqr7, sqr7;\t\n"
        "   addc.cc.u32 sqr8, sqr8, sqr8;\t\n"
        "   addc.cc.u32 sqr9, sqr9, sqr9;\t\n"
        "   addc.cc.u32 sqr10, sqr10, sqr10;\t\n"
        "   addc.cc.u32 sqr11, sqr11, sqr11;\t\n"
        "   addc.cc.u32 sqr12, sqr12, sqr12;\t\n"
        "   addc.cc.u32 sqr13, sqr13, sqr13;\t\n"
        "   addc.cc.u32 sqr14, sqr14, sqr14;\t\n"
        "   addc.u32    sqr15, 0, 0;\t\n"
        "   mad.lo.cc.u32   sqr0, in0, in0, sqr0;\t\n"
        "   madc.hi.cc.u32  sqr1, in0, in0, sqr1;\t\n"
        "   madc.lo.cc.u32  sqr2, in1, in1, sqr2;\t\n"
        "   madc.hi.cc.u32  sqr3, in1, in1, sqr3;\t\n"
        "   madc.lo.cc.u32  sqr4, in2, in2, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, in2, in2, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, in3, in3, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, in3, in3, sqr7;\t\n"
        "   madc.lo.cc.u32  sqr8, in4, in4, sqr8;\t\n"
        "   madc.hi.cc.u32  sqr9, in4, in4, sqr9;\t\n"
        "   madc.lo.cc.u32  sqr10, in5, in5, sqr10;\t\n"
        "   madc.hi.cc.u32  sqr11, in5, in5, sqr11;\t\n"
        "   madc.lo.cc.u32  sqr12, in6, in6, sqr12;\t\n"
        "   madc.hi.cc.u32  sqr13, in6, in6, sqr13;\t\n"
        "   madc.lo.cc.u32  sqr14, in7, in7, sqr14;\t\n"
        "   madc.hi.cc.u32  sqr15, in7, in7, sqr15;\t\n"
        "    }\t\n"
        "\t\n"
        "    {  // reduce\t\n"
        "   .reg.b32    top;\t\n"
        "\t\n"
        "   mad.lo.cc.u32   sqr0, sqr8, 189, sqr0;\t\n"
        "   madc.hi.cc.u32  sqr1, sqr8, 189, sqr1;\t\n"
        "   madc.lo.cc.u32  sqr2, sqr10, 189, sqr2;\t\n"
        "   madc.hi.cc.u32  sqr3, sqr10, 189, sqr3;\t\n"
        "   madc.lo.cc.u32  sqr4, sqr12, 189, sqr4;\t\n"
        "   madc.hi.cc.u32  sqr5, sqr12, 189, sqr5;\t\n"
        "   madc.lo.cc.u32  sqr6, sqr14, 189, sqr6;\t\n"
        "   madc.hi.cc.u32  sqr7, sqr14, 189, sqr7;\t\n"
        "   addc.u32    top, 0, 0;\t\n"
        "   mul.lo.u32  sqr8, sqr9, 189;\t\n"
        "   mul.hi.u32  sqr9, sqr9, 189;\t\n"
        "   mul.lo.u32  sqr10, sqr11, 189;\t\n"
        "   mul.hi.u32  sqr11, sqr11, 189;\t\n"
        "   mul.lo.u32  sqr12, sqr13, 189;\t\n"
        "   mul.hi.u32  sqr13, sqr13, 189;\t\n"
        "   mul.lo.u32  sqr14, sqr15, 189;\t\n"
        "   mul.hi.u32  sqr15, sqr15, 189;\t\n"
        "   add.cc.u32  sqr1, sqr1, sqr8;\t\n"
        "   addc.cc.u32 sqr2, sqr2, sqr9;\t\n"
        "   addc.cc.u32 sqr3, sqr3, sqr10;\t\n"
        "   addc.cc.u32 sqr4, sqr4, sqr11;\t\n"
        "   addc.cc.u32 sqr5, sqr5, sqr12;\t\n"
        "   addc.cc.u32 sqr6, sqr6, sqr13;\t\n"
        "   addc.cc.u32 sqr7, sqr7, sqr14;\t\n"
        "   addc.u32    top, top, sqr15;\t\n"
        "   mad.lo.cc.u32   sqr0, top, 189, sqr0;\t\n"
        "   addc.cc.u32 sqr1, sqr1, 0;\t\n"
        "   addc.cc.u32 sqr2, sqr2, 0;\t\n"
        "   addc.cc.u32 sqr3, sqr3, 0;\t\n"
        "   addc.cc.u32 sqr4, sqr4, 0;\t\n"
        "   addc.cc.u32 sqr5, sqr5, 0;\t\n"
        "   addc.cc.u32 sqr6, sqr6, 0;\t\n"
        "   addc.cc.u32 sqr7, sqr7, 0;\t\n"
        "   addc.u32    top, 0, 0;\t\n"
        "   mad.lo.u32  sqr0, top, 189, sqr0;\t\n"
        "    }\t\n"
        "\t\n"
        "   mov.u32 vptr, %3;\t\n"
        "   ld.u32  voff, [%2+4];\t\n"
        "   add.u32 vptr, vptr, voff;\t\n"
        "   ld.u32  sqr8, [%2+0];\t\n"
        "   ld.shared.u32   sqr9, [vptr + 4*(9-9)];\t\n"
        "   ld.shared.u32   sqr10, [vptr + 4*(10-9)];\t\n"
        "   ld.shared.u32   sqr11, [vptr + 4*(11-9)];\t\n"
        "   ld.shared.u32   sqr12, [vptr + 4*(12-9)];\t\n"
        "   ld.shared.u32   sqr13, [vptr + 4*(13-9)];\t\n"
        "   ld.shared.u32   sqr14, [vptr + 4*(14-9)];\t\n"
        "   ld.shared.u32   sqr15, [vptr + 4*(15-9)];\t\n"
        "   xor.b32 sqr0, sqr0, sqr8;\t\n"
        "   xor.b32 sqr1, sqr1, sqr9;\t\n"
        "   or.b32  sqr0, sqr0, sqr1;\t\n"
        "   xor.b32 sqr2, sqr2, sqr10;\t\n"
        "   xor.b32 sqr3, sqr3, sqr11;\t\n"
        "   or.b32  sqr2, sqr2, sqr3;\t\n"
        "   xor.b32 sqr4, sqr4, sqr12;\t\n"
        "   xor.b32 sqr5, sqr5, sqr13;\t\n"
        "   or.b32  sqr4, sqr4, sqr5;\t\n"
        "   xor.b32 sqr6, sqr6, sqr14;\t\n"
        "   xor.b32 sqr7, sqr7, sqr15;\t\n"
        "   or.b32  sqr6, sqr6, sqr7;\t\n"
        "   or.b32      sqr0, sqr0, sqr2;\t\n"
        "   or.b32      sqr4, sqr4, sqr6;\t\n"
        "   or.b32      sqr0, sqr0, sqr4;\t\n"
        "   set.ne.u32.u32  sqr0, sqr0, 0;\t\n"
        "   and.b32     sqr0, sqr0, 1;\t\n"
        "\t\n"
        "   and.b32     flag, in0, 1;\t\n"
        "   xor.b32     flag, flag, sqr0;\t\n"
        "    {  // conditional negate\t\n"
        "   .reg.b32    nzero;\t\n"
        "\t\n"
        "   or.b32  nzero, in0, in1;\t\n"
        "   or.b32  nzero, nzero, in2;\t\n"
        "   or.b32  nzero, nzero, in3;\t\n"
        "   or.b32  nzero, nzero, in4;\t\n"
        "   or.b32  nzero, nzero, in5;\t\n"
        "   or.b32  nzero, nzero, in6;\t\n"
        "   or.b32  nzero, nzero, in7;\t\n"
        "   set.ne.u32.u32  nzero, nzero, 0;\t\n"
        "   set.ne.u32.u32  flag, flag, 0;\t\n"
        "   and.b32 flag, flag, nzero;\t\n"
        "   and.b32 nzero, flag, 189;\t\n"
        "   add.cc.u32  in0, in0, nzero;\t\n"
        "   addc.cc.u32 in1, in1, 0;\t\n"
        "   addc.cc.u32 in2, in2, 0;\t\n"
        "   addc.cc.u32 in3, in3, 0;\t\n"
        "   addc.cc.u32 in4, in4, 0;\t\n"
        "   addc.cc.u32 in5, in5, 0;\t\n"
        "   addc.cc.u32 in6, in6, 0;\t\n"
        "   addc.u32    in7, in7, 0;\t\n"
        "   and.b32 nzero, flag, 1;\t\n"
        "   xor.b32 in0, in0, flag;\t\n"
        "   xor.b32 in1, in1, flag;\t\n"
        "   xor.b32 in2, in2, flag;\t\n"
        "   xor.b32 in3, in3, flag;\t\n"
        "   xor.b32 in4, in4, flag;\t\n"
        "   xor.b32 in5, in5, flag;\t\n"
        "   xor.b32 in6, in6, flag;\t\n"
        "   xor.b32 in7, in7, flag;\t\n"
        "   add.cc.u32  in0, in0, nzero;\t\n"
        "   addc.cc.u32 in1, in1, 0;\t\n"
        "   addc.cc.u32 in2, in2, 0;\t\n"
        "   addc.cc.u32 in3, in3, 0;\t\n"
        "   addc.cc.u32 in4, in4, 0;\t\n"
        "   addc.cc.u32 in5, in5, 0;\t\n"
        "   addc.cc.u32 in6, in6, 0;\t\n"
        "   addc.u32    in7, in7, 0;\t\n"
        "    }\t\n"
        "\t\n"
        "   st.u32  [%0+4*0], in0;\t\n"
        "   st.u32  [%0+4*1], in1;\t\n"
        "   st.u32  [%0+4*2], in2;\t\n"
        "   st.u32  [%0+4*3], in3;\t\n"
        "   st.u32  [%0+4*4], in4;\t\n"
        "   st.u32  [%0+4*5], in5;\t\n"
        "   st.u32  [%0+4*6], in6;\t\n"
        "   st.u32  [%0+4*7], in7;\t\n"
        "}"
        :: "l" (ret.v), "l" (sqrt.v), "l" (&inp.v0), "r" (scratchpad)
        );
        
    return ret;
}

inline struct vec256_t xor_256_val(const struct vec256_t a, 
                                   const struct vec256_shared_t b,
                                   __local unsigned int scratchpad[]) {

    struct vec256_t ret;

    asm volatile(
        "{\t\n"
        "   .reg.b32    vptr, voff;\t\n"
        "   .reg.b32    a<8>, b<8>;\t\n"
        "\t\n"
        "   mov.u32 vptr, %4;\t\n"
        "   mov.u32  voff, %3;\t\n"
        "   add.u32 vptr, vptr, voff;\t\n"
        "   mov.u32  b0, %2;\t\n"
        "   ld.u32  a0, [%1+4*0];\t\n"
        "   ld.u32  a1, [%1+4*1];\t\n"
        "   ld.shared.u32   b1, [vptr+4*(1-1)];\t\n"
        "   ld.u32  a2, [%1+4*2];\t\n"
        "   ld.shared.u32   b2, [vptr+4*(2-1)];\t\n"
        "   ld.u32  a3, [%1+4*3];\t\n"
        "   ld.shared.u32   b3, [vptr+4*(3-1)];\t\n"
        "   ld.u32  a4, [%1+4*4];\t\n"
        "   ld.shared.u32   b4, [vptr+4*(4-1)];\t\n"
        "   ld.u32  a5, [%1+4*5];\t\n"
        "   ld.shared.u32   b5, [vptr+4*(5-1)];\t\n"
        "   ld.u32  a6, [%1+4*6];\t\n"
        "   ld.shared.u32   b6, [vptr+4*(6-1)];\t\n"
        "   ld.u32  a7, [%1+4*7];\t\n"
        "   ld.shared.u32   b7, [vptr+4*(7-1)];\t\n"
        "   xor.b32 a0, a0, b0;\t\n"
        "   st.u32  [%0+4*0], a0;\t\n"
        "   xor.b32 a1, a1, b1;\t\n"
        "   st.u32  [%0+4*1], a1;\t\n"
        "   xor.b32 a2, a2, b2;\t\n"
        "   st.u32  [%0+4*2], a2;\t\n"
        "   xor.b32 a3, a3, b3;\t\n"
        "   st.u32  [%0+4*3], a3;\t\n"
        "   xor.b32 a4, a4, b4;\t\n"
        "   st.u32  [%0+4*4], a4;\t\n"
        "   xor.b32 a5, a5, b5;\t\n"
        "   st.u32  [%0+4*5], a5;\t\n"
        "   xor.b32 a6, a6, b6;\t\n"
        "   st.u32  [%0+4*6], a6;\t\n"
        "   xor.b32 a7, a7, b7;\t\n"
        "   st.u32  [%0+4*7], a7;\t\n"
        "}"
        :: "l" (ret.v), "l" (a.v), "r" (b.v0), "r" (b.v_off), "r" (scratchpad)
        );

    return ret;
}

inline bool check_mod_256_189_val(const struct vec256_t a) {

    bool ret;

    asm volatile(
        "{\t\n"
        "   .reg.b32    a<8>, ret;\t\n"
        "\t\n"
        "   ld.u32  a0, [%1+4*0];\t\n"
        "   ld.u32  a1, [%1+4*1];\t\n"
        "   ld.u32  a2, [%1+4*2];\t\n"
        "   ld.u32  a3, [%1+4*3];\t\n"
        "   ld.u32  a4, [%1+4*4];\t\n"
        "   ld.u32  a5, [%1+4*5];\t\n"
        "   ld.u32  a6, [%1+4*6];\t\n"
        "   ld.u32  a7, [%1+4*7];\t\n"
        "   add.cc.u32  a0, a0, 189;\t\n"
        "   addc.cc.u32 a1, a1, 0;\t\n"
        "   addc.cc.u32 a2, a2, 0;\t\n"
        "   addc.cc.u32 a3, a3, 0;\t\n"
        "   addc.cc.u32 a4, a4, 0;\t\n"
        "   addc.cc.u32 a5, a5, 0;\t\n"
        "   addc.cc.u32 a6, a6, 0;\t\n"
        "   addc.cc.u32 a7, a7, 0;\t\n"
        "   addc.u32    ret, 0, 0;\t\n"
        "   st.u32  [%0], ret;\t\n"
        "}"
        :: "l" (&ret), "l" (a.v)
        );

    return ret;
}