use anyhow::Result;
use itertools::Itertools;
use serde::{Deserialize, Serialize};
use strum_macros::Display;

use crate::hcl::{Resource, Value};

#[derive(Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields)]
pub struct SloOuter {
    pub business_unit: BusinessUnit,
    pub env: Environment,
    // TODO should service be a required field here?
    pub service: Option<String>,
    pub slos: Vec<Slo>,
    #[serde(default)]
    tags: Vec<String>,
}

impl SloOuter {
    fn tags(&self) -> Vec<Value> {
        let mut tags = vec![
            self.env.tag(),
            self.business_unit.tag(),
        ];

        if let Some(service) = &self.service {
            let service = format!("service:{}", service.as_str());

            tags.push(service.into());
        }

        for tag in &self.tags {
            tags.push(tag.as_str().into());
        }

        tags.into_iter().unique().collect()
    }
}

#[derive(Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields)]
pub struct Slo {
    short_name: String,
    pub name: String,
    pub threshold: SloThreshold,
    pub slis: Vec<Sli>,
    #[serde(default)]
    tags: Vec<String>,
}

#[derive(Debug, Deserialize, Serialize, PartialEq)]
#[serde(deny_unknown_fields)]
pub enum BusinessUnit {
    #[serde(rename = "devops")]
    Devops,
    #[serde(rename = "figure-tech")]
    FigureTech,
    #[serde(rename = "figurepay")]
    FigurePay,
    #[serde(rename = "lending")]
    Lending,
    #[serde(rename = "rtem")]
    Rtem,
}

impl BusinessUnit {
    fn tag(&self) -> Value {
        let business_unit = match self {
            BusinessUnit::Devops => "devops",
            BusinessUnit::FigureTech => "figure-tech",
            BusinessUnit::FigurePay => "figurepay",
            BusinessUnit::Lending => "lending",
            BusinessUnit::Rtem => "rtem",
        };

        format!("business_unit:{}", business_unit).into()
    }
}

#[derive(Debug, Deserialize, Serialize, PartialEq)]
#[serde(deny_unknown_fields)]
pub enum Environment {
    #[serde(rename = "development")]
    Development,
    #[serde(rename = "figure-pay-test")]
    FigurePayTest,
    #[serde(rename = "figure-pay-prod")]
    FigurePayProd,
    #[serde(rename = "figure-tech")]
    FigureTech,
    #[serde(rename = "figure-tech-test")]
    FigureTechTest,
    #[serde(rename = "pio")]
    Pio,
    #[serde(rename = "pio-test")]
    PioTest,
    #[serde(rename = "production")]
    Production,
}

impl Environment {
    fn name(&self) -> &'static str {
        match self {
            Environment::Development => "development",
            Environment::FigurePayTest => "figure-pay-test",
            Environment::FigurePayProd => "figure-pay-prod",
            Environment::FigureTech => "figure-tech",
            Environment::FigureTechTest => "figure-tech-test",
            Environment::Pio => "pio",
            Environment::PioTest => "pio-test",
            Environment::Production => "production",
        }
    }

    fn tag(&self) -> Value {
        format!("env:{}", self.name()).into()
    }
}

#[derive(Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields)]
pub struct SloThreshold {
    target: f32,
}

impl Slo {
    fn sli_ids(&self) -> Vec<String> {
        self.slis.iter().map(|sli| sli.id()).collect()
    }

    fn short_name(&self) -> String {
        self.short_name.replace('-', "_")
    }

    fn tags(&self, outer: &SloOuter) -> Vec<Value> {
        let mut tags = Vec::default();

        for tag in &self.tags {
            tags.push(tag.as_str().into());
        }

        for sli in &self.slis {
            tags.extend(sli.tags(outer));
        }

        tags.into_iter().unique().collect()
    }

    pub fn resource(&self, outer: &SloOuter) -> Resource {
        let threshold_target = self.threshold.target.to_string();
        let thresholds_7d = vec!(
            ("timeframe", "7d"),
            ("target", &threshold_target),
        );
        let thresholds_30d = vec!(
            ("timeframe", "30d"),
            ("target", &threshold_target),
        );
        let thresholds_90d = vec!(
            ("timeframe", "90d"),
            ("target", &threshold_target),
        );

        Resource {
            _type: String::from("datadog_service_level_objective"),
            name: self.short_name(),
            props: vec![
                ("name", self.name.clone().into()),
                ("type", "monitor".into()),
                // TODO descriptions to everything?
                // ("description", Value::String(slo.description)),
                ("monitor_ids", Value::raw_list(self.sli_ids())),
                ("thresholds", thresholds_7d.into()),
                ("thresholds", thresholds_30d.into()),
                ("thresholds", thresholds_90d.into()),
                ("tags", self.tags(outer).into()),
            ],
        }
    }
}

impl AsRef<Slo> for Slo {
    fn as_ref(&self) -> &Slo {
        self
    }
}

#[derive(Clone, Debug, Deserialize, PartialEq)]
pub struct SliBase {
    short_name: String,
    name: String,
    service: Option<String>,
    resources: Vec<String>,
    threshold: MonitorThreshold,
    message: String,
    #[serde(default)]
    tags: Vec<String>,
}

impl SliBase {
    fn tags(&self) -> Vec<Value> {
        let mut tags = Vec::default();

        if let Some(service) = &self.service {
            let service = format!("service:{}", service.as_str());

            tags.push(service.into());
        }

        for tag in &self.tags {
            tags.push(tag.as_str().into());
        }

        tags.into_iter().unique().collect()
    }

    fn short_name(&self) -> String {
        self.short_name.replace('-', "_")
    }

    pub fn default_from_parent(&mut self, outer: &SloOuter) -> Result<()> {
        crate::utils::maybe_override(&mut self.service, &outer.service, "sli service was omitted and no base service exists")?;

        Ok(())
    }
}

#[derive(Clone, Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields, tag = "type", rename_all = "kebab-case")]
pub enum Sli {
    Latency {
        #[serde(flatten)]
        base: SliBase,
        metric: String,
        percentile: Percentile,
    },
    ErrorRate {
        #[serde(flatten)]
        base: SliBase,
        metric: ErrorRateMetric,
    },
}

impl AsRef<Sli> for Sli {
    fn as_ref(&self) -> &Sli {
        self
    }
}

impl Sli {
    fn id(&self) -> String {
        format!("datadog_monitor.{}.id", self.base().short_name())
    }

    pub fn base_mut(&mut self) -> &mut SliBase {
        match self {
            Sli::Latency { base, .. } => base,
            Sli::ErrorRate { base, .. } => base,
        }
    }

    pub fn base(&self) -> &SliBase {
        match self {
            Sli::Latency { base, .. } => base,
            Sli::ErrorRate { base, .. } => base,
        }
    }

    fn tag(&self) -> Value {
        match self {
            Sli::Latency {..} => "type:latency",
            Sli::ErrorRate {..} => "type:error-rate",
        }.into()
    }

    fn tags(&self, outer: &SloOuter) -> Vec<Value> {
        let mut tags = vec![self.tag()];

        tags.extend(outer.tags());
        tags.extend(self.base().tags());

        tags.into_iter().unique().collect()
    }

    fn filters(&self, outer: &SloOuter) -> String {
        let base = self.base();
        let resources = crate::utils::resource_tags(&base.resources);
        let mut filters = Vec::default();

        filters.push(format!("({})", resources.join(" OR ")));
        filters.push(format!("env:{}", outer.env.name()));

        if let Some(service) = &base.service {
            filters.push(format!("service:{}", service));
        }

        format!("{{{}}}", filters.join(" AND "))
    }

    fn query(&self, outer: &SloOuter) -> String {
        match self {
            Sli::Latency { base, metric, percentile } => {
                format!(
                    "{}:{}:{}{} > {}",
                    percentile.aggregate(),
                    percentile,
                    metric,
                    self.filters(outer),
                    base.threshold.critical,
                )
            },
            Sli::ErrorRate { base, metric } => {
                let numerator = format!(
                    "{}:sum:{}{}.as_count()",
                    metric.aggregate(),
                    metric.numerator,
                    self.filters(outer),
                );
                let denominator = format!(
                    "sum:{}{}.as_count()",
                    metric.denominator,
                    self.filters(outer),
                );

                format!("{} / {} > {}", numerator, denominator, base.threshold.critical)
            }
        }
    }

    pub fn resource(&self, outer: &SloOuter) -> Resource {
        let query = self.query(outer);
        let base = self.base();
        let thresholds = vec!(
            ("warning", base.threshold.warning.to_string()),
            ("critical", base.threshold.critical.to_string()),
        );

        Resource {
            _type: String::from("datadog_monitor"),
            name: base.short_name(),
            props: vec![
                ("name", Value::String(base.name.to_string())),
                ("type", Value::String(String::from("metric alert"))),
                ("message", Value::String(base.message.to_string())),
                ("query", Value::String(query)),
                ("monitor_thresholds", thresholds.into()),
                ("tags", Value::List(self.tags(outer))),
            ],
        }
    }
}

trait Aggregate {
    fn aggregate(&self) -> &'static str;
}

#[derive(Clone, Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields)]
pub struct ErrorRateMetric {
    numerator: String,
    denominator: String,
}

impl Aggregate for ErrorRateMetric {
    fn aggregate(&self) -> &'static str {
        "sum(last_5m)"
    }
}

#[derive(Clone, Debug, Deserialize, Display, PartialEq)]
#[serde(rename_all = "snake_case")]
#[strum(serialize_all = "snake_case")]
pub enum Percentile {
    P50,
    P75,
    P90,
    P95,
    P99,
}

impl Aggregate for Percentile {
    fn aggregate(&self) -> &'static str {
        "percentile(last_5m)"
    }
}

#[derive(Clone, Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields)]
struct MonitorThreshold {
    warning: f32,
    critical: f32,
}


#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn latency_sli_des() {
        let actual = r#"short_name: test-sli-1
name: Test sli 1
service: devops-portal
type: latency
percentile: p95
resources:
- post_/api/v1/deployments
- get_/api/v1/deployments
- post_/api/v1/deployments/
metric: trace.http.request
threshold:
  warning: 0.8
  critical: 1
message: |-
  example message string

  end.
tags:
- tag1
- tag2
"#;
        let actual: Result<Sli, _> = serde_yaml::from_str(&actual);
        let base = SliBase {
            short_name: String::from("test-sli-1"),
            name: String::from("Test sli 1"),
            service: Some(String::from("devops-portal")),
            resources: vec![
                String::from("post_/api/v1/deployments"),
                String::from("get_/api/v1/deployments"),
                String::from("post_/api/v1/deployments/"),
            ],
            threshold: MonitorThreshold {
                warning: 0.8f32,
                critical: 1.0f32,
            },
            message: String::from("example message string\n\nend."),
            tags: vec![String::from("tag1"), String::from("tag2")],
        };
        let expected = Sli::Latency {
            base,
            metric: String::from("trace.http.request"),
            percentile: Percentile::P95,
        };

        assert_eq!(actual.unwrap(), expected);
    }

    #[test]
    fn error_rate_sli_des() {
        let actual = r#"short_name: test-sli-1
name: Test sli 1
service: devops-portal
type: error-rate
metric:
  numerator: trace.http.request.errors
  denominator: trace.http.request.hits
resources:
- "!get_/health/"
threshold:
  warning: 0.025
  critical: 0.05
message: example message string.
tags:
- tag1
- tag2
"#;
        let actual: Result<Sli, _> = serde_yaml::from_str(&actual);
        let base = SliBase {
            short_name: String::from("test-sli-1"),
            name: String::from("Test sli 1"),
            service: Some(String::from("devops-portal")),
            resources: vec![String::from("!get_/health/")],
            threshold: MonitorThreshold {
                warning: 0.025f32,
                critical: 0.05f32,
            },
            message: String::from("example message string."),
            tags: vec![String::from("tag1"), String::from("tag2")],
        };
        let expected = Sli::ErrorRate {
            base,
            metric: ErrorRateMetric {
                numerator: String::from("trace.http.request.errors"),
                denominator: String::from("trace.http.request.hits"),
            },
        };

        assert_eq!(actual.unwrap(), expected);
    }

    #[test]
    fn slo_des() {
        let actual = r#"env: figure-pay-test
business_unit: figurepay
service: top-level-service
slos:
- short_name: test-short
  name: test-long-form-name
  threshold:
    target: 1.0
  slis: []
  tags:
  - tag1
"#;
        let actual: Result<SloOuter, _> = serde_yaml::from_str(&actual);
        let expected = SloOuter {
            business_unit: BusinessUnit::FigurePay,
            env: Environment::FigurePayTest,
            service: Some(String::from("top-level-service")),
            slos: vec![
                Slo {
                    short_name: String::from("test-short"),
                    name: String::from("test-long-form-name"),
                    threshold: SloThreshold { target: 1f32 },
                    slis: Default::default(),
                    tags: vec![String::from("tag1")],
                },
            ],
            tags: Vec::default(),
        };

        assert_eq!(actual.unwrap(), expected);
    }

    #[test]
    fn dedup_tags() {
        let base = SliBase {
            short_name: String::from(""),
            name: String::from(""),
            service: Some(String::from("service-name")),
            resources: vec![String::from("")],
            threshold: MonitorThreshold {
                warning: 0f32,
                critical: 0f32,
            },
            message: String::from(""),
            tags: vec![
                String::from("tag1"),
                String::from("tag2"),
                String::from("tag1"),
                String::from("tag1"),
                String::from("tag1"),
            ],
        };
        let sli = Sli::ErrorRate {
            base,
            metric: ErrorRateMetric {
                numerator: String::from(""),
                denominator: String::from(""),
            },
        };

        let tags = sli.base().tags();

        assert_eq!(tags.len(), 3);
        assert!(tags.contains(&Value::String(String::from("tag1"))));
        assert!(tags.contains(&Value::String(String::from("tag2"))));
        assert!(tags.contains(&Value::String(String::from("service:service-name"))));
    }

    #[test]
    fn complex_tag_conversion() {
        let actual = r#"env: figure-pay-test
business_unit: figurepay
service: top-level-service
tags:
- tag1
- tag-top
slos:
- short_name: test-short
  name: test-long-form-name
  threshold:
    target: 1.0
  tags:
  - tag1
  - tag2
  slis:
  - short_name: test-sli-1
    name: Test sli 1
    type: error-rate
    metric:
      numerator: trace.http.request.errors
      denominator: trace.http.request.hits
    resources:
    - "!get_/health/"
    threshold:
      warning: 0.025
      critical: 0.05
    message: example message string.
    tags:
    - tag1
    - tag3
"#;
        let actual: Result<SloOuter, _> = serde_yaml::from_str(&actual);

        assert!(actual.is_ok());

        let actual = actual.unwrap();
        let slo = actual.slos.first().unwrap();
        let mut sli = slo.slis.first().unwrap().clone();

        sli.base_mut().default_from_parent(&actual).unwrap();

        assert_eq!(
            slo.tags(&actual),
            vec![
                "tag1".into(),
                "tag2".into(),
                "type:error-rate".into(),
                "env:figure-pay-test".into(),
                "business_unit:figurepay".into(),
                "service:top-level-service".into(),
                "tag-top".into(),
                "tag3".into(),
            ],
        );

        assert_eq!(
            sli.tags(&actual),
            vec![
                "type:error-rate".into(),
                "env:figure-pay-test".into(),
                "business_unit:figurepay".into(),
                "service:top-level-service".into(),
                "tag1".into(),
                "tag-top".into(),
                "tag3".into(),
            ],
        );
    }
}
