pub fn maybe_override<T>(field: &mut Option<T>, default: &Option<T>, err_str: &'static str) -> anyhow::Result<()>
    where T: Clone
{
    match (&field, default) {
        (Some(_), _) => Ok(()),
        (None, Some(r)) => {
            *field = Some(r.clone());

            Ok(())
        },
        (None, None) => Err(anyhow::anyhow!(err_str)),
    }
}

pub fn resource_tags(resources: &[String]) -> Vec<String> {
    let mut tags = Vec::default();

    for resource in resources {
        let tag = match resource.strip_prefix('!') {
            Some(rem) => format!("!resource_name:{}", rem),
            None => format!("resource_name:{}", resource),
        };

        tags.push(tag);
    }

    tags
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn tag_formatting() {
        let resources = vec![
            String::from("plain1"),
            String::from("plain2"),
            String::from("!not1"),
            String::from("!not2!"),
            String::from("!not!3!"),
        ];
        let actual = resource_tags(&resources);
        let expected = vec![
            String::from("resource_name:plain1"),
            String::from("resource_name:plain2"),
            String::from("!resource_name:not1"),
            String::from("!resource_name:not2!"),
            String::from("!resource_name:not!3!"),
        ];

        assert_eq!(actual, expected);
    }

    #[test]
    fn override_case_1() {
        let mut actual = Some("left");

        assert!(maybe_override(&mut actual, &None, "err").is_ok());
        assert_eq!(actual, Some("left"));
    }

    #[test]
    fn override_case_2() {
        let mut actual = Some("left");

        assert!(maybe_override(&mut actual, &Some("right"), "err").is_ok());
        assert_eq!(actual, Some("left"));
    }

    #[test]
    fn override_case_3() {
        let mut actual = None;

        assert!(maybe_override(&mut actual, &Some("right"), "err").is_ok());
        assert_eq!(actual, Some("right"));
    }

    #[test]
    fn override_case_4() {
        let mut actual: Option<&'static str> = None;

        assert!(maybe_override(&mut actual, &None, "err").is_err());
    }
}
