mod hcl;
mod utils;
mod yaml;

use anyhow::Result;
use clap::{ArgEnum, Parser};

use hcl::Resource;
use yaml::SloOuter;

// TODO add integration tests
// TODO update example yaml

/// Converts Figure SLO YAML to HCL
#[derive(Parser, Debug)]
#[clap(author, version, about, long_about = None)]
struct Cli {
    /// Path to YAML file
    #[clap(parse(from_os_str))]
    yaml: std::path::PathBuf,

    #[clap(long, arg_enum, group = "mode", default_value_t = Output::Stdout)]
    output: Output,

    #[clap(short, long, group = "mode", help = "Returns a status code based on input parsing")]
    lint: bool,
}

#[derive(Copy, Clone, Debug, Eq, PartialEq, PartialOrd, Ord, ArgEnum)]
enum Output {
    Stdout,
}

fn main() -> Result<()> {
    let cli = Cli::parse();

    if cli.yaml.is_dir() {
        return Err(anyhow::anyhow!("{:?} directories are not currently supported", &cli.yaml))
    } else if cli.yaml.is_file() {
        let contents = std::fs::read_to_string(cli.yaml)?;
        // Naively splits on --- delimiter. Parsing would fail on comments containing ---
        let yamls = contents.split(r"---");

        let mut resources = Vec::<Resource>::default();

        for yaml in yamls {
            let outer: SloOuter = serde_yaml::from_str(yaml)?;

            for slo in &outer.slos {
                for mut sli in slo.slis.clone() {
                    sli.base_mut().default_from_parent(&outer)?;

                    resources.push(sli.as_ref().resource(&outer));
                }

                resources.push(slo.as_ref().resource(&outer));
            }
        }

        // sort resources by type, name so output ordering is deterministic
        resources.sort_by(|a, b| {
            if a._type != b._type {
                a._type.cmp(&b._type)
            } else {
                a.name.cmp(&b.name)
            }
        });

        if cli.lint {
            return Ok(())
        }

        match cli.output {
            Output::Stdout => {
                let mut output = Vec::default();

                for resource in resources {
                    output.push(resource.to_string());
                }

                println!("{}", output.join("\n\n"));
            },
        }

    } else {
        return Err(anyhow::anyhow!("{:?} does not exist", &cli.yaml))
    }

    Ok(())
}
