pub struct Resource {
    pub _type: String,
    pub name: String,
    pub props: Vec<(&'static str, Value)>,
}

impl std::fmt::Display for Resource {
    fn fmt(&self, fmt: &mut std::fmt::Formatter) -> std::fmt::Result {
        fmt.write_str(&format!("resource \"{}\" \"{}\" {{\n", self._type, self.name))?;
        for (k, v) in &self.props {
            match v {
                Value::String(_) | Value::Number(_) | Value::Boolean(_) => {
                    fmt.write_str(&format!("{} = {}\n", &k, &v.to_string()))?;
                },
                Value::RawReference(_) | Value::List(_) => {
                    fmt.write_str(&format!("{} = {}\n", &k, &v))?;
                },
                Value::Map(_) => {
                    fmt.write_str(&format!("{} {{\n", &k))?;
                    fmt.write_str(&v.to_string())?;
                    fmt.write_str("}\n")?;
                },
            };
        }
        fmt.write_str("}")?;

        Ok(())
    }
}

// TODO this can be stricter and we can split out the leaf nodes
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub enum Value {
    String(String),
    RawReference(String),
    Number(u64),
    Boolean(bool),
    List(Vec<Value>),
    Map(Vec<(String, Value)>),
}

impl Value {
    pub fn raw<T>(value: T)-> Self
        where T: Into<String>,
    {
        Value::RawReference(value.into())
    }

    pub fn raw_list<T>(value: Vec<T>)-> Self
        where T: Into<String>,
    {
        let value = value.into_iter().map(Value::raw).collect();

        Value::List(value)
    }
}

impl<S, T> From<Vec<(S, T)>> for Value
    where S: Into<String>,
          T: Into<Value>,
{
    fn from(values: Vec<(S, T)>) -> Self
        where S: Into<String>,
              T: Into<Value>,
    {
        let mut props = Vec::default();

        for (k, v) in values {
            props.push((k.into(), v.into()));
        }

        Value::Map(props)
    }
}

impl<T> From<Vec<T>> for Value
    where T: Into<Value>,
{
    fn from(value: Vec<T>) -> Self {
        let value = value.into_iter()
            .map(|v| v.into())
            .collect();

        Value::List(value)
    }
}

impl From<&str> for Value {
    fn from(value: &str) -> Self {
        Value::String(value.to_owned())
    }
}

impl From<String> for Value {
    fn from(value: String) -> Self {
        Value::String(value)
    }
}

impl From<u64> for Value {
    fn from(value: u64) -> Self {
        Value::Number(value)
    }
}

impl From<bool> for Value {
    fn from(value: bool) -> Self {
        Value::Boolean(value)
    }
}

impl std::fmt::Display for Value {
    fn fmt(&self, fmt: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            Value::String(value) => {
                if value.contains('\n') {
                    fmt.write_str(&format!("<<EOT\n{}\nEOT", &value))?;
                } else {
                    fmt.write_str(&format!("\"{}\"", &value))?;
                }
            },
            Value::RawReference(value) => fmt.write_str(value)?,
            Value::Number(value) => fmt.write_str(&value.to_string())?,
            Value::Boolean(value) => fmt.write_str(&value.to_string())?,
            Value::List(value) => {
                let value = value.iter()
                    .map(|v| v.to_string())
                    .collect::<Vec<_>>()
                    .join(", ");

                fmt.write_str(&format!("[{}]", value))?
            },
            Value::Map(value) => {
                for (k, v) in value {
                    fmt.write_str(&format!("{} = {}\n", k, v))?
                }
            },
        }

        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn simple_serialization() {
        let mut props = Vec::new();
        props.push(("key", "value".into()));

        let res = Resource {
            _type: String::from("resource_type"),
            name: String::from("resource_name"),
            props,
        };

        assert_eq!(
            res.to_string(),
            r#"resource "resource_type" "resource_name" {
key = "value"
}"#,
        );
    }

    #[test]
    fn monitor_serialization() {
        let mut props = Vec::new();
        props.push(("name", "Name for monitor foo".into()));
        props.push(("type", "metric alert".into()));
        props.push(("message", "Monitor triggered.\n\nNotify: @hipchat-channel".into()));
        props.push(("escalation_message", "Escalation message @pagerduty".into()));
        props.push(("query", "avg(last_1h):avg:aws.ec2.cpu{environment:foo,host:foo} by {host} > 4".into()));
        props.push((
            "monitor_thresholds",
            Value::Map(vec![
                (String::from("warning"), Value::Number(2u64)),
                (String::from("warning_recovery"), Value::Number(1u64)),
                (String::from("critical"), Value::Number(4u64)),
                (String::from("critical_recovery"), Value::Number(3u64)),
            ]),
        ));
        props.push(("notify_no_data", Value::Boolean(false)));
        props.push(("renotify_interval", Value::Number(60u64)));
        props.push(("notify_audit", Value::Boolean(false)));
        props.push(("include_tags", Value::Boolean(true)));
        props.push(("tags", vec!["foo:bar", "baz"].into()));

        let res = Resource {
            _type: String::from("datadog_monitor"),
            name: String::from("foo"),
            props,
        };

        assert_eq!(
            res.to_string(),
            r#"resource "datadog_monitor" "foo" {
name = "Name for monitor foo"
type = "metric alert"
message = <<EOT
Monitor triggered.

Notify: @hipchat-channel
EOT
escalation_message = "Escalation message @pagerduty"
query = "avg(last_1h):avg:aws.ec2.cpu{environment:foo,host:foo} by {host} > 4"
monitor_thresholds {
warning = 2
warning_recovery = 1
critical = 4
critical_recovery = 3
}
notify_no_data = false
renotify_interval = 60
notify_audit = false
include_tags = true
tags = ["foo:bar", "baz"]
}"#,
        );
    }
}
