use serde::Deserialize;
use strum_macros::Display;

use crate::hcl::{Resource, Value};

#[derive(Debug, Deserialize)]
#[serde(deny_unknown_fields)]
pub struct SloOuter {
    pub business_unit: String,
    pub env: String,
    pub slos: Vec<Slo>,
}

#[derive(Debug, Deserialize)]
#[serde(deny_unknown_fields)]
pub struct Slo {
    short_name: String,
    pub name: String,
    pub slis: Vec<Sli>,
}

impl Slo {
    fn sli_ids(&self) -> Vec<String> {
        self.slis.iter().map(|sli| sli.id()).collect()
    }

    // TODO uniquify this
    fn tags(&self) -> Vec<Value> {
        let mut tags = Vec::default();

        for sli in &self.slis {
            tags.extend(sli.base().tags());
        }

        tags
    }
}

impl AsRef<Slo> for Slo {
    fn as_ref(&self) -> &Slo {
        self
    }
}

impl From<&Slo> for Resource {
    fn from(slo: &Slo) -> Self {
        Self {
            _type: String::from("datadog_service_level_objective"),
            name: slo.short_name.clone(),
            props: vec![
                ("name", slo.name.clone().into()),
                ("type", "monitor".into()),
                // TODO descriptions to everything?
                // ("description", Value::String(slo.description)),
                ("monitor_ids", Value::raw_list(slo.sli_ids())),
                // TODO monitor thresholds
                ("tags", slo.tags().into()),
            ],
        }
    }
}

#[derive(Debug, Deserialize, PartialEq)]
pub struct SliBase {
    short_name: String,
    name: String,
    service: String,
    resources: Vec<String>,
    threshold: Threshold,
    message: String,
    tags: Vec<String>,
}

impl SliBase {
    // TODO uniquify this
    fn tags(&self) -> Vec<Value> {
        let mut tags = Vec::default();

        for tag in &self.tags {
            tags.push(tag.as_str().into());
        }

        tags
    }
}

#[derive(Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields, tag = "type", rename_all = "kebab-case")]
pub enum Sli {
    Latency {
        #[serde(flatten)]
        base: SliBase,
        metric: String,
        percentile: Percentile,
    },
    ErrorRate {
        #[serde(flatten)]
        base: SliBase,
        metric: ErrorRateMetric,
    },
}

impl AsRef<Sli> for Sli {
    fn as_ref(&self) -> &Sli {
        self
    }
}

impl Sli {
    fn id(&self) -> String {
        format!("datadog_monitor.{}.id", self.base().short_name)
    }

    fn base(&self) -> &SliBase {
        match self {
            Sli::Latency { base, .. } => base,
            Sli::ErrorRate { base, .. } => base,
        }
    }

    fn filters(&self) -> String {
        let base = self.base();
        let resources = base.resources
            .iter()
            .map(|s| format!("resource_name:{}", s))
            .collect::<Vec<String>>();
        // TODO this currently does resource_name:!<resource>
        // this needs to be !resource_name:<resource>
        let mut filters = Vec::new();
        filters.push(format!("({})", resources.join(" OR ")));
        filters.push(format!("service:{}", base.service));
        // TODO add env
        // filters.push(&base.service);

        format!("{{{}}}", filters.join(" AND "))
    }

    fn query(&self) -> String {
        match self {
            Sli::Latency { base, metric, percentile } => {
                format!(
                    "{}:{}:{}{} > {}",
                    percentile.aggregate(),
                    percentile,
                    metric,
                    self.filters(),
                    base.threshold.error,
                )
            },
            Sli::ErrorRate { base, metric } => {
                let numerator = format!(
                    "{}:sum:{}{}.as_count()",
                    metric.aggregate(),
                    metric.numerator,
                    self.filters(),
                );
                let denominator = format!(
                    "sum:{}{}.as_count()",
                    metric.denominator,
                    self.filters(),
                );

                format!("{} / {} > {}", numerator, denominator, base.threshold.error)
            }
        }
    }
}

trait Aggregate {
    fn aggregate(&self) -> &'static str;
}

#[derive(Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields)]
struct ErrorRateMetric {
    numerator: String,
    denominator: String,
}

impl Aggregate for ErrorRateMetric {
    fn aggregate(&self) -> &'static str {
        "sum(last_5m)"
    }
}

#[derive(Debug, Deserialize, Display, PartialEq)]
#[serde(rename_all = "snake_case")]
#[strum(serialize_all = "snake_case")]
enum Percentile {
    P50,
    P75,
    P90,
    P95,
    P99,
}

impl Aggregate for Percentile {
    fn aggregate(&self) -> &'static str {
        "percentile(last_5m)"
    }
}

#[derive(Debug, Deserialize, PartialEq)]
#[serde(deny_unknown_fields)]
struct Threshold {
    warn: f32,
    error: f32,
}

impl From<&Sli> for Resource {
    fn from(sli: &Sli) -> Self {
        let query = sli.query();
        let base = sli.base();
        let tags = base.tags
            .iter()
            .map(|tag| Value::String(tag.clone()))
            .collect();

        // TODO fix clones
        Self {
            _type: String::from("datadog_monitor"),
            name: base.short_name.clone(),
            props: vec![
                ("name", Value::String(base.name.clone())),
                ("type", Value::String(String::from("metric alert"))),
                ("message", Value::String(base.message.clone())),
                ("query", Value::String(query)),
                // TODO monitor thresholds
                ("tags", Value::List(tags)),
            ],
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn latency_sli_des() {
        let actual = r#"short_name: test-sli-1
name: Test sli 1
service: devops-portal
type: latency
percentile: p95
resources:
- post_/api/v1/deployments
- get_/api/v1/deployments
- post_/api/v1/deployments/
metric: trace.http.request
threshold:
  warn: 0.8
  error: 1
message: |-
  example message string

  end.
tags:
- tag1
- tag2
"#;
        let actual: Result<Sli, _> = serde_yaml::from_str(&actual);
        let base = SliBase {
            short_name: String::from("test-sli-1"),
            name: String::from("Test sli 1"),
            service: String::from("devops-portal"),
            resources: vec![
                String::from("post_/api/v1/deployments"),
                String::from("get_/api/v1/deployments"),
                String::from("post_/api/v1/deployments/"),
            ],
            threshold: Threshold {
                warn: 0.8f32,
                error: 1.0f32,
            },
            message: String::from("example message string\n\nend."),
            tags: vec![String::from("tag1"), String::from("tag2")],
        };
        let expected = Sli::Latency {
            base,
            metric: String::from("trace.http.request"),
            percentile: Percentile::P95,
        };

        assert_eq!(actual.unwrap(), expected);
    }

    #[test]
    fn error_rate_sli_des() {
        let actual = r#"short_name: test-sli-1
name: Test sli 1
service: devops-portal
type: error-rate
metric:
  numerator: trace.http.request.errors
  denominator: trace.http.request.hits
resources:
- "!get_/health/"
threshold:
  warn: 0.025
  error: 0.05
message: example message string.
tags:
- tag1
- tag2
"#;
        let actual: Result<Sli, _> = serde_yaml::from_str(&actual);
        let base = SliBase {
            short_name: String::from("test-sli-1"),
            name: String::from("Test sli 1"),
            service: String::from("devops-portal"),
            resources: vec![String::from("!get_/health/")],
            threshold: Threshold {
                warn: 0.025f32,
                error: 0.05f32,
            },
            message: String::from("example message string."),
            tags: vec![String::from("tag1"), String::from("tag2")],
        };
        let expected = Sli::ErrorRate {
            base,
            metric: ErrorRateMetric {
                numerator: String::from("trace.http.request.errors"),
                denominator: String::from("trace.http.request.hits"),
            },
        };

        assert_eq!(actual.unwrap(), expected);
    }
}
