use crate::*;
use std::{io::Cursor, str};

#[test]
fn test_run() {
    assert!(run(Cli {
        file: Some(PathBuf::from("/dev/null")),
        include: "1".to_string(),
        ..Default::default()
    })
    .is_ok());
}

#[test]
fn test_slit_include() {
    struct Test<'a> {
        input: &'a str,
        include: &'a str,
        output: &'a str,
    }

    for case in vec![
        Test {
            input: "a",
            include: "1",
            output: "a\n",
        },
        Test {
            input: "a\n",
            include: "1",
            output: "a\n",
        },
        Test {
            input: "a b c",
            include: "1,3",
            output: "a\tc\n",
        },
        Test {
            input: "a b c\nd e\tf",
            include: "1,3",
            output: "a\tc\nd\tf\n",
        },
    ] {
        // Set up an output container.
        let mut out = Cursor::new(vec![]);

        // Run the action.
        slit(
            Box::new(Cursor::new(case.input.as_bytes())),
            &mut out,
            Cli {
                include: case.include.to_string(),
                ..Default::default()
            },
        )
        .unwrap();
        assert_eq!(str::from_utf8(&out.into_inner()).unwrap(), case.output);
    }
}

#[test]
fn test_slit_exclude() {
    struct Test<'a> {
        input: &'a str,
        include: &'a str,
        exclude: &'a str,
        output: &'a str,
    }

    for case in vec![
        Test {
            input: "a",
            include: "1",
            exclude: "1",
            output: "\n",
        },
        Test {
            input: "a\n",
            include: "1",
            exclude: "2",
            output: "a\n",
        },
        Test {
            input: "a b c",
            include: "1,3",
            exclude: "1",
            output: "c\n",
        },
        Test {
            input: "a b c\nd e\tf",
            include: "1,3",
            exclude: "2,4,1",
            output: "c\nf\n",
        },
    ] {
        // Set up an output container.
        let mut out = Cursor::new(vec![]);

        // Run the action.
        slit(
            Box::new(Cursor::new(case.input.as_bytes())),
            &mut out,
            Cli {
                include: case.include.to_string(),
                exclude: Some(case.exclude.to_string()),
                ..Default::default()
            },
        )
        .unwrap();
        assert_eq!(str::from_utf8(&out.into_inner()).unwrap(), case.output);
    }
}

#[test]
fn test_slit_output_delimiter() {
    struct Test<'a> {
        input: &'a str,
        include: &'a str,
        odelim: &'a str,
        output: &'a str,
    }

    for case in vec![
        Test {
            input: "a",
            include: "1",
            odelim: ":",
            output: "a\n",
        },
        Test {
            input: "a\n",
            include: "1",
            odelim: ":",
            output: "a\n",
        },
        Test {
            input: "a b c",
            include: "1,3",
            odelim: ":",
            output: "a:c\n",
        },
        Test {
            input: "a b c\nd e\tf",
            include: "1,3",
            odelim: ":::",
            output: "a:::c\nd:::f\n",
        },
    ] {
        // Set up an output container.
        let mut out = Cursor::new(vec![]);

        // Run the action.
        slit(
            Box::new(Cursor::new(case.input.as_bytes())),
            &mut out,
            Cli {
                include: case.include.to_string(),
                output_delimiter: Some(case.odelim.to_string()),
                ..Default::default()
            },
        )
        .unwrap();
        assert_eq!(str::from_utf8(&out.into_inner()).unwrap(), case.output);
    }
}
