use anyhow::{format_err, Result};
use jacklog::debug;
use std::{
    io::{self, BufRead, Write},
    path::PathBuf,
};
use structopt::StructOpt;

#[cfg(test)]
mod tests;

/// slit is like `cut`, but (slightly) smarter.
///
/// It also is easier to use, because it has fewer options that most people
/// want most of the time. If you're a `cut` whiz, or your last name is Aho,
/// Weinberger or Kernighan, this probably isn't for you.
///
/// # Differences from `cut`
///
/// For the most part, slit is like cut but more convenient defaults and
/// additional functionality. However, there are a few potential gotchas if
/// migrating to it:
///
/// * There is no -s, --only-delimited option. By default, slit will not print
///   any lines NOT containing the delimiter. This is normally what you want,
///   because otherwise you don't really know what you're going to get. Pass
///   --print-undelimited to get the default cut behavior of printing the whole
///   line if it has no delimiters.
///
/// Pass `<file>` to read from a file, otherwise slit will read from STDIN.
#[derive(Debug, StructOpt, Default)]
#[structopt(setting = structopt::clap::AppSettings::ColoredHelp)]
#[structopt(rename_all = "kebab-case")]
struct Cli {
    /// Select only these fields; also print any line that contains no
    /// delimiter character, unless the -s option is specified.
    #[structopt(short = "f", long = "fields")]
    include: String,

    /// Exclude only these fields; also print any line that contains no
    /// delimiter character, unless the -s option is specified. Takes precedence
    /// over included fields selected with -f, --fields.
    #[structopt(short = "F", long)]
    exclude: Option<String>,

    /// Also print lines not containing the delimited character.
    #[structopt(long)]
    print_undelimited: bool,

    /// Number of lines to skip, from the beginning of the input.
    /// TODO: Make this accept a negative number to skip at the end.
    #[structopt(short = "n", long, default_value = "0")]
    skip: usize,

    /// Delimiter to use instead of whitespace to split lines on when chunking
    /// them up into fields.
    #[structopt(short, long)]
    delimiter: Option<String>,

    /// Delimiter to use when printing the line, instead of the input
    /// delimiter.  Can be useful to transform a separated line into a line
    /// with a new separator.  them up into fields.
    #[structopt(short = "D", long)]
    output_delimiter: Option<String>,

    /// Directory to use as the root of a new tmux session. Pass the flag
    /// multiple times to increase verbosity, up to a maximum of 3.
    #[structopt(short, long, parse(from_occurrences))]
    verbose: usize,

    /// Sum the values from a file.
    file: Option<PathBuf>,
}

fn main() -> Result<()> {
    // Parse options passed at invocation.
    let opts = Cli::from_args();

    // Set up logging, based on how many -v flags were passed.
    jacklog::init(Some(match opts.verbose {
        0 => &"warn",
        1 => &"info",
        2 => &"debug",
        _ => &"trace",
    }))?;
    debug!("{:?}", &opts);

    // Execute the program based on the input.
    run(opts)?;

    // If we got this far, everything must have worked correctly.
    Ok(())
}

/// Actually figure out what we're reading, open a `BufRead` to it, and call
/// `slit()`. Mostly broken out from `main()` to facilitate testing.
fn run(opts: Cli) -> Result<()> {
    // Get a handle to STDIN, in case we need it.
    // TODO: Figure out how to avoid having to do this.
    let s = io::stdin();

    // TODO: Allow passing a path to allow writing directly to a file.
    let o = io::stdout();

    // Get an iterator over _either_ stdin _or_ a file reader. Box up the
    // iterator so we can leverage the BufRead trait when we actually iterate,
    // and not worry about where the data came from.
    let reader: Box<dyn BufRead> = if let Some(p) = &opts.file {
        let f = std::fs::File::open(p)?;
        Box::new(std::io::BufReader::new(f))
    } else {
        Box::new(s.lock())
    };

    slit(reader, &mut Box::new(o.lock()), opts)?;

    Ok(())
}

/// Read each line from a streaming `BufReader`, and call the line parser on
/// it.
///
/// We specify an explicit lifetime so that we can accept inputs like
/// `Stdin.lock()` in our box.  See
/// [https://users.rust-lang.org/t/newbie-lifetime-issue-involving-box-read-stdin-lock/4106/2](https://users.rust-lang.org/t/newbie-lifetime-issue-involving-box-read-stdin-lock/4106/2)
/// for details.
///
/// Returns the sum total.
fn slit<'a, T: BufRead + 'a, U: Write + 'a>(reader: T, writer: &mut U, opts: Cli) -> Result<()> {
    // Parse out the fields we want to print.
    let include: Vec<usize> = opts
        .include
        .split(',')
        .map(|s| s.parse().unwrap())
        .collect();

    // Parse out the fields we _don't_ want to print.
    let exclude: Vec<usize> = opts
        .exclude
        .map(|s| s.split(',').map(|s| s.parse().unwrap()).collect())
        .unwrap_or_default();

    let output_delimiter = if let Some(d) = &opts.output_delimiter {
        d.as_bytes()
    } else if let Some(d) = &opts.delimiter {
        d.as_bytes()
    } else {
        "\t".as_bytes()
    };

    // Get an enumerator over the lines.
    let mut lines = reader.lines();

    // Skip any lines at the beginning, if requested.
    for _ in 0..opts.skip {
        lines
            .next()
            .ok_or_else(|| format_err!("tried to skip more lines than exist"))??;
    }

    // Go through each line from stdin as it comes through and figure out
    // whether we need to skip it, whether it's included, and whether it's
    // excluded.
    for (i, line) in lines.enumerate() {
        // Parse the line into fields. The map_or_else takes a
        // lambda as the default "or else" case, which in this case is
        // splitting on whitespace. The second lambda takes the delimiter, and
        // splits on that.
        let line = line?;
        let fields: Vec<&str> = opts.delimiter.as_ref().map_or_else(
            // Default case, if no delimiter was passed.
            || line.split(char::is_whitespace).filter(|s| !s.is_empty()).collect(),
            // Splitting using the passed delimiter.
            |d| line.split(d).collect(),
        );

        // Now for each included field, print it unless it's also excluded.
        let mut first = true;
        for inc in &include {
            // Skip this included field if it's also excluded.
            if exclude.contains(inc) {
                continue;
            }

            // Otherwise, this must have been included, but not excluded, and
            // we can print it.

            // Write a separator unless this is our first field of the line.
            if !first {
                writer.write_all(output_delimiter)?;
            }
            writer.write_all(
                // The include/exclude args are 1-indexed, so we need to
                // subtract 1 to get the right zero-indexed field position.
                fields
                    .get(*inc - 1)
                    .ok_or_else(|| format_err!("line: {}, field: {}: missing field", i+1, inc))?
                    .as_bytes(),
            )?;

            // Indicate that we've written one record, and can start writing
            // separators.
            first = false;
        }

        // End with a newline, so we preserve each line on its own line.
        writer.write_all(b"\n")?;
    }

    Ok(())
}
