//! # Section Block
//!
//! _[slack api docs 🔗]_
//!
//! Available in surfaces:
//!  - [modals 🔗]
//!  - [messages 🔗]
//!  - [home tabs 🔗]
//!
//! A `section` is one of the most flexible blocks available -
//! it can be used as a simple text block,
//! in combination with text fields,
//! or side-by-side with any of the available [block elements 🔗]
//!
//! [slack api docs 🔗]: https://api.slack.com/reference/block-kit/blocks#section
//! [modals 🔗]: https://api.slack.com/surfaces/modals
//! [messages 🔗]: https://api.slack.com/surfaces/messages
//! [home tabs 🔗]: https://api.slack.com/surfaces/tabs
//! [block elements 🔗]: https://api.slack.com/reference/messaging/block-elements

use std::borrow::Cow;

use serde::{Deserialize, Serialize};
use validator::Validate;

use crate::{compose::text, elems::BlockElement, val_helpr::ValidationResult};

/// # Section Block
///
/// _[slack api docs 🔗]_
///
/// Available in surfaces:
///  - [modals 🔗]
///  - [messages 🔗]
///  - [home tabs 🔗]
///
/// A `section` is one of the most flexible blocks available -
/// it can be used as a simple text block,
/// in combination with text fields,
/// or side-by-side with any of the available [block elements 🔗]
///
/// [slack api docs 🔗]: https://api.slack.com/reference/block-kit/blocks#section
/// [modals 🔗]: https://api.slack.com/surfaces/modals
/// [messages 🔗]: https://api.slack.com/surfaces/messages
/// [home tabs 🔗]: https://api.slack.com/surfaces/tabs
/// [block elements 🔗]: https://api.slack.com/reference/messaging/block-elements
#[derive(Clone, Debug, Deserialize, Hash, PartialEq, Serialize, Validate)]
pub struct Section<'a> {
  #[serde(skip_serializing_if = "Option::is_none")]
  #[validate(custom = "validate::fields")]
  fields: Option<Cow<'a, [text::Text]>>,

  #[serde(skip_serializing_if = "Option::is_none")]
  #[validate(custom = "validate::text")]
  text: Option<text::Text>,

  #[serde(skip_serializing_if = "Option::is_none")]
  #[validate(custom = "validate::block_id")]
  block_id: Option<Cow<'a, str>>,

  /// One of the available [element objects 🔗][element_objects].
  ///
  /// [element_objects]: https://api.slack.com/reference/messaging/block-elements
  #[serde(skip_serializing_if = "Option::is_none")]
  accessory: Option<BlockElement<'a>>,
}

impl<'a> Section<'a> {
  /// Build a new section block
  ///
  /// For example, see `blocks::section::build::SectionBuilder`.
  pub fn builder() -> build::SectionBuilderInit<'a> {
    build::SectionBuilderInit::new()
  }

  /// Construct a Section block from a collection of text objects
  ///
  /// # Arguments
  /// - `fields` - A collection of [text objects 🔗].
  ///     Any text objects included with fields will be
  ///     rendered in a compact format that allows for
  ///     2 columns of side-by-side text.
  ///     Maximum number of items is 10.
  ///     Maximum length for the text in each item is 2000 characters.
  ///
  /// [text objects 🔗]: https://api.slack.com/reference/messaging/composition-objects#text
  ///
  /// # Errors
  /// Doesn't error. To validate your model against the length requirements,
  /// use the `validate` method.
  ///
  /// # Example
  /// ```
  /// use slack_blocks::{blocks, compose::text};
  ///
  /// # use std::error::Error;
  /// # pub fn main() -> Result<(), Box<dyn Error>> {
  /// let fields = vec!["Left column", "Right column"].into_iter()
  ///                                                 .map(|s: &str| -> text::Text {
  ///                                                   text::Plain::from(s).into()
  ///                                                 })
  ///                                                 .collect::<Vec<_>>();
  ///
  /// let block = blocks::Section::from_fields(&fields);
  ///
  /// // < send to slack API >
  /// # Ok(())
  /// # }
  /// ```
  pub fn from_fields<I>(fields: I) -> Self
    where I: Into<Cow<'a, [text::Text]>>
  {
    let fields = Some(fields.into());

    Self { fields,
           text: None,
           block_id: None,
           accessory: None }
  }

  /// Construct a Section block from a text object
  ///
  /// # Arguments
  /// - `text` - The text for the block, in the form of a [text object 🔗].
  ///     Maximum length for the text in this field is 3000 characters.
  ///
  /// [text object 🔗]: https://api.slack.com/reference/messaging/composition-objects#text
  ///
  /// # Errors
  /// Doesn't error. To validate your model against the length requirements,
  /// use the `validate` method.
  ///
  /// # Example
  /// ```
  /// use slack_blocks::{blocks, compose::text};
  ///
  /// let block = blocks::Section::from_text(text::Plain::from("I am a section!"));
  ///
  /// // < send to slack API >
  /// ```
  pub fn from_text(text: impl Into<text::Text>) -> Self {
    Self { text: Some(text.into()),
           fields: None,
           block_id: None,
           accessory: None }
  }

  /// Set a unique `block_id` to identify this instance of an Section Block.
  ///
  /// # Arguments
  ///
  /// - `block_id` - A string acting as a unique identifier for a block.
  ///     You can use this `block_id` when you receive an interaction
  ///     payload to [identify the source of the action 🔗].
  ///     If not specified, one will be generated.
  ///     Maximum length for this field is 255 characters.
  ///     `block_id` should be unique for each message and each iteration of a message.
  ///     If a message is updated, use a new `block_id`.
  ///
  /// [identify the source of the action 🔗]: https://api.slack.com/interactivity/handling#payloads
  pub fn with_block_id(mut self, block_id: impl Into<Cow<'a, str>>) -> Self {
    self.block_id = Some(block_id.into());
    self
  }

  /// Validate that this Section block agrees with Slack's model requirements
  ///
  /// # Errors
  /// - If `from_fields` was called with more than 10 fields,
  ///     or one of the fields contains text longer than
  ///     2000 chars
  /// - If `from_fields` was called with one of the fields
  ///     containing text longer than 2000 chars
  /// - If `from_text` was called with text longer than
  ///     3000 chars
  /// - If `with_block_id` was called with a block id longer
  ///     than 255 chars
  ///
  /// # Example
  /// ```
  /// use slack_blocks::{blocks, compose::text};
  ///
  /// let long_string = std::iter::repeat(' ').take(256).collect::<String>();
  ///
  /// let block = blocks::Section
  ///     ::from_text(text::Plain::from("file_id"))
  ///     .with_block_id(long_string);
  ///
  /// assert_eq!(true, matches!(block.validate(), Err(_)));
  ///
  /// // < send to slack API >
  /// ```
  pub fn validate(&self) -> ValidationResult {
    Validate::validate(self)
  }
}

/// Section block builder
pub mod build {
  use std::marker::PhantomData;

  use super::*;
  use crate::build::*;

  /// Compile-time markers for builder methods
  #[allow(non_camel_case_types)]
  pub mod method {
    /// SectionBuilder.text
    #[derive(Clone, Copy, Debug)]
    pub struct text;
  }

  /// Initial state for `SectionBuilder`
  pub type SectionBuilderInit<'a> =
    SectionBuilder<'a, RequiredMethodNotCalled<method::text>>;

  /// Build an Section block
  ///
  /// Allows you to construct safely, with compile-time checks
  /// on required setter methods.
  ///
  /// # Required Methods
  /// `SectionBuilder::build()` is only available if these methods have been called:
  ///  - `element`
  ///
  /// # Example
  /// ```
  /// use slack_blocks::{blocks::Section, elems::Image, text::ToSlackPlaintext};
  ///
  /// let block =
  ///   Section::builder().text("foo".plaintext())
  ///                     .accessory(Image::builder().image_url("foo.png")
  ///                                                .alt_text("pic of foo")
  ///                                                .build())
  ///                     .build();
  /// ```
  #[derive(Debug)]
  pub struct SectionBuilder<'a, Text> {
    accessory: Option<BlockElement<'a>>,
    text: Option<text::Text>,
    fields: Option<Vec<text::Text>>,
    block_id: Option<Cow<'a, str>>,
    state: PhantomData<Text>,
  }

  impl<'a, E> SectionBuilder<'a, E> {
    /// Create a new SectionBuilder
    pub fn new() -> Self {
      Self { accessory: None,
             text: None,
             fields: None,
             block_id: None,
             state: PhantomData::<_> }
    }

    /// Set `accessory` (Optional)
    pub fn accessory<B>(mut self, acc: B) -> Self
      where B: Into<BlockElement<'a>>
    {
      self.accessory = Some(acc.into());
      self
    }

    /// Add `text` (**Required**, can be called many times)
    ///
    /// One or many [text objects 🔗] to populate the block with.
    ///
    /// # If called once:
    /// Sets the `text` field of the section block.
    ///
    /// Maximum length for the text in this field is 3000 characters.
    ///
    /// # If called multiple times:
    /// Sets the `fields` field of the section block.
    ///
    /// Fields will be rendered in a compact format that allows for
    /// 2 columns of side-by-side text.
    /// Maximum number of items is 10.
    ///
    /// Maximum length for the text in each item is 2000 characters.
    ///
    /// [text objects 🔗]: https://api.slack.com/reference/messaging/composition-objects#text
    pub fn text<T>(self, text: T) -> SectionBuilder<'a, Set<method::text>>
      where T: Into<text::Text>
    {
      let (text, fields) = match (self.text, self.fields) {
        | (Some(t), None) => (None, Some(vec![t, text.into()])),
        | (None, None) => (Some(text.into()), None),
        | (None, Some(mut fs)) => {
          fs.push(text.into());
          (None, Some(fs))
        },
        | (Some(_), Some(_)) => {
          unreachable!("fields and text should never both be set.")
        },
      };

      SectionBuilder { accessory: self.accessory,
                       text,
                       fields,
                       block_id: self.block_id,
                       state: PhantomData::<_> }
    }

    /// Alias of `text` for XML macros, allowing fields
    /// to be used as child elements.
    ///
    /// ```
    /// use mox::mox;
    /// use slack_blocks::{mox::*, text};
    ///
    /// let as_attr = mox! {
    ///   <section_block text={text::Plain::from("Foo")} />
    /// };
    ///
    /// let as_child = mox! {
    ///   <section_block>
    ///     <text kind=plain>"Foo"</text>
    ///   </section_block>
    /// };
    ///
    /// assert_eq!(as_attr, as_child);
    /// ```
    #[cfg(feature = "xml")]
    #[cfg_attr(docsrs, doc(cfg(feature = "xml")))]
    pub fn child<T>(self, text: T) -> SectionBuilder<'a, Set<method::text>>
      where T: Into<text::Text>
    {
      self.text(text)
    }

    /// Set `block_id` (Optional)
    ///
    /// A string acting as a unique identifier for a block.
    ///
    /// You can use this `block_id` when you receive an interaction payload
    /// to [identify the source of the action 🔗].
    ///
    /// If not specified, a `block_id` will be generated.
    ///
    /// Maximum length for this field is 255 characters.
    ///
    /// [identify the source of the action 🔗]: https://api.slack.com/interactivity/handling#payloads
    pub fn block_id<S>(mut self, block_id: S) -> Self
      where S: Into<Cow<'a, str>>
    {
      self.block_id = Some(block_id.into());
      self
    }
  }

  impl<'a> SectionBuilder<'a, Set<method::text>> {
    /// All done building, now give me a darn actions block!
    ///
    /// > `no method name 'build' found for struct 'SectionBuilder<...>'`?
    /// Make sure all required setter methods have been called. See docs for `SectionBuilder`.
    ///
    /// ```compile_fail
    /// use slack_blocks::blocks::Section;
    ///
    /// let foo = Section::builder().build(); // Won't compile!
    /// ```
    ///
    /// ```
    /// use slack_blocks::{blocks::Section,
    ///                    compose::text::ToSlackPlaintext,
    ///                    elems::Image};
    ///
    /// let block =
    ///   Section::builder().text("foo".plaintext())
    ///                     .accessory(Image::builder().image_url("foo.png")
    ///                                                .alt_text("pic of foo")
    ///                                                .build())
    ///                     .build();
    /// ```
    pub fn build(self) -> Section<'a> {
      Section { text: self.text,
                fields: self.fields.map(|fs| fs.into()),
                accessory: self.accessory,
                block_id: self.block_id }
    }
  }
}
mod validate {
  use super::*;
  use crate::{compose::text,
              val_helpr::{below_len, ValidatorResult}};

  pub(super) fn text(text: &text::Text) -> ValidatorResult {
    below_len("Section.text", 3000, text.as_ref())
  }

  pub(super) fn block_id(text: &Cow<str>) -> ValidatorResult {
    below_len("Section.block_id", 255, text.as_ref())
  }

  pub(super) fn fields(texts: &Cow<[text::Text]>) -> ValidatorResult {
    below_len("Section.fields", 10, texts.as_ref()).and(
                                                        texts.iter()
                                                             .map(|text| {
                                                               below_len(
             "Section.fields",
             2000,
             text.as_ref())
                                                             })
                                                             .collect(),
    )
  }
}
