use tempfile::TempDir;

use super::InteractiveChecker;
use crate::tests::{FakeDictionary, FakeInteractor, FakeRepository};
use crate::{Checker, ProjectPath, RelativePath};

type TestChecker = InteractiveChecker<FakeInteractor, FakeDictionary, FakeRepository>;

struct TestApp {
    checker: TestChecker,
}

impl TestApp {
    fn new(temp_dir: &TempDir) -> Self {
        let interactor = FakeInteractor::new();
        let dictionary = FakeDictionary::new();
        let repository = FakeRepository::new();
        let project_path = ProjectPath::new(temp_dir.path()).unwrap();
        let checker = TestChecker::new(project_path, interactor, dictionary, repository).unwrap();
        Self { checker }
    }

    fn add_known(&mut self, words: &[&str]) {
        for word in words.iter() {
            self.checker.dictionary.add_known(word);
        }
    }

    fn push_text(&mut self, answer: &str) {
        self.checker.interactor.push_text(answer)
    }

    fn to_relative_path(&self, path: &str) -> RelativePath {
        let project_path = self.checker.project.path();
        let path = project_path.as_ref().join(path);
        RelativePath::new(project_path, &path).unwrap()
    }

    fn handle_token(&mut self, token: &str, relative_name: &str) {
        let project_path = self.checker.project().path();
        let full_path = project_path.as_ref().join(relative_name);
        std::fs::write(&full_path, "").unwrap();
        let relative_path = self.to_relative_path(relative_name);
        let context = &(3, 42);
        self.checker
            .handle_token(token, &relative_path, context)
            .unwrap()
    }

    fn is_ignored(&self, word: &str) -> bool {
        self.checker.repository().is_ignored(word).unwrap()
    }

    fn is_skipped_file_name(&self, file_name: &str) -> bool {
        self.checker
            .repository()
            .is_skipped_file_name(file_name)
            .unwrap()
    }

    fn is_skipped_path(&self, relative_name: &str) -> bool {
        let project_id = self.checker.project().id();
        let relative_path = self.to_relative_path(relative_name);
        self.checker
            .repository()
            .is_skipped_path(project_id, &relative_path)
            .unwrap()
    }

    fn is_ignored_for_extension(&self, word: &str, extension: &str) -> bool {
        self.checker
            .repository()
            .is_ignored_for_extension(word, extension)
            .unwrap()
    }

    fn is_ignored_for_project(&self, word: &str) -> bool {
        let project_id = self.checker.project().id();
        self.checker
            .repository()
            .is_ignored_for_project(word, project_id)
            .unwrap()
    }

    fn is_ignored_for_path(&self, word: &str, relative_name: &str) -> bool {
        let project_id = self.checker.project().id();
        let relative_path = self.to_relative_path(relative_name);
        self.checker
            .repository()
            .is_ignored_for_path(word, project_id, &relative_path)
            .unwrap()
    }

    fn end(&self) {
        if !self.checker.interactor.is_empty() {
            panic!("Not all answered consumed by the test");
        }
    }
}

#[test]
fn test_adding_token_to_global_ignore() {
    let temp_dir = tempfile::Builder::new()
        .prefix("test-skyspell")
        .tempdir()
        .unwrap();
    let mut app = TestApp::new(&temp_dir);
    app.add_known(&["hello", "world"]);
    app.push_text("a");

    app.handle_token("foo", "foo.txt");

    assert!(app.is_ignored("foo"));
    app.handle_token("foo", "other.ext");

    app.end();
}

#[test]
fn test_adding_token_to_extension() {
    let temp_dir = tempfile::Builder::new()
        .prefix("test-skyspell")
        .tempdir()
        .unwrap();
    let mut app = TestApp::new(&temp_dir);
    app.add_known(&["hello", "world"]);
    app.push_text("e");

    app.handle_token("defaultdict", "hello.py");

    assert!(app.is_ignored_for_extension("defaultdict", "py"));
    app.handle_token("defaultdict", "bar.py");

    app.end();
}

#[test]
fn test_adding_token_to_project() {
    let temp_dir = tempfile::Builder::new()
        .prefix("test-skyspell")
        .tempdir()
        .unwrap();
    let mut app = TestApp::new(&temp_dir);
    app.push_text("p");

    app.handle_token("foo", "foo.py");

    assert!(app.is_ignored_for_project("foo"));
    app.handle_token("foo", "foo.py");

    app.end()
}

#[test]
fn test_ignore_token_to_project_file() {
    let temp_dir = tempfile::Builder::new()
        .prefix("test-skyspell")
        .tempdir()
        .unwrap();
    let mut app = TestApp::new(&temp_dir);
    app.push_text("f");

    app.handle_token("foo", "foo.py");

    assert!(app.is_ignored_for_path("foo", "foo.py"));
    app.handle_token("foo", "foo.py");

    app.end()
}

#[test]
fn test_adding_to_skipped_file_names() {
    let temp_dir = tempfile::Builder::new()
        .prefix("test-skyspell")
        .tempdir()
        .unwrap();
    let mut app = TestApp::new(&temp_dir);
    app.add_known(&["hello", "world"]);
    app.push_text("n");

    app.handle_token("foo", "yarn.lock");

    assert!(app.is_skipped_file_name("yarn.lock"));
    app.handle_token("bar", "yarn.lock");

    app.end();
}

#[test]
fn test_adding_to_skipped_paths() {
    let temp_dir = tempfile::Builder::new()
        .prefix("test-skyspell")
        .tempdir()
        .unwrap();
    let mut app = TestApp::new(&temp_dir);
    app.push_text("s");

    app.handle_token("foo", "foo.py");

    assert!(app.is_skipped_path("foo.py"));
    app.handle_token("bar", "foo.py");

    app.end();
}

#[test]
fn test_remember_skipped_tokens() {
    let temp_dir = tempfile::Builder::new()
        .prefix("test-skyspell")
        .tempdir()
        .unwrap();
    let mut app = TestApp::new(&temp_dir);
    app.add_known(&["hello", "world"]);
    app.push_text("x");

    app.handle_token("foo", "foo.py");
    app.handle_token("foo", "foo.py");

    app.end();
}
