use super::Ream;
use crate::env::*;
use crate::{Setup, Sketch};

/// A ream consisting of a single sketch.
///
/// ```
/// # use macro_rules_attribute::apply;
/// # use sketchbook::minimal;
/// # use sketchbook::derive_sketch;
/// # use sketchbook::Setup;
/// # use sketchbook::*;
/// # use sketchbook::ream::*;
/// # use sketchbook::ream::Single;
/// #[apply(derive_sketch)]
/// #[sketch(env=minimal)]
/// struct App {
///     #[page] page: minimal::Page
/// }
///
/// impl Setup for App {
///     fn setup(page: minimal::Page, _: ()) -> Self {
///         Self { page }
///     }
/// }
///
/// // create ream
/// let mut single = Single::<App>::new();
///
/// // run ream in environment
/// minimal::Env.run(move |mill| single.update(mill));
/// ```
pub struct Single<S, I = ()> {
    sketch: Option<S>,
    input: Option<I>,
}

impl<S, I> Single<S, I> {
    /// Create new ream that will pass the default value for the input type.
    pub fn new() -> Self
    where
        I: Default,
    {
        Self {
            sketch: None,
            input: Some(I::default()),
        }
    }

    /// Create new ream that will pass an input to the sketch.
    pub fn with_input(input: I) -> Self {
        Self {
            sketch: None,
            input: Some(input),
        }
    }
}

impl<S, I> Default for Single<S, I>
where
    I: Default,
{
    fn default() -> Self {
        Self::new()
    }
}

impl<S, I> From<I> for Single<S, I> {
    fn from(input: I) -> Self {
        Self::with_input(input)
    }
}

impl<S, I> Ream for Single<S, I>
where
    S: Sketch + Setup<I>,
{
    type Env = S::Env;

    fn update(&mut self, mill: &mut MillOf<Self::Env>) -> Status {
        if let Some(sketch) = &mut self.sketch {
            // process group of events
            sketch.get_page_mut().start_group();
            while let Some(event) = sketch.get_page_mut().next_event_in_group() {
                sketch.get_page_mut().before_event(&event);
                sketch.handle_environment_event(&event);
                sketch.get_page_mut().finish_event(event);
            }
            sketch.get_page_mut().end_group();

            sketch.get_page_mut().status()
        } else {
            let page = mill.new_page();
            let sketch = S::setup(page, self.input.take().unwrap());
            self.sketch = Some(sketch);
            Status::Continue
        }
    }
}
