//! this ream is under development

use std::cell::RefCell;
use std::marker::PhantomData;
use std::rc::Rc;

use crate::Mill;
use crate::Page;
use crate::{book::Book, Environment, MillOf, PageOf, Ream, Setup, Sketch};

struct ConcurrentCreate<S, I> {
    input: I,
    marker: PhantomData<S>,
}

trait CreateItem<E>
where
    E: Environment,
{
    fn create(self, page: PageOf<E>) -> Box<dyn Sketch<Env = E>>;
    fn create_from_box(self: Box<Self>, page: PageOf<E>) -> Box<dyn Sketch<Env = E>>;
}

impl<S, I> CreateItem<S::Env> for ConcurrentCreate<S, I>
where
    S: Sketch + Setup<I> + 'static,
{
    fn create(self, page: PageOf<S::Env>) -> Box<dyn Sketch<Env = S::Env>> {
        Box::new(S::setup(page, self.input))
    }

    fn create_from_box(self: Box<Self>, page: PageOf<S::Env>) -> Box<dyn Sketch<Env = S::Env>> {
        self.create(page)
    }
}

pub struct Concurrent<E> {
    to_create_queue: Rc<RefCell<Vec<Box<dyn CreateItem<E>>>>>,
    to_creates: Vec<Box<dyn CreateItem<E>>>,
    sketches: Vec<Box<dyn Sketch<Env = E>>>,
}

impl<E> Default for Concurrent<E>
where
    E: Environment,
{
    fn default() -> Self {
        Self {
            sketches: Vec::new(),
            to_creates: Vec::new(),
            to_create_queue: Rc::new(RefCell::new(Vec::new())),
        }
    }
}

impl<E> Ream for Concurrent<E>
where
    E: Environment,
{
    type Env = E;

    fn update(&mut self, mill: &mut MillOf<Self::Env>) -> bool {
        if !self.to_creates.is_empty() {
            let to_creates = std::mem::take(&mut self.to_creates);
            for to_create in to_creates {
                self.sketches
                    .push(to_create.create_from_box(mill.new_page()));
            }
        }

        if !self.to_create_queue.borrow().is_empty() {
            let queue: Vec<Box<dyn CreateItem<E>>> =
                std::mem::take(self.to_create_queue.borrow_mut().as_mut());
            for to_create in queue {
                self.sketches
                    .push(to_create.create_from_box(mill.new_page()));
            }
        }

        for sketch in &mut self.sketches {
            if let Some(event) = sketch.as_mut().get_page_mut().try_next() {
                sketch.get_page_mut().update();
                sketch.handle_environment_event(&event);
                sketch.as_mut().get_page_mut().finish_event(event);
            }
        }

        self.sketches.retain(|sketch| !sketch.get_page().is_done());

        !self.sketches.is_empty()
    }
}

impl<E> Concurrent<E>
where
    E: Environment,
{
    pub fn create_proxy(&self) -> ConcurrentProxy<E> {
        ConcurrentProxy {
            to_create_queue: self.to_create_queue.clone(),
        }
    }
}

pub struct ConcurrentProxy<E> {
    to_create_queue: Rc<RefCell<Vec<Box<dyn CreateItem<E>>>>>,
}

impl<E> ConcurrentProxy<E>
where
    E: Environment,
{
    pub fn add_page<S>(&self)
    where
        S: Sketch<Env = E> + Setup<()> + 'static,
    {
        self.add_page_with::<S, ()>(())
    }

    pub fn add_page_with<S, I>(&self, input: I)
    where
        S: Sketch<Env = E> + Setup<I> + 'static,
        I: 'static,
    {
        let mut queue = self.to_create_queue.borrow_mut();
        queue.push(Box::new(ConcurrentCreate::<S, I> {
            marker: PhantomData,
            input,
        }));
    }
}

pub trait ConcurrentBookExt<E>
where
    E: Environment,
{
    fn ream_proxy(&mut self) -> ConcurrentProxy<E>;

    fn add_page<S>(&mut self)
    where
        S: Setup<()> + Sketch<Env = E> + 'static;

    fn add_page_with<S, I>(&mut self, input: I)
    where
        S: Setup<I> + Sketch<Env = E> + 'static,
        I: 'static;
}

impl<E> ConcurrentBookExt<E> for Book<Concurrent<E>>
where
    E: Environment,
{
    fn ream_proxy(&mut self) -> ConcurrentProxy<E> {
        self.ream.create_proxy()
    }

    fn add_page<S>(&mut self)
    where
        S: Setup<()> + Sketch<Env = E> + 'static,
    {
        self.add_page_with::<S, ()>(())
    }

    fn add_page_with<S, I>(&mut self, input: I)
    where
        S: Setup<I> + Sketch<Env = E> + 'static,
        I: 'static,
    {
        self.ream
            .to_creates
            .push(Box::new(ConcurrentCreate::<S, I> {
                marker: PhantomData,
                input,
            }));
    }
}
