//! Environment for sketches to run in.

pub mod minimal;

mod single_aspect;

/// Trait for environment page.
///
/// A page is the local context of the environment for a sketch.
/// This allows the sketch to interact with the environment directly.
/// Additionally, the environment can push events to the sketch for it to handle.
pub trait Page {
    /// Events generated by this page.
    type Event;

    /// Signals the ream is starting processing events for this page.
    ///
    /// The page should use this signal to limit the events returned by `next_event_in_group` to
    /// those already queued since the last group started.
    fn start_group(&mut self);

    /// Signals the ream has stopped processing events for this page.
    fn end_group(&mut self);

    /// Get the next event in this group of events if there is one.
    fn next_event_in_group(&mut self) -> Option<Self::Event>;

    /// Process an event before it is passed to the sketch to handle.
    fn before_event(&mut self, event: &Self::Event);

    /// Process an event after it is passed to the sketch.
    fn finish_event(&mut self, event: Self::Event);

    /// Status of the sketch/page so the ream knows to continue or stop execution.
    fn status(&self) -> Status;
}

/// Factory for creating pages from an environment.
pub trait Mill {
    /// Type of page this mill creates.
    type Page: Page;

    /// Create a new page connected to the environment.
    fn new_page(&mut self) -> Self::Page;
}

/// Status flag
pub enum Status {
    /// Operation should continue
    Continue,

    /// Operation should be stopped
    Stop,
}

/// Environment for sketches to run in and interact with.
pub trait Environment {
    /// Mill for environment to create pages from.
    type Mill: Mill;

    /// Run environment with callback function.
    ///
    /// The callback function should be a ream `.update()` function in a `move` closure.
    fn run<F>(self, func: F)
    where
        F: FnMut(&mut Self::Mill) -> Status + 'static;
}

/// Helper to get `Mill` of type implementing the `Environment` trait.
pub type MillOf<T> = <T as Environment>::Mill;

/// Helper to get `Page` of type implementing the `Environment` trait.
pub type PageOf<T> = <MillOf<T> as Mill>::Page;

/// Helper to get `Event` of type implementing the `Environment` trait.
pub type EventOf<T> = <PageOf<T> as Page>::Event;
