use core::marker::PhantomData;

use crate::env::*;
use crate::ream::*;

/// A set of sketches.
pub struct Book<R> {
    phantom_data: PhantomData<R>,
}

/// Pages of a book.
pub struct Pages<R>
where
    R: Ream,
{
    /// The ream for the pages.
    pub ream: R,

    /// The environment the pages are in.
    pub env: <R as Ream>::Env,
}

impl<R> Pages<R>
where
    R: Ream + 'static,
{
    /// Bind into a book.
    ///
    /// This runs the ream and all sketches it contains.
    pub fn bind(self) {
        let mut ream = self.ream;
        self.env.run(move |proxy| ream.update(proxy));
    }
}

impl<R> Book<R>
where
    R: Default + Ream + 'static,
    <R as Ream>::Env: Default,
{
    /// Create pages to add sketches to.
    pub fn pages() -> Pages<R> {
        Pages {
            ream: R::default(),
            env: <R as Ream>::Env::default(),
        }
    }

    /// Immediately bind a new book.
    ///
    /// This will start running the ream and therefore any sketches.
    pub fn bind() {
        Self::pages().bind()
    }
}

impl<R> Book<R>
where
    R: Ream,
    <R as Ream>::Env: Default,
{
    /// Create pages with input data.
    pub fn pages_with<I>(input: I) -> Pages<R>
    where
        R: From<I>,
    {
        Pages {
            ream: R::from(input),
            env: <R as Ream>::Env::default(),
        }
    }
}
