//! Periodicly updates sketch state.
//!
//! The `Update::update()` event handler will be called at the rate set.
//! By default, the update rate is 30 times per second.
//! The `self.pause()` and `self.resume()` methods allow for temporarily stopping update events.
//!
//! The sketch can be fully stopped with `self.stop()`.
//! This will notify the environment that the sketch should be cleaned up.
//!
//! See [UpdateExt]() for details on methods added to the sketch.
//!
//! Example:
//! ```ignore
//! struct Example {
//!     count: usize,
//! }
//!
//! impl Update for Example {
//!     fn update(&mut self, _delta_t: Duration) {
//!         self.count += 1;
//!
//!         // stop sketch after 10 updates
//!         if self.count == 10 {
//!             self.stop();
//!         }
//!     }
//! }
//! ```

use super::aspect;

aspect! {
    pub trait Handler {
        /// Handle an update event.
        ///
        /// Parameter `delta_t` is the time since the last call to update.
        fn update(delta_t: types::DeltaT<Self>);
    }

    impl SketchExt for <Sketch> {
        /// Stop running this sketch.
        fn stop(&mut self) {
            self.as_part_mut().should_stop = true;
        }

        /// Pause updates for this sketch.
        fn pause(&mut self) {
            self.as_part_mut().is_paused = true;
        }

        /// Resume updates for this sketch.
        fn resume(&mut self) {
            self.as_part_mut().is_paused = false;
        }

        /// Check if sketch is paused.
        fn is_paused(&self) -> bool {
            self.as_part().is_paused
        }

        /// Set the rate the update handler is called.
        /// The environment may not be able to actually call the update handler at the rate given.
        fn update_rate[R: crate::real::ToReal<types::UpdateRate<Self>>](&mut self, rate: R) {
            self.as_part_mut().update_rate = rate.to_real();
        }
    }

    pub mod env_api {
        {
            use super::{GetTime, UpdateDefaults};
            use crate::compose::TryAsPart;
        }

        Page: GetTime[
            Time=super::associated::types::Time<Self>,
            Duration=super::associated::types::DeltaT<Self>
        ];

        pub trait Types: UpdateDefaults[DefaultUpdateRate=Self::UpdateRate] {
            /// Type for delta times.
            type DeltaT;

            /// Type for times.
            type Time;

            /// Type for update rate.
            type UpdateRate: Copy;
        }

        pub enum Events<T> {
            /// Update sketch.
            Update
        }

        pub struct Data<T> {
            pub(super) update_rate: <T as AssociatedTypes>::UpdateRate,
            pub(super) is_paused: bool,
            pub(super) should_stop: bool,
            pub(super) last_time: Option<T::Time>,
        }

        impl Data<T> {
            /// Create data for update aspect.
            pub fn new() -> Self {
                Self {
                    _types_phantom: core::marker::PhantomData,
                    update_rate: T::default_update_rate(),
                    is_paused: false,
                    should_stop: false,
                    last_time: None,
                }
            }

            /// Get should stop flag.
            pub fn should_stop(&self) -> bool {
                self.should_stop
            }

            /// Get the update rate.
            pub fn update_rate(&self) -> T::UpdateRate {
                self.update_rate
            }

            /// Single to stop executing the sketch.
            pub fn stop(&mut self) {
                self.should_stop = true;
            }
        }

        impl Handle for <Sketch> {
            fn handle(&mut self, event: _) {
                if event.try_as_part().is_some() && !self.as_part().is_paused {
                    let now = self.get_page_mut().get_time();
                    let delta_t = if let Some(last_time) = &self.as_part().last_time {
                        <crate::env::PageOf<Self::Env> as GetTime>::duration_between(last_time, &now)
                    } else {
                        <crate::env::PageOf<Self::Env> as GetTime>::zero_duration()
                    };
                    self.update(delta_t);
                    self.as_part_mut().last_time = Some(now);
                }
            }
        }
    }
}

/// Trait to get current time from environment.
/// This is used to calculate the `delta_t` passed to the `update()` handler.
pub trait GetTime {
    /// Type for times
    type Time;

    /// Type for durations
    type Duration;

    /// Get the current time.
    fn get_time(&mut self) -> Self::Time;

    /// Value for zero duration.
    fn zero_duration() -> Self::Duration;

    /// Value for zero duration.
    fn duration_between(start: &Self::Time, end: &Self::Time) -> Self::Duration;
}

/// Default values for update aspect
pub trait UpdateDefaults {
    /// Type for real numbers
    type DefaultUpdateRate;

    /// Get default value for update rate.
    fn default_update_rate() -> Self::DefaultUpdateRate;
}

/// Implementation of update aspect using Instant.
#[cfg(feature = "std")]
#[derive(Default)]
pub struct UpdateUsingInstant {
    last_update: Option<std::time::Instant>,
}

#[cfg(feature = "std")]
impl UpdateUsingInstant {
    /// Check if an update should happen. If an update should happen, then the time of calling
    /// this function is recorded to compare against the next call time.
    pub fn should_update(&mut self, update_rate: f32) -> bool {
        let now = std::time::Instant::now();
        if let Some(last_update) = self.last_update {
            if now.duration_since(last_update).as_secs_f32() >= 1. / update_rate {
                // time has expired so flag for update
                self.last_update = Some(now);
                true
            } else {
                false
            }
        } else {
            // this is the first check so flag for update
            self.last_update = Some(now);
            true
        }
    }
}

#[test]
fn test_update() {
    use crate::aspects::update;
    use crate::env::*;
    use crate::*;

    mod single_aspect {
        use crate::aspects::update;
        use crate::env_for_aspect;
        env_for_aspect! {
            update
            {
                type Time = u8;
                type DeltaT = i16;
                type UpdateRate = f32;
            }
            {
                pub time: u8,
            }
        }
    }

    impl GetTime for single_aspect::Page {
        type Time = u8;

        type Duration = i16;

        fn get_time(&mut self) -> Self::Time {
            self.time
        }

        fn zero_duration() -> Self::Duration {
            0
        }

        fn duration_between(&start: &Self::Time, &end: &Self::Time) -> Self::Duration {
            (end as i16) - (start as i16)
        }
    }

    impl UpdateDefaults for single_aspect::TypesHolder {
        type DefaultUpdateRate = f32;

        fn default_update_rate() -> Self::DefaultUpdateRate {
            30.0
        }
    }

    derive_sketch! {
        #[sketch(
            env = single_aspect,
            aspects = (update),
        )]
        struct App {
            #[page] page: single_aspect::Page,
            updated: bool,
        }
    }

    impl Setup for App {
        fn setup(page: single_aspect::Page, _: ()) -> Self {
            App {
                page,
                updated: false,
            }
        }
    }

    impl update::Handler for App {
        fn update(&mut self, delta_t: i16) {
            if self.updated {
                assert_eq!(delta_t, 123);
            } else {
                assert_eq!(delta_t, 0);
                self.updated = true;
            }
        }
    }

    let mut mill = single_aspect::Mill;

    let mut app = App::setup(mill.new_page(), ());

    app.handle_environment_event(&single_aspect::Events::Aspect(
        update::env_api::Events::Update,
    ));

    assert!(app.updated);

    app.page.time = 123;

    app.handle_environment_event(&single_aspect::Events::Aspect(
        update::env_api::Events::Update,
    ));

    assert_eq!(app.page.aspect.update_rate, 30.0);
    assert_eq!(app.page.aspect.is_paused, false);
    assert_eq!(app.page.aspect.should_stop, false);
    assert_eq!(app.page.aspect.last_time, Some(123));

    app.stop();
    assert_eq!(app.page.aspect.should_stop, true);

    app.pause();
    assert_eq!(app.page.aspect.is_paused, true);
    assert_eq!(app.is_paused(), true);

    app.resume();
    assert_eq!(app.page.aspect.is_paused, false);
    assert_eq!(app.is_paused(), false);

    app.update_rate(10.0_f64);
    assert_eq!(app.page.aspect.update_rate, 10.0);
}

#[cfg(feature = "std")]
#[test]
fn test_update_instant_impl() {
    let mut update_impl = UpdateUsingInstant::default();

    let rate = 1_000_000.0;

    assert!(update_impl.should_update(rate));
    assert!(!update_impl.should_update(rate));
    std::thread::sleep(std::time::Duration::from_nanos(10));
    assert!(update_impl.should_update(rate));
    assert!(!update_impl.should_update(rate));
    std::thread::sleep(std::time::Duration::from_nanos(10));
    assert!(update_impl.should_update(rate));
    assert!(!update_impl.should_update(rate));
}
