//! Mouse input.

use super::aspect;
use crate::geometry::Point2;
use core::borrow::Borrow;

/// Trait for mouse buttons.
#[cfg(not(feature = "std"))]
pub trait Button: Ord {}

#[cfg(not(feature = "std"))]
impl<T: Ord> Button for T {}

#[cfg(not(feature = "std"))]
type Set<T> = alloc::collections::BTreeSet<T>;

/// Trait for mouse buttons.
#[cfg(feature = "std")]
pub trait Button: std::hash::Hash + std::cmp::Eq {}

#[cfg(feature = "std")]
impl<T: std::hash::Hash + std::cmp::Eq> Button for T {}

#[cfg(feature = "std")]
type Set<T> = std::collections::HashSet<T>;

aspect! {
    pub trait Handler {
        /// Mouse moved while a button is held.
        fn dragged();

        /// Mouse moved with no buttons pressed.
        fn moved();

        /// Mouse button was clicked.
        fn clicked(button: &types::Button<Self>);

        /// Mouse button was pressed.
        fn pressed(button: &types::Button<Self>);
        
        /// Mouse button was released.
        fn released(button: &types::Button<Self>);

        /// Mouse wheel was scrolled.
        fn wheel(count: types::WheelCount<Self>);
    }

    impl SketchExt for <Sketch> {
        /// Get the current x position of the mouse.
        fn mouse_x(&self) -> types::Position<Self> {
            self.as_part().position.x
        }

        /// Get the current y position of the mouse.
        fn mouse_y(&self) -> types::Position<Self> {
            self.as_part().position.y
        }

        /// Get the current position of the mouse.
        fn mouse_point(&self) -> Point2<types::Position<Self>> {
            self.as_part().position
        }

        /// Check if a mouse button is pressed.
        fn button_pressed[B: Borrow<types::Button<Self>>](&self, button: B) -> bool {
            self.as_part().pressed_buttons.contains(button.borrow())
        }

        /// Check if any mouse buttons are pressed.
        fn any_button_pressed(&self) -> bool {
            !self.as_part().pressed_buttons.is_empty()
        }
    }

    pub mod env_api {
        {
            use crate::compose::TryAsPart;
            use crate::geometry::Point2;
            use core::ops::*;
            use super::Button;
            use super::Set;
        }

        pub trait Types {
            /// Type used for mouse buttons.
            type Button: Clone + Button;
            
            /// Type used for mouse position.
            type Position: Default + Copy + AddAssign;

            /// Type used for scroll wheel.
            type WheelCount: Copy;
        }

        pub enum Events<T> {
            /// Mouse move relative position
            MoveRelative(Point2<T::Position>),

            /// Mouse move absolute position
            MoveAbsolute(Point2<T::Position>),

            /// Mouse button pressed
            Press(T::Button),

            /// Mouse button released
            Release(T::Button),
            
            /// Scroll wheel event
            Wheel(T::WheelCount),
        }

        pub struct Data<T> {
            pub(super) position: Point2<T::Position>,
            pub(super) pressed_buttons: Set<T::Button>,
        }

        impl Data<T> {
            /// Create new data for aspect.
            pub fn new() -> Self {
                Self {
                    _types_phantom: core::marker::PhantomData,
                    position: Point2 { x: T::Position::default(), y: T::Position::default() },
                    pressed_buttons: Set::new(),
                }
            }
        }

        impl Handle for <Sketch> {
            fn handle(&mut self, event: _) {
                match event.try_as_part() {
                    Some(Events::MoveRelative(point)) => {
                        self.as_part_mut().position += point;
                        if self.as_part().pressed_buttons.is_empty() {
                            self.moved();
                        } else {
                            self.dragged();
                        }
                    }
                    Some(Events::MoveAbsolute(point)) => {
                        self.as_part_mut().position = *point;
                        if self.as_part().pressed_buttons.is_empty() {
                            self.moved();
                        } else {
                            self.dragged();
                        }
                    }
                    Some(Events::Wheel(count)) => {
                        self.wheel(*count);
                    }
                    Some(Events::Press(button)) => {
                        self.as_part_mut().pressed_buttons.insert(button.clone());
                        self.pressed(button);
                    }
                    Some(Events::Release(button)) => {
                        let was_pressed = self.as_part_mut().pressed_buttons.remove(button);
                        self.released(button);
                        if was_pressed {
                            self.clicked(button);
                        }
                    }
                    Some(Events::_MetaPhantom(_, _)) => unreachable!(),
                    None => {}
                }
            }
        }
    }
}

#[test]
fn test_mouse() {
    use crate::aspects::mouse;
    use crate::env::*;
    use crate::*;

    mod single_aspect {
        use crate::aspects::mouse;
        use crate::env_for_aspect;
        env_for_aspect! {
            mouse
            {
                type Button = u8;
                type Position = f32;
                type WheelCount = i8;
            }
        }
    }

    derive_sketch! {
        #[sketch(
            env = single_aspect,
            aspects = (mouse),
        )]
        struct App {
            #[page] page: single_aspect::Page,
            dragged: bool,
            moved: bool,
            clicked: bool,
            pressed: bool,
            released: bool,
            wheel: bool,
        }
    }

    impl Setup for App {
        fn setup(page: single_aspect::Page, _: ()) -> Self {
            App {
                page,
                dragged: false,
                moved: false,
                clicked: false,
                pressed: false,
                released: false,
                wheel: false,
            }
        }
    }

    impl mouse::Handler for App {
        fn dragged(&mut self) {
            assert_eq!(self.mouse_x(), -2.0);
            assert_eq!(self.mouse_y(), 1.0);
            assert_eq!(self.mouse_point(), Point2 { x: -2.0, y: 1.0 });
            self.dragged = true;
        }

        fn moved(&mut self) {
            assert_eq!(self.mouse_x(), 1.0);
            assert_eq!(self.mouse_y(), -2.0);
            assert_eq!(self.mouse_point(), Point2 { x: 1.0, y: -2.0 });
            self.moved = true;
        }

        fn clicked(&mut self, &button: &types::Button<Self>) {
            assert_eq!(button, 4);
            self.clicked = true;
        }

        fn pressed(&mut self, &button: &types::Button<Self>) {
            assert_eq!(button, 4);
            self.pressed = true;
        }

        fn released(&mut self, &button: &types::Button<Self>) {
            assert_eq!(button, 4);
            self.released = true;
        }

        fn wheel(&mut self, count: types::WheelCount<Self>) {
            assert_eq!(count, -20);
            self.wheel = true;
        }
    }

    let mut mill = single_aspect::Mill;

    let mut app = App::setup(mill.new_page(), ());

    app.handle_environment_event(&single_aspect::Events::Aspect(
        mouse::env_api::Events::MoveAbsolute(Point2 { x: 1.0, y: -2.0 }),
    ));

    app.handle_environment_event(&single_aspect::Events::Aspect(
        mouse::env_api::Events::Press(4),
    ));

    app.handle_environment_event(&single_aspect::Events::Aspect(
        mouse::env_api::Events::MoveRelative(Point2 { x: -3.0, y: 3.0 }),
    ));

    app.handle_environment_event(&single_aspect::Events::Aspect(
        mouse::env_api::Events::Release(4),
    ));

    app.handle_environment_event(&single_aspect::Events::Aspect(
        mouse::env_api::Events::Wheel(-20),
    ));

    assert!(app.moved);
    assert!(app.pressed);
    assert!(app.dragged);
    assert!(app.released);
    assert!(app.clicked);
    assert!(app.wheel);
}
