//! Keyboard input.
//!
//! ```
//! use sketchbook::aspects::keyboard;
//! # use sketchbook::derive_sketch;
//! # use macro_rules_attribute::apply;
//! # mod single_aspect {
//! #     use sketchbook::env_for_aspect;
//! #     use sketchbook::aspects::keyboard;
//! #     env_for_aspect! {
//! #         keyboard
//! #         {
//! #             type Key = u8;
//! #         }
//! #     }
//! # }
//!
//! #[apply(derive_sketch)]
//! #[sketch(env=single_aspect, aspects=(keyboard))]
//! struct App {
//!     #[page] page: single_aspect::Page,
//! }
//!
//! impl keyboard::Handler for App {}
//! ```

use super::aspect;
use core::borrow::Borrow;

aspect! {
    pub trait Handler {
        /// Handle pressed key.
        fn pressed(key: &types::Key<Self>);

        /// Handle key repeat event.
        ///
        /// This happens in some environments when a key is held.
        fn repeated(key: &types::Key<Self>);

        /// Handle release key.
        fn released(key: &types::Key<Self>);
    }

    impl SketchExt for <Sketch> {
        /// Check if key is currently pressed.
        fn key_pressed[K: Borrow<types::Key<Self>>](&self, key: K) -> bool {
            self.as_part().key_pressed(key.borrow())
        }
    }

    pub mod env_api {
        {
            use alloc::collections::BTreeSet;
            use crate::compose::TryAsPart;
        }

        pub trait Types {
            /// Type used for keyboard keys.
            type Key: Ord + Clone;
        }

        pub enum Events<T> {
            /// Key is pressed.
            Press(T::Key),

            /// Key is repeated.
            Repeat(T::Key),

            /// Key is released.
            Release(T::Key),
        }

        pub struct Data<T> {
            pub(super) pressed_keys: BTreeSet<T::Key>,
        }

        impl Data<T> {
            /// Create new data for aspect.
            pub fn new() -> Self {
                Self {
                    _types_phantom: core::marker::PhantomData,
                    pressed_keys: BTreeSet::new(),
                }
            }

            pub(super) fn key_pressed(&self, key: &T::Key) -> bool {
                self.pressed_keys.contains(key)
            }
        }

        impl Handle for <Sketch> {
            fn handle(&mut self, event: _) {
                match event.try_as_part() {
                    Some(Events::Press(key)) => {
                        self.as_part_mut().pressed_keys.insert(key.clone());
                        self.pressed(key);
                    }
                    Some(Events::Repeat(key)) => {
                        self.as_part_mut().pressed_keys.insert(key.clone());
                        self.repeated(key);
                    }
                    Some(Events::Release(key)) => {
                        let _was_pressed = self.as_part_mut().pressed_keys.remove(key);
                        self.released(key);
                    }
                    Some(Events::_MetaPhantom(_, _)) => unreachable!(),
                    None => {}
                }
            }
        }
    }
}

#[test]
fn test_keyboard() {
    use crate::aspects::keyboard;
    use crate::env::*;
    use crate::*;

    mod single_aspect {
        use crate::aspects::keyboard;
        use crate::env_for_aspect;
        env_for_aspect! {
            keyboard
            {
                type Key = u8;
            }
        }
    }

    derive_sketch! {
        #[sketch(
            env = single_aspect,
            aspects = (keyboard),
        )]
        struct App {
            #[page] page: single_aspect::Page,
            pressed: bool,
            released: bool,
            repeated: bool,
        }
    }

    impl Setup for App {
        fn setup(page: single_aspect::Page, _: ()) -> Self {
            App {
                page,
                pressed: false,
                released: false,
                repeated: false,
            }
        }
    }

    impl keyboard::Handler for App {
        fn pressed(&mut self, &key: &u8) {
            assert_eq!(key, 123);
            assert!(self.key_pressed(123));
            self.pressed = true;
        }

        fn repeated(&mut self, &key: &u8) {
            assert_eq!(key, 123);
            assert!(self.key_pressed(123));
            self.repeated = true;
        }

        fn released(&mut self, &key: &u8) {
            assert_eq!(key, 123);
            assert!(!self.key_pressed(123));
            self.released = true;
        }
    }

    let mut mill = single_aspect::Mill;

    let mut app = App::setup(mill.new_page(), ());

    app.handle_environment_event(&single_aspect::Events::Aspect(
        keyboard::env_api::Events::Press(123),
    ));

    app.handle_environment_event(&single_aspect::Events::Aspect(
        keyboard::env_api::Events::Repeat(123),
    ));

    app.handle_environment_event(&single_aspect::Events::Aspect(
        keyboard::env_api::Events::Release(123),
    ));

    assert!(app.pressed);
    assert!(app.released);
    assert!(app.repeated);
}
