use super::circle::Circle;
use super::context::Context;
use super::rectangle::Rectangle;

/// Shapes that can be drawn.
#[derive(Debug, PartialEq, Clone, Copy)]
pub enum Shape<R, C> {
    /// A circle
    Circle(Circle<R, C>),

    /// A rectangle
    Rectangle(Rectangle<R, C>),
}

/// Create shape from value.
pub trait ShapeFrom<T, R, C> {
    /// Convert the value in relation to a context.
    fn shape_from(data: T, context: &Context<R, C>) -> Shape<R, C>;
}

/// Convert value into a shape.
pub trait IntoShape<T, R, C> {
    /// Convert the value in relation to a context.
    fn into_shape(self, context: &Context<R, C>) -> Shape<R, C>;
}

impl<T, R, C> ShapeFrom<&T, R, C> for T
where
    T: Clone + IntoShape<T, R, C>,
{
    fn shape_from(data: &T, context: &Context<R, C>) -> Shape<R, C> {
        data.clone().into_shape(context)
    }
}

impl<T, R, C> ShapeFrom<&mut T, R, C> for T
where
    T: Clone + IntoShape<T, R, C>,
{
    fn shape_from(data: &mut T, context: &Context<R, C>) -> Shape<R, C> {
        data.clone().into_shape(context)
    }
}

impl<T, S, R, C> IntoShape<S, R, C> for T
where
    S: ShapeFrom<T, R, C>,
{
    fn into_shape(self, context: &Context<R, C>) -> Shape<R, C> {
        S::shape_from(self, context)
    }
}
