use super::context::Context;
use super::mode::*;
use super::shape::*;
use crate::geometry::*;
use crate::real::*;

/// Radius of rectangle's corners
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub enum CornerRadius<R> {
    /// Uniform radius for all corners
    Uniform(R),

    /// Radius for each corner
    PerCorner {
        /// Top left corner
        top_left: R,
        /// Top right corner
        top_right: R,
        /// Bottom right corner
        bottom_right: R,
        /// Bottom left corner
        bottom_left: R,
    },
}

/// A rectangle
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub struct Rectangle<R, C> {
    /// Position of rectangle's center
    pub center: Point2<R>,

    /// Rectangle's size
    pub size: Size2<R>,

    /// Radius of corners
    pub corner_radius: Option<CornerRadius<R>>,

    /// Color to fill rectangle with
    pub fill: C,

    /// Color for border
    pub stroke: C,

    /// Width of the border
    pub weight: R,

    /// Angle of rotation
    pub rotation: R,
}

impl<R, C> ShapeFrom<Rectangle<R, C>, R, C> for Rectangle<R, C> {
    fn shape_from(data: Rectangle<R, C>, _context: &Context<R, C>) -> Shape<R, C> {
        Shape::Rectangle(data)
    }
}

impl<X, Y, W, H, R, C> ShapeFrom<(X, Y, W, H), R, C> for Rectangle<R, C>
where
    X: ToReal<R>,
    Y: ToReal<R>,
    W: ToReal<R>,
    H: ToReal<R>,
    R: Real,
    C: Copy,
{
    fn shape_from(data: (X, Y, W, H), context: &Context<R, C>) -> Shape<R, C> {
        let (x, y, width, height) = data;

        let (center, size) = from_mode(
            x.to_real(),
            y.to_real(),
            width.to_real(),
            height.to_real(),
            context.rectangle_mode,
        );

        // todo!("Set generics to full names");
        // let (center, size) = apply_transform(center, size, context);

        Shape::Rectangle(Rectangle {
            center,
            size,
            corner_radius: None,
            fill: context.fill,
            stroke: context.stroke,
            weight: context.weight,
            rotation: context.rotation,
        })
    }
}

impl<X, Y, W, H, D, R, C> ShapeFrom<(X, Y, W, H, D), R, C> for Rectangle<R, C>
where
    X: ToReal<R>,
    Y: ToReal<R>,
    W: ToReal<R>,
    H: ToReal<R>,
    D: ToReal<R>,
    R: Real,
    C: Copy,
{
    fn shape_from(data: (X, Y, W, H, D), context: &Context<R, C>) -> Shape<R, C> {
        let (x, y, width, height, radius) = data;

        let (center, size) = from_mode(
            x.to_real(),
            y.to_real(),
            width.to_real(),
            height.to_real(),
            context.rectangle_mode,
        );

        Shape::Rectangle(Rectangle {
            center,
            size,
            corner_radius: Some(CornerRadius::Uniform(radius.to_real())),
            fill: context.fill,
            stroke: context.stroke,
            weight: context.weight,
            rotation: context.rotation,
        })
    }
}

impl<X, Y, W, H, TL, TR, BR, BL, R, C> ShapeFrom<(X, Y, W, H, TL, TR, BR, BL), R, C>
    for Rectangle<R, C>
where
    X: ToReal<R>,
    Y: ToReal<R>,
    W: ToReal<R>,
    H: ToReal<R>,
    TL: ToReal<R>,
    TR: ToReal<R>,
    BR: ToReal<R>,
    BL: ToReal<R>,
    R: Real,
    C: Copy,
{
    fn shape_from(data: (X, Y, W, H, TL, TR, BR, BL), context: &Context<R, C>) -> Shape<R, C> {
        let (x, y, width, height, top_left, top_right, bottom_right, bottom_left) = data;

        let (center, size) = from_mode(
            x.to_real(),
            y.to_real(),
            width.to_real(),
            height.to_real(),
            context.rectangle_mode,
        );

        Shape::Rectangle(Rectangle {
            center,
            size,
            corner_radius: Some(CornerRadius::PerCorner {
                top_left: top_left.to_real(),
                top_right: top_right.to_real(),
                bottom_right: bottom_right.to_real(),
                bottom_left: bottom_left.to_real(),
            }),
            fill: context.fill,
            stroke: context.stroke,
            weight: context.weight,
            rotation: context.rotation,
        })
    }
}
