use crate::geometry::*;
use crate::real::Real;

/// Mode for interpeting the position of a shape
#[derive(Copy, Clone, Debug)]
pub enum Mode {
    /// Read as (x_left, y_top, width, height)
    Corner,

    /// Read as (x_left, y_top, x_right, y_bottom)
    Corners,

    /// Read as (x_center, y_center, width, height)
    Center,

    /// Read as (x_center, y_center, width / 2, height / 2)
    Radius,
}

pub(crate) fn from_mode<R>(a: R, b: R, c: R, d: R, mode: Mode) -> (Point2<R>, Size2<R>)
where
    R: Real,
{
    use Mode::*;
    match mode {
        Corner => (
            Point2 {
                x: a + c.div2(),
                y: b + d.div2(),
            },
            Size2 {
                width: c,
                height: d,
            },
        ),
        Corners => (
            Point2 {
                x: a + (c - a).div2(),
                y: b + (d - b).div2(),
            },
            Size2 {
                width: c - a,
                height: d - b,
            },
        ),
        Center => (
            Point2 { x: a, y: b },
            Size2 {
                width: c,
                height: d,
            },
        ),
        Radius => (
            Point2 { x: a, y: b },
            Size2 {
                width: c.mul2(),
                height: d.mul2(),
            },
        ),
    }
}
