use super::context::Context;
use super::mode::*;
use super::shape::*;
use crate::geometry::*;
use crate::real::*;

/// A circle
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub struct Circle<R, C> {
    /// Position of circle's center
    pub center: Point2<R>,

    /// Circle's diameter
    pub diameter: R,

    /// Color to fill circle with
    pub fill: C,

    /// Color to use for the border
    pub stroke: C,

    /// Width of the border
    pub weight: R,
}

impl<R, C> ShapeFrom<Circle<R, C>, R, C> for Circle<R, C> {
    fn shape_from(data: Circle<R, C>, _context: &Context<R, C>) -> Shape<R, C> {
        Shape::Circle(data)
    }
}

impl<X, Y, D, R, C> ShapeFrom<(X, Y, D), R, C> for Circle<R, C>
where
    X: ToReal<R>,
    Y: ToReal<R>,
    D: ToReal<R>,
    R: Real,
    C: Copy,
{
    fn shape_from(data: (X, Y, D), context: &Context<R, C>) -> Shape<R, C> {
        let (x, y, diameter) = data;

        let diameter = diameter.to_real();

        let (center, size) = from_mode(
            x.to_real(),
            y.to_real(),
            diameter,
            diameter,
            context.ellipse_mode,
        );

        Shape::Circle(Circle {
            center,
            diameter: size.width,
            fill: context.fill,
            stroke: context.stroke,
            weight: context.weight,
        })
    }
}
