//! Draw on page.

mod circle;
mod context;
mod mode;
mod rectangle;
mod shape;

pub use circle::*;
pub use context::*;
pub use mode::*;
pub use rectangle::*;
pub use shape::*;

use super::aspect;
use crate::geometry::*;
use crate::real::ToReal;

aspect! {
    pub trait Handler {
        /// Redraw page.
        fn draw();
    }

    impl SketchExt for <Sketch> {
        /// Get the width of the page.
        fn width(&self) -> types::Real<Self> {
            self.as_part().size.width
        }

        /// Get the height of the page.
        fn height(&self) -> types::Real<Self> {
            self.as_part().size.height
        }

        /// Get the size of the page.
        fn size(&self) -> Size2<types::Real<Self>> {
            self.as_part().size
        }

        /// Get the rectangle mode.
        fn rect_mode(&mut self, mode: Mode) {
            self.as_part_mut().context.rectangle_mode = mode;
        }

        /// Get the ellipse mode.
        fn ellipse_mode(&mut self, mode: Mode) {
            self.as_part_mut().context.ellipse_mode = mode;
        }

        /// Set the fill color for shapes.
        fn fill[C: Into<types::Color<Self>>](&mut self, color: C) {
            self.as_part_mut().context.fill = color.into();
        }

        /// Set the stroke color for shapes.
        fn stroke[C: Into<types::Color<Self>>](&mut self, color: C) {
            self.as_part_mut().context.stroke = color.into();
        }

        /// Set the stroke weight for shapes.
        fn weight[R: ToReal<types::Real<Self>>](&mut self, width: R) {
            self.as_part_mut().context.weight = width.to_real();
        }

        /// Set the background color of the page.
        ///
        /// This also clears all previously drawn shapes.
        fn background[C: Into<types::Color<Self>>](&mut self, color: C) {
            let cut = self.as_part_mut();
            cut.shapes.clear();
            cut.background = color.into();
        }

        /// Draw a circle.
        fn circle[S: IntoShape<Circle<types::Real<Self>, types::Color<Self>>, types::Real<Self>, types::Color<Self>>](&mut self, circle: S) {
            let shape = circle.into_shape(&self.as_part().context);
            self.as_part_mut().shapes.push(shape);
        }

        /// Draw a rectangle.
        fn rect[S: IntoShape<Rectangle<types::Real<Self>, types::Color<Self>>, types::Real<Self>, types::Color<Self>>](&mut self, rectangle: S) {
            let shape = rectangle.into_shape(&self.as_part().context);
            self.as_part_mut().shapes.push(shape);
        }

        /// set the frame rate.
        ///
        /// The environment may not be able to provide the requested rate.
        fn frame_rate[R: ToReal<types::Real<Self>>](&mut self, rate: R) {
            self.as_part_mut().frame_rate = rate.to_real()
        }

        // fn rotate[R: ToReal<types::Real<Self>>](&mut self, angle: R) {
        //     let data = self.as_part_mut();
        //     let t0 = shape::Transform2D::new(data.context.translation, data.context.rotation);
        //     let t1 = shape::Transform2D::new(Point { x: 0., y: 0. }, angle.to_real());
        //     let t2 = t0.mul(&t1);
        //     // let t2 = t1.mul(&t0);
        //     // println!("{} {:?}", t1.rotation(), t1.translation());
        //     // println!("{:?}", t1);
        //     data.context.rotation = t2.rotation();
        //     data.context.translation = t2.translation();
        // }
        //
        // fn translate<P: Into<Point>>(&mut self, point: P) {
        //     // self.cut_mut().context.translation += Point { x, y };
        //     let data = self.as_part_mut();
        //     let t0 = shape::Transform2D::new(data.context.translation, data.context.rotation);
        //     let t1 = shape::Transform2D::new(point.into(), 0.);
        //     let t2 = t0.mul(&t1);
        //     // let t2 = t1.mul(&t0);
        //     // println!("{} {:?}", t1.rotation(), t1.translation());
        //     // println!("{:?}", t1);
        //     data.context.rotation = t2.rotation();
        //     data.context.translation = t2.translation();
        // }
        //
        // fn scale<N: ToFloat>(&mut self, amount: N) {
        //     // TODO: fix how this works with rotate and translate
        //     self.as_part_mut().context.scale *= amount.to_float();
        // }
        //
        // fn push_matrix(&mut self) {
        //     let data = self.as_part_mut();
        //     let Context {
        //         translation,
        //         rotation,
        //         scale,
        //         ..
        //     } = data.context;
        //     data.transforms.push((translation, rotation, scale));
        // }
        //
        // fn pop_matrix(&mut self) {
        //     let data = self.as_part_mut();
        //     if let Some((translation, rotation, scale)) = data.transforms.pop() {
        //         data.context.translation = translation;
        //         data.context.rotation = rotation;
        //         data.context.scale = scale;
        //     }
        // }
    }

    pub mod env_api {
        {
            use crate::compose::TryAsPart;
            use crate::geometry::*;
            use super::DrawDefaults;
            use super::context::Context;
            use alloc::vec::Vec;
            use super::shape::Shape;
        }

        pub trait Types: DrawDefaults[DefaultReal=Self::Real, DefaultColor=Self::Color] {
            /// Type for real numbers.
            type Real: Copy;

            /// Type for colors.
            type Color: Clone;
        }

        pub enum Events<T> {
            /// Event to redraw
            Draw,
        }

        pub struct Data<T> {
            pub(super) frame_rate: T::Real,
            pub(super) shapes: Vec<Shape<T::Real, T::Color>>,
            pub(super) background: T::Color,
            pub(super) size: Size2<T::Real>,
            pub(super) context: Context<T::Real, T::Color>,
            // transforms: Vec<(Point, Float, Float)>,
        }

        impl Data<T> {
            /// Create new data for aspect.
            pub fn new() -> Self {
                Self {
                    _types_phantom: core::marker::PhantomData,
                    size: T::default_size(),
                    context: T::default_context(),
                    shapes: Vec::new(),
                    background: T::default_background(),
                    frame_rate: T::default_frame_rate(),
                }
            }

            /// Set the page's size.
            pub fn set_size(&mut self, size: Size2<T::Real>) {
                self.size = size;
            }

            /// Get the frame rate.
            pub fn frame_rate(&self) -> T::Real {
                self.frame_rate
            }

            /// Get the background color.
            pub fn background_color(&self) -> T::Color {
                self.background.clone()
            }

            /// Get drawn frame.
            pub fn frame_mut(&mut self) -> &mut Vec<Shape<T::Real, T::Color>> {
                &mut self.shapes
            }
        }

        impl Handle for <Sketch> {
            fn handle(&mut self, event: _) {
                if event.try_as_part().is_some() {
                    // let context = &mut self.as_part_mut().context;
                    // context.scale = 1.;
                    // context.rotation = 0.;
                    // context.translation = Point { x: 0., y: 0. };
                    //
                    self.draw();
                }
            }
        }
    }
}

/// Default values for draw aspect
pub trait DrawDefaults {
    /// Type for real numbers
    type DefaultReal;

    /// Type for colors
    type DefaultColor;

    /// Get the default size of page.
    fn default_size() -> Size2<Self::DefaultReal>;

    /// Get the default context for a page.
    fn default_context() -> Context<Self::DefaultReal, Self::DefaultColor>;

    /// Get the default background color for a page.
    fn default_background() -> Self::DefaultColor;

    /// Get the default frame rate for a page.
    fn default_frame_rate() -> Self::DefaultReal;
}
