//! Built-in aspects of environments.
//!
//! Each environment implements a set of aspects.
//!
//! Basic structure of an aspect:
//!
//! user api
//! - struct Event
//! - trait Handler
//! - trait Ext
//!
//! env api
//! - enum EventTypes
//! - struct Data
//! - impl Handle
//!
//! ### `struct AspectNameEvent`
//! This is an empty struct that acts as a marker for traits to signal the aspect's events are
//! setup to be handled.
//!
//! ### `trait AspectName`
//! This trait is what the user uses to handle the events of the aspect.
//! The methods of this trait act as callbacks for the environment.
//!
//! ### `trait AspectNameExt`
//! This trait provides an easy to use API for the user to access/modify the aspect.
//!
//! ### `enum env_api::AspectNameEventTypes`
//! This enum contains all the event types the aspect uses. These events will be generated by the
//! environment and then handled by the aspect handler.
//!
//! ### `struct env_api::AspectNameData`
//! Data the aspect needs to keep around to function. This allows aspects to be stateful. The page
//! inside the sketch will contain this data for environments that support it.
//!
//! ### `impl Handle<AspectNameEvent>`
//! Handler for the events of the aspect. This impl is able to call the callback methods of the
//! main trait and update info in the aspect's data on the sketch's page.

#[cfg(feature = "aspect_update")]
pub mod update;

#[cfg(feature = "aspect_draw")]
pub mod draw;

#[cfg(feature = "aspect_mouse")]
pub mod mouse;

#[cfg(feature = "aspect_keyboard")]
pub mod keyboard;

use crate::{env::EventOf, Sketch};

/// Marker trait to signal event handler is attached.
pub trait HandlerImplementedFor<E> {}

/// Trait for sketch to handle an event.
pub trait Handle<E>
where
    Self: Sketch,
{
    /// Handle an event from the environment.
    fn handle(&mut self, event: &EventOf<Self::Env>);
}

/// Macro to generate module for an aspect.
#[macro_export]
macro_rules! aspect {
    {
        pub trait Handler {
            $(
                $(#[$handler_meta:meta])*
                fn $handler_name:ident ($($handler_args:tt)*);
            )*
        }

        impl SketchExt for <Sketch> {
            $(
                $(#[$ext_meta:meta])*
                fn $ext_name:ident $([$($ext_generic:tt)*])? ($($ext_args:tt)*) $(-> $ext_return:ty)? {
                    $($ext_code:tt)*
                }
            )*
        }

        pub mod env_api {
            {$($env_api_use:tt)*}

            $(Page: $extra_bound:ident$([$($extra_generic_bound:tt)+])?$(+ $extra_bound_extra:ident$([$($extra_generic_bound_extra:tt)+])?)*;)?

            pub trait Types$(: $types_bound:ident$([$($types_generic_bound:tt)+])?$(+ $types_bound_extra:ident$([$($types_generic_bound_extra:tt)+])?)*)? {
                $($(#[$($meta_meta:tt)*])? type $meta_name:ident$(: $meta_bound:ident$(+ $meta_bound_extra:ident)*)?;)*
            }

            pub enum Events<$event_meta_generic:ident> {
                $($events_fields:tt)*
            }

            pub struct Data<$meta_generic:ident> {
                $($data_fields:tt)*
            }

            impl Data<$meta_generic_impl:ident> {
                $($data_code:tt)*
            }

            impl Handle for <Sketch> {
                fn handle(&mut $handle_self:ident, $handle_event:ident: _) {
                    $($handle_code:tt)*
                }
            }
        }
    } => {
        use associated::*;

        /// Handler for aspect's events.
        pub trait Handler
        where
            Self: env_api::EnvAssociatedTypes + $crate::aspects::HandlerImplementedFor<Event>
        {
            #![allow(unused_variables)]

            $(
                $(#[$handler_meta])*
                fn $handler_name (&mut self, $($handler_args)*) {}
            )*
        }

        /// Extension methods provided by aspect.
        pub trait SketchExt: env_api::EnvAssociatedTypes {
            $(
                $(#[$ext_meta])*
                fn $ext_name $(<$($ext_generic)*>)? ($($ext_args)*) $(-> $ext_return)?;
            )*
        }

        /// Implemented on any type that contains the aspect's data.
        impl<T> SketchExt for T
        where
            Self: $crate::compose::AsPart<PageDataForEnv<Self>> + env_api::EnvAssociatedTypes,
        {
            $(
                fn $ext_name $(<$($ext_generic)*>)? ($($ext_args)*) $(-> $ext_return)? {
                    $($ext_code)*
                }
            )*
        }

        /// Marker for events generated by this aspect.
        pub enum Event {}

        /// Associated types for the aspect.
        ///
        /// These types are set by the underlying environment.
        pub mod associated {
            use super::env_api;

            /// Helper type to get associated types for aspect from environment.
            pub type AssociatedTypesForEnv<T> = <T as env_api::EnvAssociatedTypes>::AssociatedTypesHolder;

            /// Helper type to get aspect's data based on environment.
            pub type PageDataForEnv<T> = env_api::PageData<AssociatedTypesForEnv<T>>;

            /// Helper type to get aspect's events based on environment.
            pub type EventsForEnv<T> = env_api::Events<AssociatedTypesForEnv<T>>;

            /// Helper types to get associated types for aspect.
            pub mod types {
                $(
                    /// Helper type to get specific associated type for aspect.
                    pub type $meta_name<T> = <super::AssociatedTypesForEnv<T> as super::env_api::AssociatedTypes>::$meta_name;
                )*
            }
        }

        /// API for environments to implement this aspect.
        pub mod env_api {
            $($env_api_use)*

            /// Types associated with the aspect the environment needs to set.
            pub trait AssociatedTypes$(: $types_bound$(<$($types_generic_bound)+>)?$(+ $types_bound_extra$(<$($types_generic_bound_extra)+>)?)*)? {
                $($(#[$($meta_meta)*])? type $meta_name$(: $meta_bound$(+ $meta_bound_extra)*)?;)*
            }

            /// Helper trait to get associated types for environment.
            pub trait EnvAssociatedTypes {
                /// Type with the associated types defined on it.
                type AssociatedTypesHolder: AssociatedTypes;
            }

            impl<S> EnvAssociatedTypes for S where S: $crate::Sketch, S::Env: EnvAssociatedTypes {
                type AssociatedTypesHolder = <S::Env as EnvAssociatedTypes>::AssociatedTypesHolder;
            }

            /// Events the aspect generates/handles.
            pub enum Events<$event_meta_generic: AssociatedTypes> {
                #[doc(hidden)]
                _MetaPhantom(core::convert::Infallible, core::marker::PhantomData<$event_meta_generic>),

                $($events_fields)*
            }

            /// Data the aspect needs to function.
            ///
            /// This data is kept in the environment's page type.
            pub struct PageData<$meta_generic: AssociatedTypes> {
                #[doc(hidden)]
                _types_phantom: core::marker::PhantomData<$meta_generic>,

                $($data_fields)*
            }

            impl<$meta_generic_impl: AssociatedTypes> PageData<$meta_generic_impl> {
                $($data_code)*
            }

            /// Needed for environments to create their page type.
            impl<$meta_generic_impl: AssociatedTypes> Default for PageData<$meta_generic_impl> {
                fn default() -> Self {
                    Self::new()
                }
            }

            /// Handles the aspect's events.
            impl<T> $crate::aspects::Handle<super::Event> for T
            where
                Self: $crate::Sketch + super::Handler + $crate::compose::AsPart<super::associated::PageDataForEnv<Self>> + EnvAssociatedTypes,
                $crate::env::EventOf<Self::Env>: $crate::compose::TryAsPart<super::associated::EventsForEnv<Self>>,
                $($crate::env::PageOf<Self::Env>: $extra_bound$(<$($extra_generic_bound)+>)?$(+ $extra_bound_extra$(<$($extra_generic_bound_extra)+>)?)*,)?
            {
                fn handle(&mut $handle_self, $handle_event: &$crate::env::EventOf<Self::Env>) {
                    $($handle_code)*
                }
            }
        }
    };
}

#[allow(unused_imports)] // remove this use by changing use paths
pub(crate) use aspect;
