// Copyright 2022 Daniel Mowitz
//
// This file is part of sing.
//
// sing is free software: you can redistribute it and/or modify it 
// under the terms of the GNU Affero General Public License 
// as published by the Free Software Foundation, 
// either version 3 of the License, or (at your option) any later version.
//
// sing is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
// See the GNU Affero General Public License for more details.
// 
// You should have received a copy of the GNU Affero General Public License
//  along with sing. If not, see <https://www.gnu.org/licenses/>. 

use core::fmt::Debug;
use std::{error::Error, collections::HashMap};

use proc_macro2::TokenStream;
use serde::{Serialize, Deserialize};
pub use serde_wrapper::TokenStreamWrapper;

mod serde_wrapper;

/// Represents a trait as Function names associated with tokenstreams.
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct TraitProfile{
	 functions: HashMap<String, TokenStreamWrapper>,
}	  

impl TraitProfile{
	 pub fn new(funs: HashMap<String, TokenStream>) -> Self {
		  let mut inner = HashMap::new();

		  for (name, stream) in funs {
				inner.insert(name, TokenStreamWrapper::new(stream));
		  }

		  Self{
				functions: inner
		  }
	 }

	 pub fn get_funs(&self) -> Result<HashMap<String, TokenStream>, Box<dyn Error>> {
		  let wrappers = self.functions.clone();
		  let mut funs = HashMap::new();

		  for (name, wrapper) in wrappers {
				funs.insert(name, wrapper.get_inner()?);
		  }

		  Ok(funs)
	 }
}

/// Represents the state of a function name,
/// with it either being only present once or existing multiple times.
#[derive(Debug, Clone)]
pub enum DeduplicatedFunctionProfile {
	 Single(String, TokenStreamWrapper),
	 Multiple(Vec<String>),
}

/// Needs to be implemented by all structs that are used to represent
/// function calls that are sent through a sing I/O stream
pub trait TraitCallMessage {
	 /// The representation of rust data structures that is used internally to store the parameters.
	 /// Usually also the type used by the I/O stream.
	 type Representation;

	 /// Returns the called functions name.
	 fn get_fun_name(&self) -> String;

	 /// Returns the name of the trait the function is called from.
	 fn get_trait_name(&self) -> Option<String>;

	 /// Returns the parameters (or arguments) of the function call.
	 fn get_params(&self) -> Vec<Self::Representation>;

	 /// Replaces the parameters currently stored in this TraitCallMessage object.
	 fn new_params(&mut self, p: Vec<Self::Representation>);
}

#[cfg(test)]
mod tests {

	 #[test]
    fn it_works() {
        let result = 2 + 2;
        assert_eq!(result, 4);
    }

}
