// Copyright 2022 Daniel Mowitz
//
// This file is part of sing.
//
// sing is free software: you can redistribute it and/or modify it 
// under the terms of the GNU Affero General Public License 
// as published by the Free Software Foundation, 
// either version 3 of the License, or (at your option) any later version.
//
// sing is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
// See the GNU Affero General Public License for more details.
// 
// You should have received a copy of the GNU Affero General Public License
//  along with sing. If not, see <https://www.gnu.org/licenses/>. 

use std::fmt;
use sing_util::TraitCallMessage;
use serde::{Serialize, Deserialize};

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct CallObj {
	 trait_string: Option<String>,
	 fun_string: String,
	 index: usize,
	 data: Vec<String>,
}

/// Default message representation for the sing_loop! macro.
impl CallObj {
    pub fn new((trait_string, fun_string, index): (Option<String>, String, usize), data: Vec<String>) -> Self {
		  Self{
				trait_string,
				fun_string,
				index,
				data,
		  }
	 }
}

impl TraitCallMessage for CallObj {
	 type Representation = String;

	 fn get_fun_name(&self) -> String {
		  self.fun_string.clone()
	 }

	 fn get_trait_name(&self) -> Option<String> {
		  self.trait_string.clone()
	 }

	 fn get_params(&self) -> Vec<Self::Representation> {
		  self.data.clone()
	 }

	 fn new_params(&mut self, p: Vec<Self::Representation>) {
		  self.data = p;
	 }
}

impl fmt::Display for CallObj {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		  let mut repr = String::new();

		  match &self.trait_string {
				Some(tstr) => {
					 repr.push_str(tstr);
					 repr.push('>');
				},
				None => {}
		  }

		  repr.push_str(&self.fun_string);

		  if self.index != usize::MAX {
				repr.push('>');
				repr.push_str(&self.index.to_string());
		  }

		  for element in &self.data {
				repr.push(' ');
				repr.push_str(element);
		  }

		  write!(f, "{}", repr)
	 }
}
