// Copyright 2022 Daniel Mowitz
//
// This file is part of sing.
//
// sing is free software: you can redistribute it and/or modify it 
// under the terms of the GNU Affero General Public License 
// as published by the Free Software Foundation, 
// either version 3 of the License, or (at your option) any later version.
//
// sing is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
// See the GNU Affero General Public License for more details.
// 
// You should have received a copy of the GNU Affero General Public License
//  along with sing. If not, see <https://www.gnu.org/licenses/>. 

use proc_macro::TokenStream;
use proc_macro2::{Span, TokenStream as TokenStream2};
use quote::{quote, spanned::Spanned, ToTokens};
use sing_util::{DeduplicatedFunctionProfile, TokenStreamWrapper, TraitProfile};
use std::{
    collections::HashMap,
    env,
    error::Error,
    fs,
    io::{Read, Write},
};
use syn::{
    bracketed,
    parse::{Parse, ParseStream},
    parse2, parse_macro_input, Arm, AttributeArgs, FnArg, Ident, ImplItem, ItemImpl, LitStr, Meta,
    NestedMeta, Path, Signature, Stmt, Token,
};

extern crate sing_parse;
extern crate proc_macro;
extern crate sing_util;

mod keyword;

/// Add the trait in this impl block to sings trait store.
///
/// The impl block needs to contain all functions as usual.
/// After adding a trait, it can be used by the sing_loop macro.
///
/// ## Example
/// ```
/// trait FruitTree {
///    fn shake(&self, shakes: u32) -> Vec<Fruit>;
/// }
///
/// #[derive(Debug, Serialize, Deserialize)]
/// struct Fruit {
///    fruit_type: String,
/// }
///
/// struct BananaTree;
///
/// #[sing_add_trait()]
/// impl FruitTree for BananaTree {
///    fn shake(&self, shakes: u32) -> Vec<Fruit>{
///       let mut out = vec!();
///
///       for _ in 0..shakes {
///          out.push(Fruit{fruit_type: String::from("Banana")});
///       }
///                  
///       out
///    }
/// }
/// ```
#[proc_macro_attribute]
pub fn sing_add_trait(annotated_item: TokenStream, input: TokenStream) -> TokenStream {
    let input = parse_macro_input!(input as ItemImpl);
    let annotated_item = parse_macro_input!(annotated_item as AttributeArgs);

    let output = add_trait_inner(input, annotated_item);

    TokenStream::from(output)
}

/// Used internally by the sing_add_trait macro.
fn add_trait_inner(input: ItemImpl, annotated_item: Vec<NestedMeta>) -> TokenStream2 {
    let trait_name = match &input.trait_ {
        Some((_b, p, _f)) => p.into_token_stream().to_string(),
        None => {
            compile_error_panic(
                input.__span(),
                "This attribute can only be used on trait impl blocks.",
            );
            String::from("")
        }
    };

    // collect all the functions of this block into a hashmap with name -> function mapping
    let mut trait_functions = HashMap::new();
    for fun in &input.items {
        match fun {
            ImplItem::Method(m) => {
                trait_functions.insert(m.sig.ident.to_string(), m.sig.to_token_stream());
            }
            _ => {
                compile_error_panic(
                    input.__span(),
                    "Found unexpected item that is not a method.",
                );
            }
        }
    }

    // Get the currently saved list of traits
    let mut trait_state = load_trait_state().unwrap();

    // Add this trait to it
    let profile = TraitProfile::new(trait_functions);
    if trait_state.contains_key(trait_name.as_str()) {
        let err = format!("Trait {} is implemented twice!", trait_name);
        syn::Error::new(input.__span(), err.as_str()).to_compile_error();
    } else {
        trait_state.insert(trait_name, profile);
    }

    save_trait_state(trait_state).unwrap();

    let gobble_ident = parse2::<Ident>(quote! {gobble}).unwrap();

    let mut gobble_flag = false;

    for flag in annotated_item {
        if let NestedMeta::Meta(Meta::Path(p)) = flag {
            if p.get_ident().unwrap() == &gobble_ident {
                gobble_flag = true;
            }
        }
	 }

    if gobble_flag {
        quote! {}
    } else {
        quote! { #input }
    }
}

/// Creates a main loop that parses stdin (or the given input stream),
/// evaluates the called function and returns the result.
///
/// ## Usage
///
/// A call of this macro looks like:
/// ```sing_loop!(trait_object, ser_de <, msg: [msg_type, arg_type]> <, ioe: [input, output, error]>)```
/// where all items in angle brackets are optional.
///
/// with the arguments:
/// - trait_object: An instance of a struct that implements all desired traits.
/// - argument_ser_de: Serialization and deserialization functions. It either looks like `[equal, ser_fun, de_fun]` in which case both messages and arguments are serialized an deserialized by the same functions or `[different, arg_ser_fun, arg_de_fun <, msg_ser_fun, msg_de_fun>` in which case different functions are used. 
/// - msg_type: A struct that implements [`sing_util::TraitCallMessage`] and should be used as the message type.
/// - arg_type: The type to which the arguments of function calls should be serialized to.
/// - input: The input Stream. Needs to implement [`std::io::BufRead`].
/// - output: The output Stream. Needs to implement [`std::io::Write`].
/// - error: The error output Stream. Needs to implement [`std::io::Write`].
///
/// `msg`, `ioe`, `equal` and `different` are keywords and need to be used literally in the call.
///
/// The optional parameters have the following defaults:
/// - msg_ser_fun: [`sing_parse::callobj_to_string`]
/// - msg_de_fun: [`sing_parse::callobj_to_string`]
/// - msg_type: [`sing_parse::CallObj`]
/// - arg_type: [`String`]
/// - input: [`std::io::StdinLock`]
/// - output: [`std::io::StdoutLock`]
/// - error: [`std::io::StderrLock`]
///
/// ## Example
/// ```
///    let tree = BananaTree {};
///         
///    sing_loop!(tree, [different, ron::to_string, ron::from_str],);
/// ```
#[proc_macro]
pub fn sing_loop(input: TokenStream) -> TokenStream {
    let span = TokenStream2::from(input.clone()).__span();
    let input = parse_macro_input!(input as LoopParams);

    let output = match loop_inner(input, span) {
        Ok(s) => s,
        Err(e) => {
            compile_error_panic(span, &e.to_string());
            TokenStream2::new()
        }
    };

    TokenStream::from(output)
}

/// Used internally in the sing_loop macro
fn loop_inner(input: LoopParams, span: Span) -> Result<TokenStream2, Box<dyn Error>> {
    let LoopParams {
        trait_obj,
        arg_ser_fun,
        arg_de_fun,
        msg_ser_fun,
        msg_de_fun,
        reader,
        writer,
        err_writer,
        message_type,
		  arg_type,
    } = input;
	 
	 // If no msg serializer function is passed,
	 // use the sing message serializer function as default.
    let msg_ser_fun = match msg_ser_fun {
        Some(f) => f,
        None => {
            parse2::<Path>(quote! {
                 sing::callobj_to_string
            })?
        }
    };

	 // Do the same for the deserializer function.
    let msg_de_fun = match msg_de_fun {
        Some(f) => f,
        None => {
            parse2::<Path>(quote! {
                 sing::callobj_from_string
            })?
        }
    };
	 
    // In the case that no streams were passed,
    // use stdin, stdout and stderr.
    // First, create a vec for the let statements
    let mut initializations = vec![];

    // Then, push the statement and name for stdin…
    let reader = match reader {
        Some(r) => r,
        None => {
            initializations.push(parse2::<Stmt>(quote! {
                 let mut reader = std::io::stdin().lock();
            })?);

            parse2::<Ident>(quote! {
                 reader
            })?
        }
    };

    // for stdout…
    let writer = match writer {
        Some(w) => w,
        None => {
            initializations.push(parse2::<Stmt>(quote! {
                 let mut writer = std::io::stdout().lock();
            })?);

            parse2::<Ident>(quote! {
                 writer
            })?
        }
    };

    // and stderr
    let err_writer = match err_writer {
        Some(w) => w,
        None => {
            initializations.push(parse2::<Stmt>(quote! {
                 let mut err_writer = std::io::stderr().lock();
            })?);

            parse2::<Ident>(quote! {
                 err_writer
            })?
        }
    };

    let initializations = initializations;

    // If no message type is given, use default Type
    let message_type = match message_type {
        Some(m) => m,
        None => parse2::<Path>(quote! {
				sing::CallObj
        })?,
    };

	 // If no argument type is given, use default Type
    let arg_type = match arg_type {
        Some(a) => a,
        None => parse2::<Path>(quote! {
				String
        })?,
    };
	 
    // TODO: Make it possible to choose a subset of the provided traits
    let traits = load_trait_state().unwrap();

    // Create vector of functions for trait-less calls.
    // If a function name is used in more than one trait,
    // the user should be alerted when trying to call the ambiguos function.
    let mut funs = HashMap::new();
    for (t_name, t_funs) in &traits {
        for (f_name, fun) in t_funs.get_funs()? {
            if funs.contains_key(&f_name) {
                funs.insert(
                    f_name.clone(),
                    match &funs[&f_name] {
                        DeduplicatedFunctionProfile::Single(t, _) => {
                            DeduplicatedFunctionProfile::Multiple(vec![t_name.clone(), t.clone()])
                        }
                        DeduplicatedFunctionProfile::Multiple(v) => {
                            let mut v = v.clone();
                            v.push(t_name.clone());
                            DeduplicatedFunctionProfile::Multiple(v)
                        }
                    },
                );
            } else {
                funs.insert(
                    f_name.clone(),
                    DeduplicatedFunctionProfile::Single(f_name, TokenStreamWrapper::new(fun)),
                );
            }
        }
    }
	 
    // Construct the match arms for the functions
    let mut fun_arms = vec![];
    for (f_name, f_profile) in funs {
        // The function name is needed both as a literal string and
        // an ident keyword for the quote invocation.
        let fn_lit = LitStr::new(&f_name, span);
        let fn_ident = Ident::new(&f_name, span);
		  
        let arm = match f_profile {
            DeduplicatedFunctionProfile::Multiple(v) => {
                quote! {
                     #fn_lit => {
                           let traits = vec!( #( #v ),* );
                           return Err( format!("The function {} is found in the following traits: {}.
                Plese specify which traits function you want to use.",
                                                        #f_name,
                                                        traits,
                           ))
                     },
                }
            }
            DeduplicatedFunctionProfile::Single(_tr, s) => {
                get_fun_arm(
                    fn_lit,
                    fn_ident,
                    arg_ser_fun.clone(),
                    arg_de_fun.clone(),
                    trait_obj.clone(),
                    s.get_inner()?,
                )?
            }
        };

        let arm = parse2::<Arm>(arm)?;
        fun_arms.push(arm);
    }

    // Add two special cases for the function match statement
    fun_arms.push(parse2::<Arm>(quote! {
          "" => {
              return Err(String::from("No function name was given! Aborting."))
          },
    })?);

    fun_arms.push(parse2::<Arm>(quote! {
          s => {
               return Err(format!("The function {} is not known! Aborting.", s))
          },
    })?);
	 
    let mut trait_arms = vec![];

    // Create vector of match arms for calls with a trait name.
    // All functions of a given trait should be represented,
    // as well as The case of an empty function name, which should warn the user
    for (t_name, t_profile) in traits {
        // The trait name is needed both as a literal string and
        // an ident keyword for the quote invocation.
        let t_lit = LitStr::new(&t_name, span);

        // Create vector of all possible function calls for this trait
        let mut f_arms = vec![];
        for (f_name, f_profile) in t_profile.get_funs()? {
            let fn_lit = LitStr::new(&f_name, span);
            let fn_ident = Ident::new(&f_name, span);
				
            let f_arm = get_fun_arm(
                fn_lit,
                fn_ident,
                arg_ser_fun.clone(),
                arg_de_fun.clone(),
                trait_obj.clone(),
                f_profile,
				)?;
				
            f_arms.push(parse2::<Arm>(f_arm)?);
        }
		  
        // Add two special cases for the function match statement
        f_arms.push(parse2::<Arm>(quote! {
             "" => {
                 return Err(String::from("No function name was given! Aborting."))
             },
        })?);

        f_arms.push(
				parse2::<Arm>(quote! {
					 s => {
						  return Err(format!("The function {} was not found in the trait {}! Aborting.", s, #t_lit ))
					 }
				})?
		  );

        let t_arm = quote! {
              #t_lit => {
                   match message .get_fun_name().as_str() {
                        #( #f_arms )*
                   }
              }
        };

        trait_arms.push(parse2::<Arm>(t_arm)?);
    }

	 trait_arms.push(parse2::<Arm>(quote! {
          s => {
              return Err(format!("The trait {} is not known! Aborting.", s))
          },
    })?);
	 
    // Construct the finished Tokenstream
    // First, initialize the input output and error streams if needed
    // Then loop by getting a call from input,
    // processing it by the match statements constructed above and write to output.
    Ok(quote! {
		  use std::io::{BufRead, Write};
		  use sing::TraitCallMessage;
		  
        #( #initializations )*

		  // TODO: make this return the messages arg type vector
		  let mut sing_do_fn_call = |mut message: Box<dyn TraitCallMessage < Representation = #arg_type > >| -> Result<Vec< #arg_type >, std::string::String> {
				match message.get_trait_name() {
					 Some(n) => match n.as_str() {
						  #( #trait_arms )*
					 }
					 None => match message.get_fun_name().as_str() {
						  #( #fun_arms )*
					 }
				}
		  };
		  
		  let mut sing_loop_inner = |mut inner_reader: Box<dyn std::io::BufRead>, mut inner_writer: Box<dyn std::io::Write>, mut inner_err_writer: Box<dyn std::io::Write>| {
				loop {
					 let mut buf = String::new();
					 inner_reader.read_line(&mut buf);
					 let result: Result< #message_type, Box< dyn std::error::Error >> = #msg_de_fun (buf.clone());
					 
					 match result {
						  Ok(message) => {
								let mut message = message;

								match sing_do_fn_call(Box::new(message.clone())) {
									 Ok(res) => message.new_params(res),
									 Err(e) => inner_err_writer.write_all(e.as_bytes()).unwrap(),
								};
								
								match #msg_ser_fun(message) {
									 Ok(s) => {
										  inner_writer.write_all(s.as_bytes()).unwrap();
										  inner_writer.write_all("\n".as_bytes()).unwrap();
										  inner_writer.flush();
									 },
									 Err(e) => {
										  let message = format!("Could not encode call result: {}", e);
										  inner_err_writer.write_all(message.as_bytes()).unwrap();
									 },
								}
						  },
						  Err(e) => {
								let message = format!("Could not decode {} as trait message: {}", buf, e);
								inner_err_writer.write_all(message.as_bytes()).unwrap();
						  },
					 }
				}
		  };

		  sing_loop_inner(Box::new(#reader), Box::new(#writer), Box::new(#err_writer));		  
        
    })
}

/// Used for parsing the ser and de function syntax
enum EqDiff {
    Equal(keyword::equal),
    Diff(keyword::different),
}

/// Representation of the parsed syntax for the sing_loop macro
struct LoopParams {
    trait_obj: Ident,
    arg_ser_fun: Path,
    arg_de_fun: Path,
    msg_ser_fun: Option<Path>,
    msg_de_fun: Option<Path>,
    reader: Option<Ident>,
    writer: Option<Ident>,
    err_writer: Option<Ident>,
    message_type: Option<Path>,
	 arg_type: Option<Path>,
}

impl Parse for LoopParams {
    fn parse(input: ParseStream) -> syn::Result<Self> {
        let trait_obj = input.parse()?;
        input.parse::<Token![,]>()?;

        let content;
        bracketed!(content in input);
        let lh_inner = content.lookahead1();

        let eq_dif_token = if lh_inner.peek(keyword::equal) {
            EqDiff::Equal(content.parse()?)
        } else {
            EqDiff::Diff(content.parse()?)
        };
		  content.parse::<Token![,]>()?;

        let arg_ser_fun: Path = content.parse()?;
		  content.parse::<Token![,]>()?;
        let arg_de_fun: Path = content.parse()?;
		  
        let mut msg_ser_fun = None;
        let mut msg_de_fun = None;

        match eq_dif_token {
            EqDiff::Equal(_) => {
                msg_ser_fun = Some(arg_ser_fun.clone());
                msg_de_fun = Some(arg_de_fun.clone());
            }
            EqDiff::Diff(_) => {
                if !content.is_empty() {
						  content.parse::<Token![,]>()?;
                    msg_ser_fun = Some(content.parse()?);
						  content.parse::<Token![,]>()?;
                    msg_de_fun = Some(content.parse()?);
                }
            }
        }

        let lookahead = input.lookahead1();

        let mut reader = None;
        let mut writer = None;
        let mut err_writer = None;
        let mut message_type = None;
		  let mut arg_type = None;

        'parse: loop {
            input.parse::<Token![,]>()?;
            if input.is_empty() {
                break 'parse;
            }
            if lookahead.peek(keyword::msg) {
                input.parse::<keyword::msg>()?;
                input.parse::<Token![:]>()?;

					 let content;
                bracketed!(content in input);
                let m_a_type_err_generic = "The msg keyword expects two variables in the following order: [MessageType, ArgumentType],";

					 message_type = match content.parse() {
						  Ok(m) => Some(m),
						  Err(e) => {
								compile_error_panic(
									 e.span(),
									 format!(
										  "{}, but an error occured while parsing the MessageType: {}",
										  m_a_type_err_generic,
										  e,
									 ).as_str(),
								);
								None
						  }
					 };
					 
					 input.parse::<Token![,]>()?;

					 arg_type = match content.parse() {
						  Ok(a) => Some(a),
						  Err(e) => {
								compile_error_panic(
									 e.span(),
									 format!(
										  "{}, but an error occured while parsing the ArgumentType: {}",
										  m_a_type_err_generic,
										  e,
									 ).as_str(),
								);
								None
						  }
					 };
					 
            } else if lookahead.peek(keyword::ioe) {
                input.parse::<keyword::ioe>()?;
                input.parse::<Token![:]>()?;

                let content;
                bracketed!(content in input);
                let lh_inner = content.lookahead1();
                let ioe_err_generic = "The ioe keyword expects three variables in the following order: [Input, Output, Error],";

                if lh_inner.peek(Ident) {
                    let r = content.parse()?;
                    reader = Some(r);
                } else {
                    compile_error_panic(
                        content.span(),
                        format!(
                            "{}, but the Input variable was not supplied.",
                            ioe_err_generic
                        )
                        .as_str(),
                    );
                }

					 input.parse::<Token![,]>()?;

                if lh_inner.peek(Ident) {
                    let w = content.parse()?;
                    writer = Some(w);
                } else {
                    compile_error_panic(
                        content.span(),
                        format!(
                            "{}, but the Output variable was not supplied.",
                            ioe_err_generic
                        )
                        .as_str(),
                    );
                }

					 input.parse::<Token![,]>()?;

                if lh_inner.peek(Ident) {
                    let e = content.parse()?;
                    err_writer = Some(e);
                } else {
                    compile_error_panic(
                        content.span(),
                        format!(
                            "{}, but the Error variable was not supplied.",
                            ioe_err_generic
                        )
                        .as_str(),
                    );
                }
            }
            if input.is_empty() {
                break 'parse;
            }
        }		  

        Ok(Self {
            trait_obj,
            arg_ser_fun,
            arg_de_fun,
            msg_ser_fun,
            msg_de_fun,
            reader,
            writer,
            err_writer,
            message_type,
				arg_type,
        })
    }
}

/// Construct the match arm for a given trait function as a Tokenstream
fn get_fun_arm(
    fn_lit: LitStr,
    fn_ident: Ident,
    arg_ser_fun: Path,
    arg_de_fun: Path,
    trait_obj: Ident,
    s: TokenStream2,
) -> Result<TokenStream2, Box<dyn Error>> {
	 let sig = parse2::<Signature>(s)?;
    let mut args = vec![];
    let mut parse_lets = vec![];
	 let mut lets_index: usize = 0;
	 
    // Extract the types of the functions arguments and
    // produce let statements that create them from Strings using the de_fun function
    for a in sig.inputs {
        match a {
            // This is just a "self" value which does not need to be in the call
            FnArg::Receiver(_) => (),
            FnArg::Typed(t) => {
                let pat = *t.pat;
                let ty = t.ty;

                args.push(pat.clone());
					 
                parse_lets.push(parse2::<Stmt>(quote! {
                    let #pat = params[ #lets_index ].clone();
					 })?);

					 parse_lets.push(parse2::<Stmt>(quote! {
						  let #pat : #ty = match #arg_de_fun ( & #pat ) {
								Ok(p) => p,
								Err(e) => return Err(format!("Could not serialize input \"{}\" : {}", #pat, e)),
						  };
                })?);

					 lets_index += 1
            }
        }
    }

	 // TODO: check length of params before call
	 
    // Construct the arm, matching the function name,
    // parsing the parameters and calling the function
    Ok(quote! {
		  #fn_lit => {
		      let params = message.get_params();
			
            #( #parse_lets )*
				
            let result = #trait_obj . #fn_ident ( #( #args ),* );
				match #arg_ser_fun (&result) {
					 Ok(r) => {
						  return Ok(vec!(r))
					 }
					 Err(e) => {
						  return Err( format!("Could not serialize result of function call {}: {}", #fn_lit, e))
					 }
				}
        },
	 })
}

/// Retrieves the unique ID for this compilation run from the COMP_RUN_ID env variable
/// Creates a random ID if none is present and saves it
fn get_run_id() -> Result<usize, Box<dyn Error>> {
    match env::var("COMP_RUN_ID") {
        Ok(v) => Ok(v.parse()?),
        Err(_) => {
            let id: usize = rand::random();
            let id_str = format!("{}", id);
            env::set_var("COMP_RUN_ID", id_str);
            Ok(id)
        }
    }
}

/// Constructs and returns a path unique to this compilation run to save the trait state in
/// Utilizes the get_run_id function
fn get_state_file_path() -> Result<String, Box<dyn Error>> {
    let compilation_run_identifier = get_run_id()?;
    Ok(format!(
        "/tmp/sing-trait-store-{}",
        compilation_run_identifier
    ))
}

/// Saves the trait state constructed in the sing_add_trait macro
fn save_trait_state(
    map: HashMap<String, sing_util::TraitProfile>,
) -> Result<usize, Box<dyn Error>> {
    let state_file_path = get_state_file_path()?;
    let mut state_file = fs::File::create(state_file_path)?;

    let state_string = ron::to_string(&map)?;
    Ok(state_file.write(state_string.as_bytes())?)
}

/// Loads the trait state constructed in the sing_add_trait macro.
/// Creates new trait state file if none is found.
fn load_trait_state() -> Result<HashMap<String, sing_util::TraitProfile>, Box<dyn Error>> {
	 let state_file_path = get_state_file_path()?;
	 let state_file_path = std::path::Path::new(&state_file_path);

	 if !state_file_path.exists() {
		  return Ok(HashMap::new())
	 }
	 
    let mut state_file = fs::File::open(state_file_path)?;

    let mut state_string = String::from("");
    state_file.read_to_string(&mut state_string)?;
    Ok(ron::from_str(&state_string)?)
}

/// Shorthand for creating a compile error and then panicing.
/// Useful for testing macros outside of a compilation run.
fn compile_error_panic(span: Span, msg: &str) {
    syn::Error::new(span, msg).to_compile_error();
    panic!("{}", msg)
}

#[cfg(test)]
mod tests {
    use std::{collections::HashMap, str::FromStr};

    use proc_macro2::TokenStream;
    use sing_util::TraitProfile;

    use crate::{add_trait_inner, load_trait_state, save_trait_state};

    #[test]
    fn save_tokenstream() {
        let mut orig = HashMap::new();

        orig.insert(
            String::from("test"),
            TokenStream::from_str(
                "
fn test(a: Astruct, b: [Box<dyn Error>, String]) -> String {}
",
            )
            .unwrap(),
        );
        orig.insert(
            String::from("other_test"),
            TokenStream::from_str(
                "
fn other_test(a: Trait1 + Trait2) -> Result<String, Box<dyn Error>> {}
",
            )
            .unwrap(),
        );

        let orig_name = String::from("Test");
        let mut orig_state = HashMap::new();
        orig_state.insert(orig_name.clone(), TraitProfile::new(orig));

        save_trait_state(orig_state.clone()).unwrap();

        let new_state = load_trait_state().unwrap();

        assert_eq!(
            orig_state[&orig_name].get_funs().unwrap()["test"].to_string(),
            new_state[&orig_name].get_funs().unwrap()["test"].to_string()
        )
    }

    #[test]
    fn trait_call() {
        let call = TokenStream::from_str(
            "
impl FruitTree for BananaTree {
    fn shake(&self) -> Box<dyn Fruit> {
        Box::new(self.bananas.pop())
    }
}
		  ",
        )
        .unwrap();
        let call = syn::parse2(call).unwrap();

        add_trait_inner(call, vec![]);

        let new_state = load_trait_state().unwrap();

        assert_eq!(
            TokenStream::from_str(
                "
fn shake(&self) -> Box<dyn Fruit>
"
            )
            .unwrap()
            .to_string(),
            new_state["FruitTree"].get_funs().unwrap()["shake"].to_string()
        )
    }

    #[test]
    #[should_panic]
    fn non_trait_call() {
        let call = TokenStream::from_str(
            "
impl Banana {
    pub fn peel(&self) -> Food {
        self.inner.clone()
    }
}
		  ",
        )
        .unwrap();
        let call = syn::parse2(call).unwrap();

        add_trait_inner(call, vec![]);
    }
}
