pub(crate) struct Number {
	pub positive: bool,
	pub numerator: u128,
	pub denominator: u128,
}

impl Number {
	fn from_int(numerator: u128) -> Number {
		Number {
			positive: true,
			numerator,
			denominator: 1,
		}
	}

	fn from_frac(numerator: u128, denominator: u128) -> Number {
		Number {
			positive: true,
			numerator,
			denominator,
		}
	}
}

pub(crate) fn parse_num(mut s: &str) -> Result<Number, crate::de::DeserializeError> {
	let positive = if s.starts_with('-') {
		s = &s[1..];
		false
	} else {
		true
	};

	if s.is_empty() {
		return Err(crate::de::DeserializeError::Invalid("Can't parse \"\" as a number.".into()))
	}

	let bytes = s.as_bytes();

	let mut n = if bytes[0] == b'0' {
		match bytes.get(1) {
			None => Number::from_int(0),
			Some(b'b') => parse_num_base(2, &bytes[2..])?,
			Some(b'd') => parse_num_base(10, &bytes[2..])?,
			Some(b'o') => parse_num_base(8, &bytes[2..])?,
			Some(b'x') => parse_num_base(16, &bytes[2..])?,
			Some(_) => parse_num_base(10, &bytes[1..])?,
		}
	} else {
		parse_num_base(10, bytes)?
	};

	if n.numerator != 0 {
		n.positive = positive;
	}
	Ok(n)
}

fn parse_num_base(base: u8, mut bytes: &[u8]) -> Result<Number, crate::de::DeserializeError> {
	let mut n = 0;

	while let Some(c) = bytes.get(0) {
		match c {
			b'_' => {}
			b'0'..=b'9' |
			b'a'..=b'f' |
			b'A'..=b'F' => {
				if let Some(d) = (*c as char).to_digit(base.into()) {
					n *= base as u128;
					n += d as u128;
				} else {
					break
				}
			}
			b'.' => return parse_num_decimal(base, n, bytes),
			_ => {
				break
			}
		}
		bytes = &bytes[1..];
	}

	parse_num_exponent(base, Number::from_int(n), bytes)
}

fn parse_num_decimal(base: u8, mut numerator: u128, mut bytes: &[u8]) -> Result<Number, crate::de::DeserializeError> {
	let mut denominator = 1;

	while let Some(c) = bytes.get(0) {
		match c {
			b'_' => {}
			b'0'..=b'9' |
			b'a'..=b'f' |
			b'A'..=b'F' => {
				if let Some(d) = (*c as char).to_digit(base.into()) {
					numerator *= base as u128;
					numerator += d as u128;
					denominator *= base as u128;
				} else {
					break
				}
			}
			_ => {
				break
			}
		}
		bytes = &bytes[1..];
	}

	parse_num_exponent(base, Number::from_frac(numerator, denominator), bytes)
}

fn parse_num_exponent(base: u8, mut n: Number, mut bytes: &[u8]) -> Result<Number, crate::de::DeserializeError> {
	let s = unsafe { crate::from_utf8_unchecked(bytes) };

	let bindec = |s: &str, bin, dec| {
		match s.len() {
			0 => Ok(dec),
			1 if bytes[1] == b'i' => Ok(bin),
			_ => Err(crate::de::DeserializeError::Invalid(format!("Unexpected characters after number {:?}", s)))
		}
	};

	match s.chars().next() {
		Some('E') => n.numerator *= bindec(&s[1..], 2u128.pow(60), 10u128.pow(18))?,
		Some('P') => n.numerator *= bindec(&s[1..], 2u128.pow(50), 10u128.pow(15))?,
		Some('T') => n.numerator *= bindec(&s[1..], 2u128.pow(40), 10u128.pow(12))?,
		Some('G') => n.numerator *= bindec(&s[1..], 2u128.pow(30), 10u128.pow(9))?,
		Some('M') => n.numerator *= bindec(&s[1..], 2u128.pow(20), 10u128.pow(6))?,
		Some('K') => n.numerator *= bindec(&s[1..], 2u128.pow(10), 10u128.pow(3))?, // Accept this as kilo is the only >1 lower case.
		Some('k') => n.numerator *= bindec(&s[1..], 2u128.pow(10), 10u128.pow(3))?,
		Some('m') => n.denominator *= bindec(&s[1..], 2u128.pow(10), 10u128.pow(3))?,
		Some('u') => n.denominator *= bindec(&s[1..], 2u128.pow(20), 10u128.pow(6))?,
		Some('µ') => n.denominator *= bindec(&s['µ'.len_utf8()..], 2u128.pow(30), 10u128.pow(9))?,
		Some('n') => n.denominator *= bindec(&s[1..], 2u128.pow(40), 10u128.pow(12))?,
		Some('p') => n.denominator *= bindec(&s[1..], 2u128.pow(50), 10u128.pow(15))?,
		Some('f') => n.denominator *= bindec(&s[1..], 2u128.pow(60), 10u128.pow(18))?,
		Some('a') => n.denominator *= bindec(&s[1..], 2u128.pow(70), 10u128.pow(21))?,
		Some('e') => {
			if base != 10 {
				return Err(crate::de::DeserializeError::Invalid("e exponents are only supported in base 10".into()))
			}

			bytes = &bytes[1..];

			let positive = match bytes.get(0) {
				Some(b'+') => {
					bytes = &bytes[1..];
					true
				}
				Some(b'-') => {
					bytes = &bytes[1..];
					false
				}
				Some(_) => true,
				None => true,
			};

			let exp: u128 = unsafe { crate::from_utf8_unchecked(bytes) }.parse()?;
			if positive {
				n.numerator *= exp;
			} else {
				n.denominator *= exp;
			}
		}
		Some(_) => return Err(crate::de::DeserializeError::Invalid(format!("Unexpected characters after number {:?}", s))),
		None => {}
	}
	
	Ok(n)
}
