pub(crate) struct MapDeserializer<'a, T: buffered_reader::BufferedReader<()>> {
	pub root: &'a mut crate::Deserializer<T>,
	pub first: bool,
}

impl<'d, 'a, T: buffered_reader::BufferedReader<()>> serde::de::MapAccess<'d> for MapDeserializer<'a, T> {
	type Error = crate::DeserializeError;

	fn next_key_seed<K: serde::de::DeserializeSeed<'d>>(&mut self, seed: K) -> Result<Option<K::Value>, crate::DeserializeError> {
		let b = self.root.input.read_to(b':')?;

		let mut s = std::str::from_utf8(b)?;
		let mut indent_len = 0;
		while let Some(c) = s[indent_len..].chars().next() {
			if c == '\n' {
				let len = s.len() - 1;
				self.root.input.consume(indent_len + 1);
				let b = &self.root.input.buffer()[..len];
				s = unsafe { crate::from_utf8_unchecked(b) };
				indent_len = 0;
			} else if c == '#' {
				self.root.input.consume(1);
				self.root.input.drop_through(b"\n", true)?;
				// This does recheck the UTF-8 between the start of the comment and the next `:`. It could be avoided.
				return self.next_key_seed(seed)
			} else if c == ':' {
				return Ok(None)
			} else if c.is_whitespace() {
				indent_len += c.len_utf8();
			} else {
				break
			}
		}

		let (indent, key) = s.split_at(indent_len);

		if let Some(root) = &mut self.root.indent {
			if !self.first && root == indent {
				// Match.
			} else if root.starts_with(indent) {
				return Ok(None)
			} else if self.first && indent.starts_with(&*root) {
				root.push_str(&indent[root.len()..]);
				self.first = false;
			} else {
				return Err(crate::DeserializeError::Invalid(format!("Unmatched indent, was {:?} now {:?}", self.root.indent, indent)))
			}
		} else {
			self.root.indent = Some(indent.into());
			self.first = false;
		}

		if key.is_empty() {
			// EOF
			return Ok(None)
		}
		if !key.ends_with(':') {
			return Err(crate::DeserializeError::Invalid("Unexpected EOF, expecting :".into()))
		}
		let key = &key[..key.len()-1];
		let key_len = key.len();

		self.root.input.consume(indent_len);

		seed.deserialize(crate::KeyDeserializer {
			root: self.root,
			key_len,
		}).map(Some)
	}

	fn next_value_seed<K: serde::de::DeserializeSeed<'d>>(&mut self, seed: K) -> Result<K::Value, crate::DeserializeError> {
		seed.deserialize(&mut *self.root)
	}
}
