pub(crate) struct SeqDeserializer<'a, T: buffered_reader::BufferedReader<()>> {
	pub root: &'a mut crate::Deserializer<T>,
	pub first: bool,
}

impl<'d, 'a, T: buffered_reader::BufferedReader<()>> serde::de::SeqAccess<'d> for SeqDeserializer<'a, T> {
	type Error = crate::DeserializeError;

	fn next_element_seed<K: serde::de::DeserializeSeed<'d>>(&mut self, seed: K) -> Result<Option<K::Value>, crate::DeserializeError> {
		let mut b = self.root.input.buffer();
		let mut indent_end = 0;
		let mut eof = false;
		loop {
			if let Some((_i, i_end, c)) = bstr::ByteSlice::char_indices(&b[indent_end..]).next() {
				if c == '\n' {
					self.root.input.consume(indent_end + i_end);
					return self.next_element_seed(seed)
				} else if c == '#' {
					self.root.input.consume(indent_end + i_end);
					self.root.input.drop_through(b"\n", true)?;
					return self.next_element_seed(seed)
				} else if c.is_whitespace() {
					indent_end += i_end;
				} else {
					break
				}
			} else if eof {
				return Ok(None)
			} else {
				let req_size = b.len() * 2 + 1024;
				b = self.root.input.data(req_size)?;
				if b.len() < req_size {
					eof = true;
				}
			}
		}
		
		let b = &self.root.input.data(indent_end)?[..indent_end];
		let indent = unsafe { crate::from_utf8_unchecked(b) };

		if let Some(ref mut root) = &mut self.root.indent {
			if !self.first && root == indent {
				// Match.
			} else if root.starts_with(indent) {
				// Dedent, the list is over.
				return Ok(None)
			} else if self.first && indent.starts_with(root.as_str()) {
				// First indent, remember it.
				root.push_str(&indent[root.len()..]);
				self.first = false;
			} else {
				return Err(crate::DeserializeError::Invalid(format!("Unmatched indent, was {:?} now {:?}", self.root.indent, indent)))
			}
		} else {
			self.root.indent = Some(indent.into());
			self.first = false;
		}

		self.root.input.consume(indent_end);

		// A `:` in a list introduces a multi-line block. We simply leave the input stream at the newline after the `:`. The main Deserializer treats this as opening a multi-line block.
		if self.root.input.data(1)?[0] == b':' {
			self.root.input.consume(1);
			let trailing = self.root.input.read_to(b'\n')?;
			let mut trailing_len = trailing.len();
			if trailing.ends_with(b"\n") {
				trailing_len -= 1;
			}
			let s = std::str::from_utf8(trailing)?;
			if let Some(i) = s.find(|c: char| !c.is_whitespace()) {
				if s.as_bytes()[i] != b'#' {
					return Err(crate::DeserializeError::Invalid(format!("Expected newline after `:` found {:?}", s)))
				}
			}
			self.root.input.consume(trailing_len);
			return seed.deserialize(&mut *self.root).map(Some)
		}

		seed.deserialize(&mut *self.root).map(Some)
	}
}
