mod error; pub use error::*;
mod helpers; pub use helpers::*;
mod key; pub(crate) use key::*;
mod location; pub(crate) use location::*;
mod map; pub(crate) use map::*;
mod num; pub(crate) use num::*;
mod seq; pub(crate) use seq::*;

pub struct Deserializer<R: buffered_reader::BufferedReader<()>> {
	input: R,
	indent: Option<String>,
	location: Location,
}

impl<R: buffered_reader::BufferedReader<()>> Deserializer<R> {
	pub fn from(i: R) -> Self {
		Self {
			input: i,
			indent: Default::default(),
			location: Location{line: 1, column: 1},
		}
	}

	fn parse_inline_string(&mut self) -> Result<Option<(std::borrow::Cow<str>, crate::Location)>, DeserializeError> {
		let len = self.input.read_to(b'\n')?.len();
		let mut b = &self.input.consume(len)[..len];

		if let Some(comment_start) = b.iter().position(|c| c == &b'#') {
			b = &b[..comment_start];
		}

		let s = crate::from_utf8(b, self.location)?;

		let content = s.trim();

		let mut location = self.location;
		location.add_columns(content.as_ptr() as usize - s.as_ptr() as usize);
		self.location.add_lines(1);

		if content.starts_with('"') {
			return Err(crate::DeserializeErrorKind::Invalid(
				"Inline strings can't start with `\"`.\n\nQuoted strings are reserved for future features. If you want a string that starts with a quote put the string indented on its own line.".into())
				.at(location))
		}

		if !content.is_empty() {
			Ok(Some((std::borrow::Cow::from(content), location)))
		} else {
			Ok(None)
		}
	}

	fn parse_indented_string(&mut self) -> Result<(std::borrow::Cow<str>, Location), DeserializeError> {
		let mut b = self.input.read_to(b'\n')?;
		let size = b.len();
		let newline = b.last() == Some(&b'\n');
		if newline {
			b = &b[..b.len()-1];
		}
		let s = crate::from_utf8(b, self.location)?;

		let content_start = s.find(|c: char| !c.is_whitespace()).unwrap_or(s.len());
		let (indent, content) = s.split_at(content_start);

		self.location.add_columns(content_start);
		let location = self.location;

		let root_indent = self.indent.as_ref().map(|s| s.as_str()).unwrap_or("");
		let prev_root_indent = root_indent.len();

		if !(indent.len() > root_indent.len() && indent.starts_with(root_indent)) {
			return Ok((std::borrow::Cow::from(""), location))
		}

		if let Some(ref mut root_indent) = self.indent {
			root_indent.push_str(&indent[prev_root_indent..]);
		} else {
			self.indent = Some(indent.into());
		}

		let root_indent = self.indent.as_ref().unwrap().as_bytes();

		let mut r = content.to_owned(); // TODO: don't copy for single-line "multiline" block.
		self.input.consume(size);
		self.location.add_lines(1);

		loop {
			let b = self.input.read_to(b'\n')?;
			let size = b.len();

			if b.is_empty() {
				break
			}

			let content = if b.starts_with(root_indent) {
				let mut content = &b[root_indent.len()..];
				if content.ends_with(b"\n") {
					content = &content[..content.len()-1];
				}
				crate::from_utf8(content, location)?
			} else if crate::from_utf8(b, location)?.find(|c: char| !c.is_whitespace()).is_none() {
				// We don't care about the indent of "empty lines".
				""
			} else {
				break
			};

			r.push('\n');
			r.push_str(content);

			self.input.consume(size);
			self.location.add_lines(1);
		}

		while r.as_bytes()[r.len() - 1] == b'\n' {
			r.truncate(r.len() - 1);
		}

		self.indent.as_mut().unwrap().truncate(prev_root_indent);
		Ok((std::borrow::Cow::from(r), location))
	}

	fn parse_string(&mut self) -> Result<(std::borrow::Cow<str>, crate::Location), DeserializeError> {
		if let Some((s, location)) = self.parse_inline_string()? {
			// IDK why rust forces us to copy this.
			Ok((std::borrow::Cow::from(s.into_owned()), location))
		} else {
			self.parse_indented_string()
		}
	}

	fn parse_num(&mut self) -> Result<(Number, crate::Location, std::borrow::Cow<str>), DeserializeError> {
		let (s, location) = self.parse_string()?;
		let n = parse_num(&s, location)?;
		Ok((n, location, s))
	}

	fn parse_sint<T: std::convert::TryFrom<i128> + std::ops::Neg>(&mut self) -> Result<T, DeserializeError>
	where
		<T as std::convert::TryFrom<i128>>::Error: std::fmt::Debug,
		<T as std::ops::Neg>::Output: std::convert::Into<T>,
	{
		let (n, location, source) = self.parse_num()?;
		n.to_sint(&source).map_err(|e| e.at(location))
	}

	fn parse_uint<T: std::convert::TryFrom<u128>>(&mut self) -> Result<T, DeserializeError>
	where
		<T as std::convert::TryFrom<u128>>::Error: std::fmt::Debug,
	{
		let (n, location, source) = self.parse_num()?;
		n.to_uint(&source).map_err(|e| e.at(location))
	}

	fn parse_float(&mut self) -> Result<f64, DeserializeError> {
		self.parse_num().map(|(n, ..)| n.to_f64())
	}
}

impl<'d, R: buffered_reader::BufferedReader<()>> serde::de::Deserializer<'d> for &mut Deserializer<R> {
	type Error = DeserializeError;

	fn deserialize_any<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		self.deserialize_str(visitor)
	}

	fn deserialize_bool<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		let (s, location) = self.parse_string()?;
		let b = match s.as_ref() {
			"true" => true,
			"false" => false,
			other => return Err(
				DeserializeErrorKind::Invalid(
					format!("Expected bool, got {:?}", other))
					.at(location)),
		};
		visitor.visit_bool(b)
	}

	fn deserialize_i8<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_i8(self.parse_sint()?)
	}

	fn deserialize_i16<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_i16(self.parse_sint()?)
	}

	fn deserialize_i32<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_i32(self.parse_sint()?)
	}

	fn deserialize_i64<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_i64(self.parse_sint()?)
	}

	fn deserialize_i128<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_i128(self.parse_sint()?)
	}

	fn deserialize_u8<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_u8(self.parse_uint()?)
	}

	fn deserialize_u16<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_u16(self.parse_uint()?)
	}

	fn deserialize_u32<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_u32(self.parse_uint()?)
	}

	fn deserialize_u64<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_u64(self.parse_uint()?)
	}

	fn deserialize_u128<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_u128(self.parse_uint()?)
	}

	fn deserialize_f32<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_f32(self.parse_float()? as f32)
	}

	fn deserialize_f64<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		visitor.visit_f64(self.parse_float()?)
	}

	fn deserialize_char<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		let (s, location) = self.parse_string()?;
		if s.len() == 1 {
			visitor.visit_char(s.chars().next().unwrap())
		} else {
			Err(crate::DeserializeErrorKind::Invalid(format!("Expected single character, got {:?}", s)).at(location))
		}
	}

	fn deserialize_str<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		let (s, _location) = self.parse_string()?;
		match s {
			std::borrow::Cow::Borrowed(s) => visitor.visit_str(s),
			std::borrow::Cow::Owned(s) => visitor.visit_string(s),
		}
	}

	fn deserialize_string<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		self.deserialize_str(visitor)
	}

	fn deserialize_bytes<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, crate::DeserializeError> {
		self.deserialize_byte_buf(visitor)
	}

	fn deserialize_byte_buf<V: serde::de::Visitor<'d>>(self, _visitor: V) -> Result<V::Value, crate::DeserializeError> {
		Err(crate::DeserializeErrorKind::Unimplemented("deserialize_byte_buf").at(self.location))
	}

	fn deserialize_option<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		// This scans looking for a "none". It is hacky and duplicates a lot of code from regular string parsing. However it isn't clear how to reuse the original code. It would both be less efficient (as it has to buffer entire strings) and harder to understand (as it can consume irrelevant data and forget about it). On the bright side this code only handles single-line strings so it can be much simpler.

		let mut b = self.input.read_to(b'\n')?;
		let mut line_start = b.len();

		if let Some(comment_start) = b.iter().position(|c| c == &b'#') {
			b = &b[..comment_start];
		}

		let s = crate::from_utf8(b, self.location)?;
		let mut content = s.trim();

		let root_indent = self.indent.as_ref().map(|s| s.as_str()).unwrap_or("");

		if !content.is_empty() {
			return match content {
				"none" => {
					self.input.consume(line_start);
					visitor.visit_none()
				}
				_ => visitor.visit_some(self), // TODO: Don't consume.
			}
		}

		let mut offset = line_start;
		let mut content_start = 0;
		let mut location = self.location;

		loop {
			b = self.input.data(offset + 1)?;
			for (i, i_end, c) in bstr::ByteSlice::char_indices(&b[offset..]) {
				if content_start != 0 && c.is_whitespace() {
					content = unsafe { crate::from_utf8_unchecked(&b[content_start..offset+i]) };
					return match content {
						"none" => {
							self.input.consume(offset + i_end);
							visitor.visit_none()
						}
						_ => visitor.visit_some(self), // TODO: Don't consume.
					}
				} else if c == '\n' {
					line_start = offset + i_end;
					location.add_lines(1);
					continue
				} else if c == std::char::REPLACEMENT_CHARACTER {
					crate::from_utf8(&b[i..], self.location)?;
					unreachable!();
				} else if content_start == 0 && !c.is_whitespace() {
					let indent = &b[line_start..offset+i];
					if !(indent.len() > root_indent.len() && indent.starts_with(root_indent.as_bytes())) {
						return visitor.visit_some(self)
					}
					content_start = offset + i;
				}

				location.add_columns(1);
			}

			offset += b.len();
		}

	}

	fn deserialize_unit<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		self.parse_string()?;
		visitor.visit_unit()
	}

	fn deserialize_unit_struct<V: serde::de::Visitor<'d>>(
		self,
		_name: &'static str,
		visitor: V,
	) -> Result<V::Value, DeserializeError> {
		self.deserialize_unit(visitor)
	}

	fn deserialize_newtype_struct<V: serde::de::Visitor<'d>>(
		self,
		_name: &'static str,
		visitor: V,
	) -> Result<V::Value, DeserializeError> {
		visitor.visit_newtype_struct(self)
	}

	fn deserialize_seq<V: serde::de::Visitor<'d>>(mut self, visitor: V) -> Result<V::Value, DeserializeError> {
		let indent_len = self.indent.as_ref().map(|s| s.len()).unwrap_or(0);
		let r = visitor.visit_seq(SeqDeserializer {
			root: &mut self,
			first: true,
		});
		self.indent.as_mut().unwrap().truncate(indent_len);
		r
	}

	fn deserialize_tuple<V: serde::de::Visitor<'d>>(self, _len: usize, visitor: V) -> Result<V::Value, DeserializeError> {
		self.deserialize_seq(visitor)
	}

	fn deserialize_tuple_struct<V: serde::de::Visitor<'d>>(
		self,
		_name: &'static str,
		_len: usize,
		visitor: V,
	) -> Result<V::Value, DeserializeError> {
		self.deserialize_seq(visitor)
	}

	fn deserialize_map<V: serde::de::Visitor<'d>>(mut self, visitor: V) -> Result<V::Value, DeserializeError> {
		let indent_len = self.indent.as_ref().map(|s| s.len()).unwrap_or(0);
		let r = visitor.visit_map(MapDeserializer {
			root: &mut self,
			first: true,
		});
		self.indent.as_mut().unwrap().truncate(indent_len);
		r
	}

	fn deserialize_struct<V: serde::de::Visitor<'d>>(
		self,
		_name: &'static str,
		_fields: &'static [&'static str],
		visitor: V,
	) -> Result<V::Value, DeserializeError> {
		self.deserialize_map(visitor)
	}

	fn deserialize_enum<V: serde::de::Visitor<'d>>(
		self,
		_name: &'static str,
		_variants: &'static [&'static str],
		_visitor: V,
	) -> Result<V::Value, DeserializeError> {
		Err(crate::DeserializeErrorKind::Unimplemented("deserialize_enum").at(self.location))
	}

	fn deserialize_identifier<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		self.deserialize_str(visitor)
	}

	fn deserialize_ignored_any<V: serde::de::Visitor<'d>>(self, visitor: V) -> Result<V::Value, DeserializeError> {
		self.deserialize_unit(visitor)
	}
}
