#[derive(Debug,thiserror::Error)]
#[non_exhaustive]
pub struct DeserializeError {
	#[source]
	pub kind: DeserializeErrorKind,
	pub location: Option<crate::Location>,
}

impl std::fmt::Display for DeserializeError {
	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
		if let Some(location) = self.location {
			write!(f, "{} at {}", self.kind, location)
		} else {
			write!(f, "{}", self.kind)
		}
	}
}

impl serde::de::Error for DeserializeError {
	fn custom<T: std::fmt::Display>(msg: T) -> Self {
		DeserializeError {
			kind: DeserializeErrorKind::Custom(format!("{}", msg)),
			location: None,
		}
	}
}

impl From<std::io::Error> for DeserializeError {
	fn from(e: std::io::Error) -> Self {
		DeserializeError {
			kind: DeserializeErrorKind::Io(e),
			location: None,
		}
	}
}

#[non_exhaustive]
#[derive(Debug,thiserror::Error)]
pub enum DeserializeErrorKind {
	#[error("{0}")]
	Custom(String),
	#[error("{0}")]
	Invalid(String),
	#[error(transparent)]
	Io(#[from] std::io::Error),
	#[error(transparent)]
	ParseInt(#[from] std::num::ParseIntError),
	#[error("{0} is not supported.")]
	Unimplemented(&'static str),
	#[error("Invalid UTF-8")]
	InvalidUtf8,
}

impl DeserializeErrorKind {
	pub(crate) fn at(self, location: crate::Location) -> DeserializeError {
		DeserializeError {
			kind: self,
			location: Some(location),
		}
	}
}
