#![cfg_attr(not(any(test, feature = "std")), no_std)]

#![deny(clippy::all, clippy::cargo)]
#![allow(clippy::missing_safety_doc)] // TODO

#[allow(unused_macros)]
macro_rules! item_group {
    ($($item:item)*) => {
        $($item)*
    }
}

#[cfg(any(target_arch = "x86", target_arch = "x86_64"))]
item_group! {
    mod avx2;
    pub use self::avx2::AVX2;

    mod sse41;
    pub use self::sse41::SSE41;
}

pub unsafe trait InstructionSet: Copy {
    fn detect() -> Option<Self>;
    unsafe fn new_unchecked() -> Self;
}

pub unsafe trait SIMD128: InstructionSet {
    type V128: Copy;

    unsafe fn v128_load(self, addr: *const u8) -> Self::V128;
    unsafe fn v128_loadu(self, addr: *const u8) -> Self::V128;
    unsafe fn v128_storeu(self, addr: *mut u8, a: Self::V128);

    fn v128_or(self, a: Self::V128, b: Self::V128) -> Self::V128;
    fn v128_and(self, a: Self::V128, b: Self::V128) -> Self::V128;
    fn v128_to_bytes(self, a: Self::V128) -> [u8; 16];

    fn u8x16_splat(self, x: u8) -> Self::V128;

    fn i8x16_shuffle(self, a: Self::V128, b: Self::V128) -> Self::V128;

    fn i16x8_sll<const IMM8: i32>(self, a: Self::V128) -> Self::V128;
    fn i16x8_srl<const IMM8: i32>(self, a: Self::V128) -> Self::V128;
    fn i16x8_extract<const IMM3: i32>(self, a: Self::V128) -> i16;

    fn i32x4_extract<const IMM2: i32>(self, a: Self::V128) -> i32;
}

pub unsafe trait SIMD256: SIMD128 {
    type V256: Copy;

    unsafe fn v256_load(self, addr: *const u8) -> Self::V256;
    unsafe fn v256_loadu(self, addr: *const u8) -> Self::V256;
    unsafe fn v256_storeu(self, addr: *mut u8, a: Self::V256);

    fn v256_or(self, a: Self::V256, b: Self::V256) -> Self::V256;
    fn v256_and(self, a: Self::V256, b: Self::V256) -> Self::V256;
    fn v256_to_bytes(self, a: Self::V256) -> [u8; 32];
    fn v256_zero(self) -> Self::V256;

    fn v128_from_low_v256(self, a: Self::V256) -> Self::V128;
    fn v128_from_high_v256(self, a: Self::V256) -> Self::V128;

    fn u8x32_splat(self, x: u8) -> Self::V256;

    fn u64x2_from_low_u128x2(self, a: Self::V256) -> Self::V128;

    fn i8x32_shuffle(self, a: Self::V256, b: Self::V256) -> Self::V256;
    fn i8x32_all_non_zero(self, a: Self::V256) -> bool;
    fn i8x32_add(self, a: Self::V256, b: Self::V256) -> Self::V256;

    fn i16x16_from_u8x16(self, a: Self::V128) -> Self::V256;
    fn i16x16_sll<const IMM8: i32>(self, a: Self::V256) -> Self::V256;
    fn i16x16_srl<const IMM8: i32>(self, a: Self::V256) -> Self::V256;
}
