// This file is part of simavr-section, a Rust port of the *simavr*
// header `avr_mcu_section.h`.
//
// Copyright 2021 Andrew Dona-Couch
//
// simavr-section is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// simavr-section is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

extern crate proc_macro;

use proc_macro::TokenStream;
use quote::quote;
use syn::parse::{self, Parse, ParseStream};
use syn::{parse_macro_input, Expr, Ident, LitByteStr, Token};

struct StringInput {
    tag: Ident,
    string: LitByteStr,
}

impl Parse for StringInput {
    fn parse(input: ParseStream) -> parse::Result<StringInput> {
        let tag = input.parse()?;
        input.parse::<Token![,]>()?;
        let string = input.parse()?;
        Ok(StringInput { tag, string })
    }
}

/// Add an AVR metadata string value to the binary.
///
/// The first argument should be the tag name as defined by `simavr`, and
/// the second argument is the byte string value.
///
/// # Example
///
/// ```
/// # use simavr_section::avr_mcu_string;
/// avr_mcu_string!(AVR_MMCU_TAG_VCD_FILENAME, b"tracefile.vcd");
/// ```
#[proc_macro]
pub fn avr_mcu_string(input: TokenStream) -> TokenStream {
    let StringInput { tag, string } = parse_macro_input!(input as StringInput);

    let value = string
        .value()
        .into_iter()
        .chain(std::iter::repeat(0))
        .take(64)
        .collect::<Vec<_>>();

    let name = Ident::new(&format!("_{}", tag), tag.span());

    let array = quote! {
        [
            #(
                #value
            ),*
        ]
    };

    quote!(
        #[used]
        #[no_mangle]
        #[link_section = ".mmcu"]
        pub static #name: ::simavr_section::bindings::avr_mmcu_string_t = ::simavr_section::bindings::avr_mmcu_string_t {
            tag: ::simavr_section::bindings::#tag as _,
            len: ::core::mem::size_of::<::simavr_section::bindings::avr_mmcu_string_t>() as u8 - 2,
            string: #array,
        };
    ).into()
}

struct LongInput {
    tag: Ident,
    val: Expr,
}

impl Parse for LongInput {
    fn parse(input: ParseStream) -> parse::Result<LongInput> {
        let tag = input.parse()?;
        input.parse::<Token![,]>()?;
        let val = input.parse()?;
        Ok(LongInput { tag, val })
    }
}

/// Add an AVR metadata long (`u32`) value to the binary.
///
/// The first argument should be the tag name as defined by `simavr`, and
/// the second argument is the `u32` value.
///
/// # Example
///
/// ```
/// # use simavr_section::avr_mcu_long;
/// avr_mcu_long!(AVR_MMCU_TAG_VCC, 5000);
/// ```
#[proc_macro]
pub fn avr_mcu_long(input: TokenStream) -> TokenStream {
    let LongInput { tag, val } = parse_macro_input!(input as LongInput);

    let name = Ident::new(&format!("_{}{}", tag, tag.span().start().line), tag.span());

    quote!(
        #[used]
        #[no_mangle]
        #[link_section = ".mmcu"]
        pub static #name: ::simavr_section::bindings::avr_mmcu_long_t = ::simavr_section::bindings::avr_mmcu_long_t {
            tag: ::simavr_section::bindings::#tag as _,
            len: ::core::mem::size_of::<::simavr_section::bindings::avr_mmcu_long_t>() as u8 - 2,
            val: #val,
        };
    ).into()
}

struct TraceInput {
    tag: Ident,
    mask: Expr,
    what: Expr,
    name: LitByteStr,
}

impl Parse for TraceInput {
    fn parse(input: ParseStream) -> parse::Result<TraceInput> {
        let tag = input.parse()?;
        input.parse::<Token![,]>()?;
        let mask = input.parse()?;
        input.parse::<Token![,]>()?;
        let what = input.parse()?;
        input.parse::<Token![,]>()?;
        let name = input.parse()?;
        Ok(TraceInput {
            tag,
            mask,
            what,
            name,
        })
    }
}

/// Add an AVR metadata VCD trace value to the binary.
///
/// The first argument should be the tag name as defined by `simavr`, and
/// the subsequent arguments are specific to the tag.
///
/// # Example
///
/// ```
/// # use simavr_section::avr_mcu_vcd_trace;
/// avr_mcu_vcd_trace!(AVR_MMCU_TAG_VCD_PORTPIN, b'B', 5, b"PINB5");
/// ```
#[proc_macro]
pub fn avr_mcu_vcd_trace(input: TokenStream) -> TokenStream {
    let TraceInput {
        tag,
        mask,
        what,
        name,
    } = parse_macro_input!(input as TraceInput);

    let id = Ident::new(&format!("_{}{}", tag, tag.span().start().line), tag.span());

    let value = name
        .value()
        .into_iter()
        .chain(std::iter::repeat(0))
        .take(32)
        .collect::<Vec<_>>();
    let array = quote! {
        [
            #(
                #value
            ),*
        ]
    };

    quote!(
        #[used]
        #[no_mangle]
        #[link_section = ".mmcu"]
        pub static #id: ::simavr_section::bindings::UnsafeVcdTrace = ::simavr_section::bindings::UnsafeVcdTrace(::simavr_section::bindings::avr_mmcu_vcd_trace_t {
            tag: ::simavr_section::bindings::#tag as _,
            len: ::core::mem::size_of::<::simavr_section::bindings::avr_mmcu_vcd_trace_t>() as u8 - 2,
            mask: #mask,
            what: #what as *mut ::simavr_section::bindings::libc::c_void,
            name: #array,
        });
    ).into()
}
