use num_traits::Float;

/// The analysis of simulation outputs is often an analysis of means.  In
/// these cases, the central limit theorem can be used for (among other
/// things), the construction of confidence intervals.  A T score (Student T
/// distribution) is used when the degrees of freedom for the data is less
/// than 100, and a Z core (Normal distribution) is used when the degrees of
/// freedom is greater than 100.
pub fn t_score<T: Float>(alpha: T, df: usize) -> T {
    let alphas: [T; 7] = [
        T::from(0.1).unwrap(),
        T::from(0.05).unwrap(),
        T::from(0.025).unwrap(),
        T::from(0.01).unwrap(),
        T::from(0.005).unwrap(),
        T::from(0.001).unwrap(),
        T::from(0.0005).unwrap(),
    ];
    let alpha_index = alphas
        .iter()
        .position(|alpha_option| *alpha_option == alpha)
        .unwrap();
    if df > 100 {
        // Z Scores
        z_lookup(alpha_index)
    } else {
        // T Scores
        t_lookup(alpha_index, df)
    }
}

fn z_lookup<T: Float>(alpha_index: usize) -> T {
    T::from([1.2816, 1.6449, 1.9600, 2.3263, 2.5758, 3.0902, 3.2905][alpha_index]).unwrap()
}

fn t_lookup<T: Float>(alpha_index: usize, df: usize) -> T {
    // Clippy Allow: 2.718 is a coincidence - unrelated to f{32, 64}::consts::E
    #[allow(clippy::approx_constant)]
    T::from(
        [
            [3.078, 6.314, 12.706, 31.821, 63.656, 318.289, 636.578],
            [1.886, 2.920, 4.303, 6.965, 9.925, 22.328, 31.600],
            [1.638, 2.353, 3.182, 4.541, 5.841, 10.214, 12.924],
            [1.533, 2.132, 2.776, 3.747, 4.604, 7.173, 8.610],
            [1.476, 2.015, 2.571, 3.365, 4.032, 5.894, 6.869],
            [1.440, 1.943, 2.447, 3.143, 3.707, 5.208, 5.959],
            [1.415, 1.895, 2.365, 2.998, 3.499, 4.785, 5.408],
            [1.397, 1.860, 2.306, 2.896, 3.355, 4.501, 5.041],
            [1.383, 1.833, 2.262, 2.821, 3.250, 4.297, 4.781],
            [1.372, 1.812, 2.228, 2.764, 3.169, 4.144, 4.587],
            [1.363, 1.796, 2.201, 2.718, 3.106, 4.025, 4.437],
            [1.356, 1.782, 2.179, 2.681, 3.055, 3.930, 4.318],
            [1.350, 1.771, 2.160, 2.650, 3.012, 3.852, 4.221],
            [1.345, 1.761, 2.145, 2.624, 2.977, 3.787, 4.140],
            [1.341, 1.753, 2.131, 2.602, 2.947, 3.733, 4.073],
            [1.337, 1.746, 2.120, 2.583, 2.921, 3.686, 4.015],
            [1.333, 1.740, 2.110, 2.567, 2.898, 3.646, 3.965],
            [1.330, 1.734, 2.101, 2.552, 2.878, 3.610, 3.922],
            [1.328, 1.729, 2.093, 2.539, 2.861, 3.579, 3.883],
            [1.325, 1.725, 2.086, 2.528, 2.845, 3.552, 3.850],
            [1.323, 1.721, 2.080, 2.518, 2.831, 3.527, 3.819],
            [1.321, 1.717, 2.074, 2.508, 2.819, 3.505, 3.792],
            [1.319, 1.714, 2.069, 2.500, 2.807, 3.485, 3.768],
            [1.318, 1.711, 2.064, 2.492, 2.797, 3.467, 3.745],
            [1.316, 1.708, 2.060, 2.485, 2.787, 3.450, 3.725],
            [1.315, 1.706, 2.056, 2.479, 2.779, 3.435, 3.707],
            [1.314, 1.703, 2.052, 2.473, 2.771, 3.421, 3.689],
            [1.313, 1.701, 2.048, 2.467, 2.763, 3.408, 3.674],
            [1.311, 1.699, 2.045, 2.462, 2.756, 3.396, 3.660],
            [1.310, 1.697, 2.042, 2.457, 2.750, 3.385, 3.646],
            [1.309, 1.696, 2.040, 2.453, 2.744, 3.375, 3.633],
            [1.309, 1.694, 2.037, 2.449, 2.738, 3.365, 3.622],
            [1.308, 1.692, 2.035, 2.445, 2.733, 3.356, 3.611],
            [1.307, 1.691, 2.032, 2.441, 2.728, 3.348, 3.601],
            [1.306, 1.690, 2.030, 2.438, 2.724, 3.340, 3.591],
            [1.306, 1.688, 2.028, 2.434, 2.719, 3.333, 3.582],
            [1.305, 1.687, 2.026, 2.431, 2.715, 3.326, 3.574],
            [1.304, 1.686, 2.024, 2.429, 2.712, 3.319, 3.566],
            [1.304, 1.685, 2.023, 2.426, 2.708, 3.313, 3.558],
            [1.303, 1.684, 2.021, 2.423, 2.704, 3.307, 3.551],
            [1.303, 1.683, 2.020, 2.421, 2.701, 3.301, 3.544],
            [1.302, 1.682, 2.018, 2.418, 2.698, 3.296, 3.538],
            [1.302, 1.681, 2.017, 2.416, 2.695, 3.291, 3.532],
            [1.301, 1.680, 2.015, 2.414, 2.692, 3.286, 3.526],
            [1.301, 1.679, 2.014, 2.412, 2.690, 3.281, 3.520],
            [1.300, 1.679, 2.013, 2.410, 2.687, 3.277, 3.515],
            [1.300, 1.678, 2.012, 2.408, 2.685, 3.273, 3.510],
            [1.299, 1.677, 2.011, 2.407, 2.682, 3.269, 3.505],
            [1.299, 1.677, 2.010, 2.405, 2.680, 3.265, 3.500],
            [1.299, 1.676, 2.009, 2.403, 2.678, 3.261, 3.496],
            [1.298, 1.675, 2.008, 2.402, 2.676, 3.258, 3.492],
            [1.298, 1.675, 2.007, 2.400, 2.674, 3.255, 3.488],
            [1.298, 1.674, 2.006, 2.399, 2.672, 3.251, 3.484],
            [1.297, 1.674, 2.005, 2.397, 2.670, 3.248, 3.480],
            [1.297, 1.673, 2.004, 2.396, 2.668, 3.245, 3.476],
            [1.297, 1.673, 2.003, 2.395, 2.667, 3.242, 3.473],
            [1.297, 1.672, 2.002, 2.394, 2.665, 3.239, 3.470],
            [1.296, 1.672, 2.002, 2.392, 2.663, 3.237, 3.466],
            [1.296, 1.671, 2.001, 2.391, 2.662, 3.234, 3.463],
            [1.296, 1.671, 2.000, 2.390, 2.660, 3.232, 3.460],
            [1.296, 1.670, 2.000, 2.389, 2.659, 3.229, 3.457],
            [1.295, 1.670, 1.999, 2.388, 2.657, 3.227, 3.454],
            [1.295, 1.669, 1.998, 2.387, 2.656, 3.225, 3.452],
            [1.295, 1.669, 1.998, 2.386, 2.655, 3.223, 3.449],
            [1.295, 1.669, 1.997, 2.385, 2.654, 3.220, 3.447],
            [1.295, 1.668, 1.997, 2.384, 2.652, 3.218, 3.444],
            [1.294, 1.668, 1.996, 2.383, 2.651, 3.216, 3.442],
            [1.294, 1.668, 1.995, 2.382, 2.650, 3.214, 3.439],
            [1.294, 1.667, 1.995, 2.382, 2.649, 3.213, 3.437],
            [1.294, 1.667, 1.994, 2.381, 2.648, 3.211, 3.435],
            [1.294, 1.667, 1.994, 2.380, 2.647, 3.209, 3.433],
            [1.293, 1.666, 1.993, 2.379, 2.646, 3.207, 3.431],
            [1.293, 1.666, 1.993, 2.379, 2.645, 3.206, 3.429],
            [1.293, 1.666, 1.993, 2.378, 2.644, 3.204, 3.427],
            [1.293, 1.665, 1.992, 2.377, 2.643, 3.202, 3.425],
            [1.293, 1.665, 1.992, 2.376, 2.642, 3.201, 3.423],
            [1.293, 1.665, 1.991, 2.376, 2.641, 3.199, 3.421],
            [1.292, 1.665, 1.991, 2.375, 2.640, 3.198, 3.420],
            [1.292, 1.664, 1.990, 2.374, 2.640, 3.197, 3.418],
            [1.292, 1.664, 1.990, 2.374, 2.639, 3.195, 3.416],
            [1.292, 1.664, 1.990, 2.373, 2.638, 3.194, 3.415],
            [1.292, 1.664, 1.989, 2.373, 2.637, 3.193, 3.413],
            [1.292, 1.663, 1.989, 2.372, 2.636, 3.191, 3.412],
            [1.292, 1.663, 1.989, 2.372, 2.636, 3.190, 3.410],
            [1.292, 1.663, 1.988, 2.371, 2.635, 3.189, 3.409],
            [1.291, 1.663, 1.988, 2.370, 2.634, 3.188, 3.407],
            [1.291, 1.663, 1.988, 2.370, 2.634, 3.187, 3.406],
            [1.291, 1.662, 1.987, 2.369, 2.633, 3.185, 3.405],
            [1.291, 1.662, 1.987, 2.369, 2.632, 3.184, 3.403],
            [1.291, 1.662, 1.987, 2.368, 2.632, 3.183, 3.402],
            [1.291, 1.662, 1.986, 2.368, 2.631, 3.182, 3.401],
            [1.291, 1.662, 1.986, 2.368, 2.630, 3.181, 3.399],
            [1.291, 1.661, 1.986, 2.367, 2.630, 3.180, 3.398],
            [1.291, 1.661, 1.986, 2.367, 2.629, 3.179, 3.397],
            [1.291, 1.661, 1.985, 2.366, 2.629, 3.178, 3.396],
            [1.290, 1.661, 1.985, 2.366, 2.628, 3.177, 3.395],
            [1.290, 1.661, 1.985, 2.365, 2.627, 3.176, 3.394],
            [1.290, 1.661, 1.984, 2.365, 2.627, 3.175, 3.393],
            [1.290, 1.660, 1.984, 2.365, 2.626, 3.175, 3.392],
            [1.290, 1.660, 1.984, 2.364, 2.626, 3.174, 3.390],
        ][df - 1][alpha_index],
    )
    .unwrap()
}
