use std::fmt::Debug;
use async_std::future::Future;
use async_std::net::{SocketAddr, ToSocketAddrs, TcpStream, TcpListener};
use futures::{AsyncReadExt, AsyncWriteExt, SinkExt, StreamExt, FutureExt, io::{WriteHalf, ReadHalf}};
use futures::channel::mpsc;
use crate::{Client, ConnectionEvent, MailError, Result};

// TODO: Change some errors to more specific ones.

// TODO: Create a struct for the tuple to make it cleaner.
pub async fn tcp_create(addr: impl ToSocketAddrs) -> Result<(SocketAddr, ReadHalf<TcpStream>, WriteHalf<TcpStream>)> {
    let tcp_stream = TcpStream::connect(addr).await.map_err(MailError::UnknownIo)?;
    let addr = tcp_stream.peer_addr().map_err(MailError::UnknownIo)?;
    let (read, write) = tcp_stream.split();
    Ok((addr, read, write))
}

pub async fn tcp_listen(addr: impl ToSocketAddrs) -> Result<TcpListener> {
    TcpListener::bind(addr).await.map_err(MailError::UnknownIo)
}

pub async fn write_task(mut messages: mpsc::UnboundedReceiver<Vec<u8>>, mut stream: WriteHalf<TcpStream>) -> Result<()> {
    while let Some(data) = messages.next().await {
        stream.write(&data).await.map_err(MailError::UnknownIo)?;
        println!("wrote data{:?} to stream.", data);
    }
    Ok(())
}

pub async fn broker_task<C: Client, F: Send + 'static + Future<Output=()>>(mut events: mpsc::UnboundedReceiver<ConnectionEvent<C>>, on_event: fn(ConnectionEvent<C>) -> F) -> Result<()> {
    while let Some(event) = events.next().await {
        // We can't process multiple events at once.
        on_event(event).await;
    }
    Ok(())
}

pub async fn blocking_dispatcher<C: Client, S: Debug>(
    mut broker: mpsc::UnboundedSender<ConnectionEvent<C>>,
    buffer: &mut [u8],
    client: C,
    mut stream_read: ReadHalf<TcpStream>,
    mut shutdown_recv: mpsc::UnboundedReceiver<S>
) -> Result<()> {
    loop {
        // Poll two futures at once to read both at the same time. (Kind of like using threads but better)
        futures::select! {
            shutdown_res = shutdown_recv.next().fuse() => {
                match shutdown_res {
                    Some(_shutdown) => {
                        // TODO: Maybe do stream.shutdown or something to make it peaceful for other end?
                        broker.send(ConnectionEvent::Terminate { client: C::clone(&client), peaceful: true }).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })?;
                        break;
                    },
                    None => {
                        broker.send(ConnectionEvent::Error { error: MailError::ShutdownBridgeClosed }).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })?;
                    }
                }
            },
            read_result = stream_read.read(buffer).fuse() => {
                match read_result {
                    Ok(n) => {
                        if n == 0 {
                            broker.send(ConnectionEvent::Terminate { client: C::clone(&client), peaceful: true }).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })?;
                            break;
                        }

                        let data = &buffer[..n];
                        broker.send(ConnectionEvent::Packet { bytes: data.to_vec(), client: C::clone(&client) }).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })?;
                    },
                    Err(error) => {
                        broker.send(ConnectionEvent::Terminate { client: C::clone(&client), peaceful: false }).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })?;
                        return Err(MailError::UnknownIo(error));
                    }
                }
            },
        }
    }
    Ok(())
}