use std::pin::Pin;
use std::sync::Arc;
use async_std::{
    net::{
        SocketAddr,
        ToSocketAddrs
    },
    task,
    task::JoinHandle,
};
use async_std::net::{TcpListener, TcpStream};
use futures::{
    task::{
        Context,
        Poll
    },
    io::{
        WriteHalf,
        ReadHalf
    },
    lock::Mutex,
    Future,
    channel::mpsc,
    SinkExt,
    FutureExt,
    AsyncReadExt,
    AsyncWriteExt,
    StreamExt
};
use async_trait::async_trait;
use crate::{shared_internals, BuilderError, Client, ConnectionEvent, MailError, Result};

#[derive(Debug)]
struct RemoteClientInner {
    addr: SocketAddr,
    msg: Mutex<mpsc::UnboundedSender<Vec<u8>>>,
    shutdown: Mutex<mpsc::UnboundedSender<ServerShutdown>>,
}

#[derive(Debug, Clone)]
pub struct RemoteClient(Arc<RemoteClientInner>);

#[async_trait]
impl Client for RemoteClient {
    async fn send_pkg(&mut self, pkg: Vec<u8>) -> Result<()> {
        self.0.msg.lock().await.send(pkg).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })
    }

    async fn terminate(mut self) -> Result<()> {
        self.0.shutdown.lock().await.send(ServerShutdown).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })
    }

    fn address(&self) -> SocketAddr {
        self.0.addr
    }
}

#[derive(Debug, Copy, Clone)]
pub struct ServerShutdown;

fn req<T>(opt: Option<T>, name: &str) -> Result<T> {
    opt.ok_or_else(|| MailError::InvalidStructBuild(BuilderError::RequiredField(String::from(name))))
}

/// Struct used for building Server instances. This is a public API and the Server struct cannot be created any other way.
///
/// Using this you can construct a Server object using [ServerBuilder::addr()], [ServerBuilder::event_handler()] and finally [ServerBuilder::start()].
/// [ServerHandle] is used to terminate all connections or (WIP) iterate through all current connections.
///
/// # Examples
/// ```
/// use sign_mail::prelude::*;
///
/// async fn on_event(event: ConnectionEvent<RemoteClient>) {
///     println!("got event: {:?}", event);
/// }
///
/// let server = ServerBuilder::new()
///     .addr("127.0.0.1:8080").unwrap()
///     .event_handler(on_event).unwrap()
///     .start().await.unwrap();
/// // Wait until server.terminate() is called. (Which in this case, is never.)
/// server.await;
///
/// ```
pub struct ServerBuilder<F: Future<Output = ()> + Send + 'static, SA: ToSocketAddrs> {
    addr: Option<SA>,
    event_handler: Option<fn(ConnectionEvent<RemoteClient>) -> F>
}

impl<F, SA> Default for ServerBuilder<F, SA>
    where
        F: Future<Output = ()> + Send + 'static,
        SA: ToSocketAddrs
{
    fn default() -> Self {
        ServerBuilder::new()
    }
}

impl<F, SA> ServerBuilder<F, SA>
    where
        F: Future<Output = ()> + Send + 'static,
        SA: ToSocketAddrs
{
    pub fn new() -> ServerBuilder<F, SA> {
        ServerBuilder {
            addr: None,
            event_handler: None
        }
    }

    /// Sets the address used by this [ServerBuilder]. This will address will be used when the [ServerBuilder::start()] method is called.
    ///
    ///# Examples
    ///
    /// See usage in [ServerBuilder] docs.
    // TODO: Try to addr.to_socket_addrs() to see if it fails.
    pub fn addr(mut self, addr: SA) -> Result<Self> {
        if self.addr.is_none() {
            self.addr = Some(addr);
            Ok(self)
        } else {
            Err(MailError::InvalidStructBuild(BuilderError::FieldAlreadySet(String::from("Address"))))
        }
    }

    /// Set an event handler to be used by the server. This is required since the generic parameter cannot be resolved otherwise.
    ///
    /// # Examples
    ///
    /// See usage in [ServerBuilder] docs.
    pub fn event_handler(mut self, handler: fn(ConnectionEvent<RemoteClient>) -> F) -> Result<Self> {
        if self.event_handler.is_none() {
            self.event_handler = Some(handler);
            Ok(self)
        } else {
            Err(MailError::InvalidStructBuild(BuilderError::FieldAlreadySet(String::from("Event handler"))))
        }
    }

    /// Start a server with the current configuration.
    ///
    /// This function is async due to multiple servers being able to run at once (Using different socket addresses, this is a requirement by the os).
    ///
    /// ## Note: This function *will* block until a TCP Listener can be created an the target address.
    ///
    /// # Examples
    ///
    /// See usage in [ServerBuilder] docs.
    pub async fn start(self) -> Result<ServerHandle> {
        let (shutdown_all_channel, shutdown_recv) = mpsc::unbounded();
        let bootstrap = task::spawn(internal::bind_server(shared_internals::tcp_listen(req(self.addr, "Address")?).await?, req(self.event_handler, "Event handler")?, shutdown_recv));
        Ok(ServerHandle {
            bootstrap,
            shutdown_all_channel,
        })
    }
}

/// Struct for handling a running server.
///
/// # Note:
/// * When calling `server.await`; fatal errors may be propagated through a [Result] which is defined in sign_mail/lib.rs.
///
/// # Examples
/// Opening a server and not closing it
/// ```
/// use sign_mail::prelude::*;
///
/// async fn do_stuff(event: ConnectionEvent<LocalClient>) {
///     println!("doing stuff: {:?}", event);
/// }
///
/// let server = ServerBuilder::new()
///     .event_handler(do_stuff).unwrap()
///     .addr("127.0.0.1:8080").unwrap()
///     .start().await.unwrap();
///
/// // Wait for server to finish (Which is never)
/// server.await.unwrap();
/// ```
/// Stopping a server when someone says goodbye
/// ```
/// use sign_mail::prelude::*;
///
/// async fn polite_stuff(event: ConnectionEvent<LocalClient>) {
///     println!("doing polite stuff: {:?}", event);
///     if let ConnectionEvent::Packet { mut client, bytes } = event {
///         if bytes == b"Goodbye!" {
///             client.terminate().await.expect("Could not terminate connection for some reason!");
///         }
///     }
/// }
///
/// let server = ServerBuilder::new()
///     .event_handler(polite_stuff).unwrap()
///     .addr("127.0.0.1:8080").unwrap()
///     .start().await.expect("Couldn't start server!");
///
/// // Wait for server to stop
/// server.await;
/// ```
pub struct ServerHandle {
    bootstrap: JoinHandle<Result<()>>,
    shutdown_all_channel: mpsc::UnboundedSender<ServerShutdown>,
}

impl ServerHandle {
    /// Terminate all connections to the server.
    ///
    /// # Errors
    /// This function will attempt to send values through a [mpsc::UnboundedSender] which can fail in some circumstances.
    /// The error returned will be a [MailError::MpscSendError] or [MailError::NoConnection].
    ///
    /// # Usage
    /// You wouldn't normally use this function very much. [RemoteClient] has a [RemoteClient::terminate()] method that is used more often.
    pub async fn terminate(mut self) -> Result<()> {
        self.shutdown_all_channel.send(ServerShutdown).await.map_err(|err| if err.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(err) })
    }
}

impl Future for ServerHandle {
    type Output = Result<()>;

    fn poll(mut self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Self::Output> {
        self.bootstrap.poll_unpin(cx)
    }
}

mod internal {
    use crate::{Client, ConnectionEvent};

    use super::*;
    pub fn spawn_task<C, R, F>(future: F, mut broker_send: mpsc::UnboundedSender<ConnectionEvent<C>>) -> task::JoinHandle<()>
        where
            F: Future<Output=Result<R>> + Send + 'static,
            R: Send,
            C: Client + 'static
    {
        task::spawn(async move {
            if let Err(error) = future.await {
                broker_send.send(ConnectionEvent::Error { error }).await.unwrap();
                //eprintln!("Error occurred in handle: {:#?}", e);
            }
        })
    }

    pub async fn bind_server<F: Send + 'static + Future<Output=()>>(
        listener: TcpListener,
        on_event: fn(ConnectionEvent<RemoteClient>) -> F,
        mut shutdown_all: mpsc::UnboundedReceiver<ServerShutdown>
    ) -> Result<()> {
        //let listener: TcpListener = TcpListener::bind(addr).await.map_err(MailError::UnknownIo)?;

        let mut incoming = listener.incoming();

        let mut connection_handles: Vec<(mpsc::UnboundedSender<ServerShutdown>, _)> = vec![];

        let (broker_send, broker_receive) = mpsc::unbounded();
        let broker_handle = spawn_task(broker_task(broker_receive, on_event), broker_send.clone());

        loop {
            let r: Result<()> = futures::select! {
            stream = incoming.next().fuse() => {
                if let Some(stream) = stream {
                    let stream = stream.map_err(MailError::UnknownIo)?;
                    let (shutdown_send, shutdown_recv) = mpsc::unbounded();
                    let addr = stream.peer_addr().map_err(MailError::UnknownIo)?;
                    let (tcp_read, tcp_write) = stream.split();
                    let connection_handle = spawn_task(handle_connection(broker_send.clone(), (addr, tcp_read, tcp_write), shutdown_send.clone(), shutdown_recv), broker_send.clone());
                    connection_handles.push((shutdown_send, connection_handle));
                    Ok(())
                } else {
                    break;
                }
            },
            shutdown = shutdown_all.next().fuse() => {
                match shutdown {
                    None => eprintln!("main: shutdown was closed."),
                    Some(shutdown) => {
                        for (shutdown_channel, _) in &mut connection_handles {
                            shutdown_channel.send(shutdown).await.map_err(MailError::MpscSendError)?;
                        }
                        break;
                    }
                }
                Ok(())
            }
        };
            r?;
        }

        for (_, connection) in connection_handles.into_iter() {
            connection.await;
        }
        drop(broker_send);
        broker_handle.await;
        Ok(())
    }

    async fn handle_connection(
        mut broker: mpsc::UnboundedSender<ConnectionEvent<RemoteClient>>,
        (addr, tcp_read, tcp_write): (SocketAddr, ReadHalf<TcpStream>, WriteHalf<TcpStream>),
        shutdown_send: mpsc::UnboundedSender<ServerShutdown>,
        shutdown_recv: mpsc::UnboundedReceiver<ServerShutdown>
    ) -> Result<()> {
        let (msg, msg_recv) = mpsc::unbounded();
        let client = RemoteClient(Arc::new(RemoteClientInner {
            addr,
            msg: Mutex::new(msg),
            shutdown: Mutex::new(shutdown_send),
        }));
        let _write_handle = spawn_task(write_task(msg_recv, tcp_write), broker.clone());

        let mut buffer = [0u8; 4096];
        broker.send(ConnectionEvent::Established { client: RemoteClient::clone(&client) }).await.map_err(MailError::MpscSendError)?;
        shared_internals::blocking_dispatcher(
            broker,
            &mut buffer,
            client,
            tcp_read,
            shutdown_recv
        ).await
    }

    async fn write_task(mut messages: mpsc::UnboundedReceiver<Vec<u8>>, mut stream: WriteHalf<TcpStream>) -> Result<()> {
        while let Some(data) = messages.next().await {
            stream.write(&data).await.map_err(MailError::UnknownIo)?;
            println!("wrote data{:?} to stream.", data);
        }
        Ok(())
    }

    async fn broker_task<C: Client, F: Send + 'static + Future<Output=()>>(mut events: mpsc::UnboundedReceiver<ConnectionEvent<C>>, on_event: fn(ConnectionEvent<C>) -> F) -> Result<()> {
        while let Some(event) = events.next().await {
            on_event(event).await;
        }
        Ok(())
    }
}