use async_trait::async_trait;
pub mod server;
pub mod client;
pub mod prelude;
pub(crate) mod shared_internals;

use std::net::SocketAddr;
use futures::channel::mpsc::SendError;

// TODO: Move errors to 'error' module.
#[derive(Debug)]
pub enum MailError {
    MpscSendError(SendError),
    UnknownIo(std::io::Error),
    NoConnection,
    ShutdownBridgeClosed,
    InternalTaskError(String),
    InvalidStructBuild(BuilderError)
}

#[derive(Debug)]
pub enum BuilderError {
    RequiredField(String),
    FieldAlreadySet(String),
    InvalidSocketAddr(std::io::Error),
    // TODO: Remove these errors below because they aren't needed in the client and shouldn't be for the server.
    AmbiguousSocketAddr,
    NoSocketAddrs,
}

/// Events which can occur during a connection.
#[derive(Debug)]
pub enum ConnectionEvent<C: Client> {
    /// A connection with a client was recently established.
    Established { client: C },
    /// A connection with a client was terminated (stopped).
    /// The peaceful field indicates if the stream was closed as the other machine may have crashed.
    Terminate { client: C, peaceful: bool },
    /// A packet was received from a client with the following bytes.
    Packet { client: C, bytes: Vec<u8> },
    /// An error occurred during processing.
    Error { error: MailError }
}

#[async_trait]
pub trait Client: Clone + Send + Sync {
    // TODO: Add usage and errors
    /// Send a packet to the client (Only bytes are accepted)
    async fn send_pkg(&mut self, data: Vec<u8>) -> Result<()>;
    // TODO: Add info, usage and errors
    async fn terminate(mut self) -> Result<()>;
    // TODO: Add info, usage and errors
    fn address(&self) -> SocketAddr;
}

/// The [std::result::Result] type used by the whole library.
pub type Result<T> = std::result::Result<T, MailError>;