use std::pin::Pin;
use std::sync::Arc;
use std::task::{Context, Poll};
use async_std::net::{SocketAddr, ToSocketAddrs};
use async_std::task;
use futures::channel::mpsc;
use futures::{Future, FutureExt, SinkExt};
use futures::lock::Mutex;
use async_trait::async_trait;
use crate::{shared_internals, BuilderError, Client, ConnectionEvent, MailError, Result};

pub struct LocalClientBuilder<F: Future<Output = ()> + Send + 'static, SA: ToSocketAddrs> {
    event_handler: Option<fn(ConnectionEvent<LocalClient>) -> F>,
    addr: Option<SA>,
}

impl<F: Future<Output = ()> + Send + 'static, SA: ToSocketAddrs> Default for LocalClientBuilder<F, SA> {
    fn default() -> Self {
        LocalClientBuilder::new()
    }
}

impl<F: Future<Output = ()> + Send + 'static, SA: ToSocketAddrs> LocalClientBuilder<F, SA> {
    pub fn new() -> LocalClientBuilder<F, SA> {
        LocalClientBuilder {
            event_handler: None,
            addr: None
        }
    }

    pub fn event_handler(mut self, event_handler: fn(ConnectionEvent<LocalClient>) -> F) -> Result<Self> {
        if self.event_handler.is_some() {
            Err(MailError::InvalidStructBuild(BuilderError::FieldAlreadySet(String::from("Event handler"))))
        } else {
            self.event_handler = Some(event_handler);
            Ok(self)
        }
    }

    // TODO: Try to addr.to_socket_addrs() to see if it fails.
    pub fn addr(mut self, addr: SA) -> Result<Self> {
        if self.addr.is_some() {
            Err(MailError::InvalidStructBuild(BuilderError::FieldAlreadySet(String::from("Address"))))
        } else {
            self.addr = Some(addr);
            Ok(self)
        }
    }

    // TODO: Add new function called spawn_connect that essentially is the same as task::spawn(connect()). You'll have to do .await to actually get the handle and await again to wait for the server to finish.
    // TODO: Do this for server too  ^^^^^^^^^^^^^
    /// Connect to the target address.
    ///
    /// Connects to the specified address using TCP. It blocks until a connection has been made but does not block while the connection is alive.
    pub async fn connect(self) -> Result<LocalClientHandle> {
        let addr = self.addr.ok_or_else(|| MailError::InvalidStructBuild(BuilderError::RequiredField(String::from("Address"))))?;
        // The reason we shadow the address variable here is because TcpStream returns the connection which actually succeeded. If you're still confused the addr: impl ToSocketAddrs can parse multiple addresses.
        let (addr, read, write) = shared_internals::tcp_create(addr).await?;
        let (shutdown, shutdown_recv) = mpsc::unbounded();
        let (msg, msg_recv) = mpsc::unbounded();
        let clt = LocalClient(Arc::new(LocalClientInner {
            shutdown: Mutex::new(shutdown),
            msg: Mutex::new(msg),
            addr
        }));
        let event_handler = self.event_handler.ok_or_else(|| MailError::InvalidStructBuild(BuilderError::RequiredField(String::from("Event handler"))))?;
        let handle_clt = LocalClient::clone(&clt);
        let raw_handle = task::spawn(async move {
            internal::local_connect(
                addr,
                read,
                write,
                handle_clt,
                msg_recv,
                shutdown_recv,
                event_handler
            ).await
        });
        let client_handle = LocalClientHandle {
            /*shutdown,
            msg,*/
            clt,
            handle: raw_handle
        };
        Ok(client_handle)
    }
}

pub struct LocalClientHandle {
    clt: LocalClient,
    handle: task::JoinHandle<Result<()>>
}

impl LocalClientHandle {
    pub fn clt_mut(&mut self) -> &mut LocalClient {
        &mut self.clt
    }

    pub fn clt(&self) -> &LocalClient {
        &self.clt
    }
}

impl Future for LocalClientHandle {
    type Output = Result<()>;

    fn poll(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Self::Output> {
        self.get_mut().handle.poll_unpin(cx)
    }
}

#[derive(Debug)]
struct LocalClientInner {
    shutdown: Mutex<mpsc::UnboundedSender<LocalClientShutdown>>,
    msg: Mutex<mpsc::UnboundedSender<Vec<u8>>>,
    addr: SocketAddr
}

#[derive(Debug, Clone)]
pub struct LocalClient(Arc<LocalClientInner>);

#[async_trait]
impl Client for LocalClient {
    async fn send_pkg(&mut self, data: Vec<u8>) -> Result<()> {
        self.0.msg.lock().await.send(data).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })
    }

    async fn terminate(mut self) -> Result<()> {
        self.0.shutdown.lock().await.send(LocalClientShutdown).await.map_err(|error| if error.is_disconnected() { MailError::NoConnection } else { MailError::MpscSendError(error) })
    }

    fn address(&self) -> SocketAddr {
        self.0.addr
    }
}

#[derive(Debug)]
pub struct LocalClientShutdown;

mod internal {
    use async_std::future::Future;
    use async_std::net::{SocketAddr, TcpStream};
    use futures::{SinkExt, io::{WriteHalf, ReadHalf}};
    use crate::ConnectionEvent;
    use super::*;

    pub fn spawn_task<R, F>(future: F, mut broker_send: mpsc::UnboundedSender<ConnectionEvent<LocalClient>>) -> task::JoinHandle<()>
        where
            F: Future<Output=Result<R>> + Send + 'static,
            R: Send,
    {
        task::spawn(async move {
            if let Err(error) = future.await {
                broker_send.send(ConnectionEvent::Error { error }).await.unwrap();
            }
        })
    }

    pub async fn local_connect<F: Future<Output = ()> + Send + 'static>(
        addr: SocketAddr,
        stream_read: ReadHalf<TcpStream>,
        stream_write: WriteHalf<TcpStream>,
        local_client: LocalClient,
        msg_recv: mpsc::UnboundedReceiver<Vec<u8>>,
        shutdown_recv: mpsc::UnboundedReceiver<LocalClientShutdown>,
        on_event: fn(ConnectionEvent<LocalClient>) -> F
    ) -> Result<()> {
        /* Channels */
        let (broker_send, broker_recv) = mpsc::unbounded();

        /* Handles */
        let _broker_handle = spawn_task(shared_internals::broker_task(broker_recv, on_event), broker_send.clone());
        let _write_handle = spawn_task(shared_internals::write_task(msg_recv, stream_write), broker_send.clone());

        println!("Connected to <{:?}>.", addr);
        /* Reading and shutdown */
        let mut buffer = [0u8; 4096];
        shared_internals::blocking_dispatcher(
            broker_send,
            &mut buffer,
            local_client,
            stream_read,
            shutdown_recv
        ).await
    }
}